!
!    Copyright  2013, 2014, 2020, 2021 Guy Munhoven
!
!    This file is part of SolveSAPHE v. 2

!    SolveSAPHE is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Lesser General Public License as published by
!    the Free Software Foundation, either version 3 of the License, or
!    (at your option) any later version.
!
!    SolveSAPHE is distributed in the hope that it will be useful,
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU Lesser General Public License for more details.
!
!    You should have received a copy of the GNU Lesser General Public License
!    along with SolveSAPHE.  If not, see <http://www.gnu.org/licenses/>.
!





! **********************
! Precompiler directives
! **********************

! SAFEGEOMEAN_INIT:
! - if not defined, use standard initialisation
! - if defined, use the geometric mean of the brackets.

#undef SAFEGEOMEAN_INIT


! **************************
! End precompiler directives
! **************************



MODULE MOD_PHSOLVERS_LOGGING

USE MOD_PRECISION

IMPLICIT NONE

! General parameters

! Threshold relative difference between successive iterates
! (convergence criterion)
REAL(KIND=wp), PARAMETER :: pp_rdel_ah_target = 1.E-08_wp

! LN(10)
REAL(KIND=wp), PARAMETER :: pp_ln10 = 2.302585092994045684018_wp

! NaN for [H^+] results
REAL(KIND=wp), PARAMETER :: pp_hnan = -1._wp

! Flag values to select the independent DIC variable
INTEGER, PARAMETER :: jp_dic  = 1
INTEGER, PARAMETER :: jp_co2  = 2
INTEGER, PARAMETER :: jp_hco3 = 3
INTEGER, PARAMETER :: jp_co3  = 4


! Maximum number of iterations for each method

INTEGER, PARAMETER ::  jp_maxniter_atgen    = 100
INTEGER, PARAMETER ::  jp_maxniter_atsec    = 100


! Bookkeeping variables for each method

! - SOLVE_AT_GENERAL
INTEGER,       DIMENSION(2) ::  niter_atgen    = jp_maxniter_atgen    + 2
INTEGER,       DIMENSION(2) ::  niter_atgen_limin
INTEGER,       DIMENSION(2) ::  niter_atgen_limax
INTEGER,       DIMENSION(2) ::  niter_atgen_limabs
REAL(KIND=wp), DIMENSION(2) ::  ahmin_atgen_ini, vhmin_atgen_ini
REAL(KIND=wp), DIMENSION(2) ::  ahmax_atgen_ini, vhmax_atgen_ini
REAL(KIND=wp), DIMENSION(2) ::  ah_atgen_ini

! - SOLVE_AT_GENERAL_SEC
INTEGER,       DIMENSION(2) ::  niter_atsec    = jp_maxniter_atsec    + 2
INTEGER,       DIMENSION(2) ::  niter_atsec_limin
INTEGER,       DIMENSION(2) ::  niter_atsec_limax
INTEGER,       DIMENSION(2) ::  niter_atsec_limabs
REAL(KIND=wp), DIMENSION(2) ::  ahmin_atsec_ini, vhmin_atsec_ini
REAL(KIND=wp), DIMENSION(2) ::  ahmax_atsec_ini, vhmax_atsec_ini
REAL(KIND=wp), DIMENSION(2) ::  ah_atsec_ini

! - HINSUPINI/ALK_TAN
INTEGER                     ::  niter_alktan   = 0
REAL(KIND=wp)               ::  ahtan, aatan

! - Critical parameters for \gamma > 0:
REAL(KIND=wp)               ::  ahmin_gampos
REAL(KIND=wp)               ::  almin_gampos
REAL(KIND=wp)               ::  acrit2_gampos
REAL(KIND=wp)               ::  acrit0_gampos


! Keep the following functions private to avoid conflicts with
! other modules that provide similar ones.

PRIVATE HINI_ACB_DIC, HINI_ACB_CO2, HINI_ACBW_HCO3, HINI_ACBW_CO3


CONTAINS

!===============================================================================
 SUBROUTINE ANW_INFSUP(p_dictot, p_bortot,                                     &
                       p_po4tot, p_siltot,                                     &
                       p_nh4tot, p_h2stot,                                     &
                       p_so4tot, p_flutot,                                     &
                       p_alknw_inf, p_alknw_sup,                               &
                       p_alknw_asympt_coeff                                    )
!===============================================================================

! Subroutine returns the lower and upper bounds of "non-water-selfionization"
! contributions to total alkalinity (the infimum and the supremum), i.e
! inf(TA - [OH-] + [H+]) and sup(TA - [OH-] + [H+])

USE MOD_CHEMCONST


IMPLICIT NONE


!--------------------!
! Argument variables !
!--------------------!

REAL(KIND=wp), INTENT(IN)            :: p_dictot
REAL(KIND=wp), INTENT(IN)            :: p_bortot
REAL(KIND=wp), INTENT(IN)            :: p_po4tot
REAL(KIND=wp), INTENT(IN)            :: p_siltot
REAL(KIND=wp), INTENT(IN)            :: p_nh4tot
REAL(KIND=wp), INTENT(IN)            :: p_h2stot
REAL(KIND=wp), INTENT(IN)            :: p_so4tot
REAL(KIND=wp), INTENT(IN)            :: p_flutot
REAL(KIND=wp), INTENT(OUT)           :: p_alknw_inf
REAL(KIND=wp), INTENT(OUT)           :: p_alknw_sup
REAL(KIND=wp), INTENT(OUT), OPTIONAL :: p_alknw_asympt_coeff


!-------------------------------------------------------------------------------


! p_alknw_inf = -\Sum_i m_i Xtot_i

! p_alknw_inf =-p_dictot*0._wp &          ! n = 2, m = 0
!              -p_bortot*0._wp &          ! n = 1, m = 0
!              -p_po4tot*1._wp &          ! n = 3, m = 1
!              -p_siltot*0._wp &          ! n = 1, m = 0
!              -p_nh4tot*0._wp &          ! n = 1, m = 0
!              -p_h2stot*0._wp &          ! n = 1, m = 0
!              -p_so4tot*1._wp &          ! n = 1, m = 1
!              -p_flutot*1._wp            ! n = 1, m = 1

p_alknw_inf =    -p_po4tot - p_so4tot - p_flutot


! p_alknw_sup = \Sum_i (n_i - m_i) Xtot_i

! p_alknw_sup = p_dictot*(2._wp-0._wp) &  ! n = 2, m = 0
!               p_bortot*(1._wp-0._wp) &  ! n = 1, m = 0
!               p_po4tot*(3._wp-1._wp) &  ! n = 3, m = 1
!               p_siltot*(1._wp-0._wp) &  ! n = 1, m = 0
!               p_nh4tot*(1._wp-0._wp) &  ! n = 1, m = 0
!               p_h2stot*(1._wp-0._wp) &  ! n = 1, m = 0
!               p_so4tot*(1._wp-1._wp) &  ! n = 1, m = 1
!               p_flutot*(1._wp-1._wp)    ! n = 1, m = 1

p_alknw_sup =   p_dictot + p_dictot + p_bortot &
              + p_po4tot + p_po4tot + p_siltot &
              + p_nh4tot + p_h2stot

IF (PRESENT(p_alknw_asympt_coeff)) THEN
  p_alknw_asympt_coeff = &
                p_dictot*api1_dic + p_bortot*api1_bor &
              + p_po4tot*api1_po4 + p_siltot*api1_sil &
              + p_nh4tot*api1_nh4 + p_h2stot*api1_h2s &
              + p_so4tot*api1_so4 + p_flutot*api1_flu
ENDIF


RETURN


!===============================================================================
 END SUBROUTINE ANW_INFSUP
!===============================================================================




!===============================================================================
 FUNCTION ANW(p_h,    p_dicvar, p_bortot,                                      &
                      p_po4tot, p_siltot,                                      &  
                      p_nh4tot, p_h2stot,                                      &
                      p_so4tot, p_flutot,                                      &
                      k_dicsel, p_derivanw                                     )
!===============================================================================

USE MOD_CHEMCONST


IMPLICIT NONE

REAL(KIND=wp) :: ANW


!--------------------!
! Argument variables !
!--------------------!

REAL(KIND=wp), INTENT(IN)            :: p_h
REAL(KIND=wp), INTENT(IN)            :: p_dicvar
REAL(KIND=wp), INTENT(IN)            :: p_bortot
REAL(KIND=wp), INTENT(IN)            :: p_po4tot
REAL(KIND=wp), INTENT(IN)            :: p_siltot
REAL(KIND=wp), INTENT(IN)            :: p_nh4tot
REAL(KIND=wp), INTENT(IN)            :: p_h2stot
REAL(KIND=wp), INTENT(IN)            :: p_so4tot
REAL(KIND=wp), INTENT(IN)            :: p_flutot
INTEGER,       INTENT(IN)            :: k_dicsel
REAL(KIND=wp), INTENT(OUT), OPTIONAL :: p_derivanw


!-----------------!
! Local variables !
!-----------------!

REAL(KIND=wp) :: znumer_dic, zdnumer_dic, zdenom_dic, zalk_dic, zdalk_dic
REAL(KIND=wp) :: znumer_bor, zdnumer_bor, zdenom_bor, zalk_bor, zdalk_bor
REAL(KIND=wp) :: znumer_po4, zdnumer_po4, zdenom_po4, zalk_po4, zdalk_po4
REAL(KIND=wp) :: znumer_sil, zdnumer_sil, zdenom_sil, zalk_sil, zdalk_sil
REAL(KIND=wp) :: znumer_nh4, zdnumer_nh4, zdenom_nh4, zalk_nh4, zdalk_nh4
REAL(KIND=wp) :: znumer_h2s, zdnumer_h2s, zdenom_h2s, zalk_h2s, zdalk_h2s
REAL(KIND=wp) :: znumer_so4, zdnumer_so4, zdenom_so4, zalk_so4, zdalk_so4
REAL(KIND=wp) :: znumer_flu, zdnumer_flu, zdenom_flu, zalk_flu, zdalk_flu
REAL(KIND=wp) ::                                      zalk_wat, zdalk_wat


!-------------------------------------------------------------------------------


! H2CO3 - HCO3 - CO3 : n=2, m=0
znumer_dic = 2._wp*api2_dic + p_h*       api1_dic
SELECT CASE(k_dicsel)
CASE(jp_dic)
  ! DIC is the control variable of the carbonate system
  zdenom_dic =     api2_dic + p_h*(      api1_dic + p_h)
CASE(jp_co2)
  ! [CO2] is the control variable of the carbonate system
  zdenom_dic =                p_h*                  p_h
CASE(jp_hco3)
  ! [HCO3] is the control variable of the carbonate system
  zdenom_dic =                p_h*       api1_dic
CASE(jp_co3)
  ! [CO3] is the control variable of the carbonate system
  zdenom_dic =     api2_dic
CASE DEFAULT
  CALL ABORT()
END SELECT
zalk_dic   = p_dicvar * (znumer_dic/zdenom_dic)

! B(OH)3 - B(OH)4 : n=1, m=0
znumer_bor =       api1_bor
zdenom_bor =       api1_bor + p_h
zalk_bor   = p_bortot * (znumer_bor/zdenom_bor)

! H3PO4 - H2PO4 - HPO4 - PO4 : n=3, m=1
znumer_po4 = 3._wp*api3_po4 + p_h*(2._wp*api2_po4 + p_h* api1_po4)
zdenom_po4 =       api3_po4 + p_h*(      api2_po4 + p_h*(api1_po4 + p_h))
zalk_po4   = p_po4tot * (znumer_po4/zdenom_po4 - 1._wp) ! Zero level of H3PO4 = 1

! H4SiO4 - H3SiO4 : n=1, m=0
znumer_sil =       api1_sil
zdenom_sil =       api1_sil + p_h
zalk_sil   = p_siltot * (znumer_sil/zdenom_sil)

! NH4 - NH3 : n=1, m=0
znumer_nh4 =       api1_nh4
zdenom_nh4 =       api1_nh4 + p_h
zalk_nh4   = p_nh4tot * (znumer_nh4/zdenom_nh4)

! H2S - HS : n=1, m=0
znumer_h2s =       api1_h2s
zdenom_h2s =       api1_h2s + p_h
zalk_h2s   = p_h2stot * (znumer_h2s/zdenom_h2s)

! HSO4 - SO4 : n=1, m=1
znumer_so4 =       api1_so4
zdenom_so4 =       api1_so4 + p_h
zalk_so4   = p_so4tot * (znumer_so4/zdenom_so4 - 1._wp)

! HF - F : n=1, m=1
znumer_flu =       api1_flu
zdenom_flu =       api1_flu + p_h
zalk_flu   = p_flutot * (znumer_flu/zdenom_flu - 1._wp)


ANW =   zalk_dic + zalk_bor + zalk_po4 + zalk_sil &
      + zalk_nh4 + zalk_h2s + zalk_so4 + zalk_flu


IF (PRESENT(p_derivanw)) THEN

  ! H2CO3 - HCO3 - CO3 : n=2
  SELECT CASE(k_dicsel)
  CASE(jp_dic)
    ! DIC is the control variable of the carbonate system
    zdnumer_dic = api1_dic*api2_dic + p_h*(4._wp*api2_dic                      &
                                    + p_h*       api1_dic)
    zdalk_dic   = -p_dicvar*(zdnumer_dic/zdenom_dic**2)
  CASE(jp_co2)
    ! [CO2] is the control variable of the carbonate system
    zdnumer_dic = 4._wp*api2_dic    + p_h*(2._wp*api1_dic                      &
                                    - p_h)
    zdalk_dic   = -p_dicvar*(zdnumer_dic/p_h**3)
  CASE(jp_hco3)
    ! [HCO3] is the control variable of the carbonate system
    zdalk_dic   = -p_dicvar*(2._wp*api2_dic/p_h**2)
  CASE(jp_co3)
    ! [CO3] is the control variable of the carbonate system
    zdalk_dic   =  p_dicvar/api2_dic
  END SELECT

  ! B(OH)3 - B(OH)4 : n=1
  zdnumer_bor = api1_bor
  zdalk_bor   = -p_bortot*(zdnumer_bor/zdenom_bor**2)

  ! H3PO4 - H2PO4 - HPO4 - PO4 : n=3
  zdnumer_po4 = api2_po4*api3_po4 + p_h*(4._wp*api1_po4*api3_po4              &
                                  + p_h*(9._wp*api3_po4 + api1_po4*api2_po4   &
                                  + p_h*(4._wp*api2_po4                       &
                                  + p_h*       api1_po4)))
  zdalk_po4   = -p_po4tot * (zdnumer_po4/zdenom_po4**2)

  ! H4SiO4 - H3SiO4 : n=1
  zdnumer_sil = api1_sil
  zdalk_sil   = -p_siltot * (zdnumer_sil/zdenom_sil**2)

  ! NH4 - NH3 : n=1
  zdnumer_nh4 = api1_nh4
  zdalk_nh4   = -p_nh4tot * (zdnumer_nh4/zdenom_nh4**2)

  ! H2S - HS : n=1
  zdnumer_h2s = api1_h2s
  zdalk_h2s   = -p_h2stot * (zdnumer_h2s/zdenom_h2s**2)

  ! HSO4 - SO4 : n=1
  zdnumer_so4 = api1_so4
  zdalk_so4   = -p_so4tot * (zdnumer_so4/zdenom_so4**2)

  ! HF - F : n=1
  zdnumer_flu = api1_flu
  zdalk_flu   = -p_flutot * (zdnumer_flu/zdenom_flu**2)

  p_derivanw =   zdalk_dic + zdalk_bor + zdalk_po4 + zdalk_sil &
               + zdalk_nh4 + zdalk_h2s + zdalk_so4 + zdalk_flu 

ENDIF

RETURN

!===============================================================================
 END FUNCTION ANW
!===============================================================================




!===============================================================================
 FUNCTION EQUATION_AT(p_alktot, p_h, p_dicvar, p_bortot,                       &
                                    p_po4tot, p_siltot,                        &
                                    p_nh4tot, p_h2stot,                        &
                                    p_so4tot, p_flutot,                        &
                                    k_dicsel, p_deriveqn                       )
!===============================================================================

USE MOD_CHEMCONST,                  ONLY: api1_wat, aphscale


IMPLICIT NONE

REAL(KIND=wp) :: EQUATION_AT


!--------------------!
! Argument variables !
!--------------------!

REAL(KIND=wp), INTENT(IN)            :: p_alktot
REAL(KIND=wp), INTENT(IN)            :: p_h
REAL(KIND=wp), INTENT(IN)            :: p_dicvar
REAL(KIND=wp), INTENT(IN)            :: p_bortot
REAL(KIND=wp), INTENT(IN)            :: p_po4tot
REAL(KIND=wp), INTENT(IN)            :: p_siltot
REAL(KIND=wp), INTENT(IN)            :: p_nh4tot
REAL(KIND=wp), INTENT(IN)            :: p_h2stot
REAL(KIND=wp), INTENT(IN)            :: p_so4tot
REAL(KIND=wp), INTENT(IN)            :: p_flutot
INTEGER,       INTENT(IN),  OPTIONAL :: k_dicsel
REAL(KIND=wp), INTENT(OUT), OPTIONAL :: p_deriveqn


!-----------------!
! Local variables !
!-----------------!

REAL(KIND=wp) :: zalk_nw,  zdalk_nw
REAL(KIND=wp) :: zalk_wat, zdalk_wat
REAL(KIND=wp) :: z_deriv_anw
INTEGER       :: i_dicsel


!-------------------------------------------------------------------------------


IF (.NOT. PRESENT(k_dicsel)) THEN
  i_dicsel = jp_dic                 ! Default = DIC, to ensure compatibility
                                    ! with SolveSAPHE 1.0.x
ELSE
  i_dicsel = k_dicsel
ENDIF


! H2O - OH
zalk_wat   = api1_wat/p_h - p_h/aphscale

IF (.NOT. PRESENT(p_deriveqn)) THEN

  EQUATION_AT =   ANW(p_h, p_dicvar, p_bortot,     &
                           p_po4tot, p_siltot,     &
                           p_nh4tot, p_h2stot,     &
                           p_so4tot, p_flutot,     &
                           K_DICSEL = i_dicsel     )   &
                + zalk_wat - p_alktot

ELSE

  EQUATION_AT =   ANW(p_h, p_dicvar, p_bortot,     &
                           p_po4tot, p_siltot,     &
                           p_nh4tot, p_h2stot,     &
                           p_so4tot, p_flutot,     &
                           K_DICSEL = i_dicsel,    &
                           P_DERIVANW = z_deriv_anw)   &
                + zalk_wat - p_alktot  

  p_deriveqn =  z_deriv_anw - api1_wat/p_h**2 - 1._wp/aphscale

ENDIF

RETURN

!===============================================================================
 END FUNCTION EQUATION_AT
!===============================================================================




!===============================================================================
 FUNCTION HINI_ACB_DIC(p_alkcb, p_dictot, p_bortot, p_hinf, p_hsup)
!===============================================================================

! Function returns the root for the 2nd order approximation of the
! DIC -- B_T -- A_CB equation for [H+] (reformulated as a cubic polynomial)
! around the local minimum, if it exists.

! Returns * p_hsup if p_alkcb <= 0
!         * p_hinf if p_alkcb >= 2*p_dictot + p_bortot
!         * the geometric mean of p_hinf and p_hsup if
!           0 < p_alkcb < 2*p_dictot + p_bortot
!           but the 2nd order approximation does not have a solution


USE MOD_CHEMCONST, ONLY : api1_dic, api2_dic, api1_bor


IMPLICIT NONE


REAL(KIND=wp) :: HINI_ACB_DIC


!--------------------!
! Argument variables !
!--------------------!

REAL(KIND=wp), INTENT(IN) :: p_alkcb, p_dictot, p_bortot
REAL(KIND=wp), INTENT(IN) :: p_hinf, p_hsup


!-----------------!
! Local variables !
!-----------------!

REAL(KIND=wp)  ::  zca, zba
REAL(KIND=wp)  ::  zd, zsqrtd, zhmin
REAL(KIND=wp)  ::  za2, za1, za0

REAL(KIND=wp)  ::  z_hini


!-------------------------------------------------------------------------------


IF (p_alkcb <= 0._wp) THEN

  z_hini = p_hsup

ELSEIF (p_alkcb >= (2._wp*p_dictot + p_bortot)) THEN

  z_hini = p_hinf

ELSE

  zca = p_dictot / p_alkcb
  zba = p_bortot / p_alkcb

                                    ! Coefficients of the cubic polynomial
  za2 = api1_bor*(1._wp - zba) + api1_dic*(1._wp-zca)
  za1 = api1_dic*api1_bor*(1._wp - zba - zca) + api2_dic*(1._wp - (zca+zca))
  za0 = api2_dic*api1_bor*(1._wp - zba - (zca+zca))


                                    ! Taylor expansion around the minimum

  zd = za2*za2 - 3._wp*za1          ! Discriminant of the quadratic equation
                                    ! for the minimum close to the root

  IF (zd > 0._wp) THEN              ! If the discriminant is positive, i.e.,
                                    ! if the cubic has two distinct extrema
    zsqrtd = SQRT(zd)

    IF(za2 < 0) THEN
      zhmin =        (-za2 + zsqrtd) / 3._wp
    ELSE
      zhmin = -za1 / ( za2 + zsqrtd)
    ENDIF

    z_hini = zhmin + SQRT(-(za0 + zhmin*(za1 + zhmin*(za2 + zhmin)))/zsqrtd)

                                    ! make h_ini is within [p_hinf,p_hsup]
    z_hini = MAX(MIN(p_hsup, z_hini), p_hinf)

  ELSE

    z_hini = SQRT(p_hinf*p_hsup)

  ENDIF

ENDIF

HINI_ACB_DIC = z_hini


RETURN


!===============================================================================
 END FUNCTION HINI_ACB_DIC
!===============================================================================




!===============================================================================
 FUNCTION HINI_ACB_CO2(p_alkcb, p_co2, p_bortot, p_hinf, p_hsup)
!===============================================================================

! Function returns the root for the 2nd order approximation of the
! DIC -- B_T -- A_CB equation for [H+] (reformulated as a cubic polynomial)
! around the local minimum, if it exists.

! Returns * p_hsup if p_alkcb <= 0
!         * the geometric mean of p_hinf and p_hsup if 0 < p_alkcb
!           but the 2nd order approximation does not have a solution


USE MOD_CHEMCONST,                  ONLY : api1_dic, api2_dic, api1_bor


IMPLICIT NONE


REAL(KIND=wp) :: HINI_ACB_CO2

!--------------------!
! Argument variables !
!--------------------!

REAL(KIND=wp), INTENT(IN) :: p_alkcb, p_co2, p_bortot
REAL(KIND=wp), INTENT(IN) :: p_hinf, p_hsup


!-----------------!
! Local variables !
!-----------------!

REAL(KIND=wp)  ::  zac, zbc
REAL(KIND=wp)  ::  zd, zsqrtd
REAL(KIND=wp)  ::  za3, za2, za1, za0
REAL(KIND=wp)  ::  zhmin, zpcb_hmin

REAL(KIND=wp)  ::  z_hini, zpcb_hini


!-------------------------------------------------------------------------------


IF (p_alkcb <= 0._wp) THEN

  z_hini = p_hsup

ELSE

  zac = p_alkcb  / p_co2
  zbc = p_bortot / p_co2

                                    ! Coefficients of the cubic polynomial
  za3 = zac
  za2 = -api1_dic + api1_bor * (zac - zbc)
  za1 = -(api1_dic*api1_bor + 2._wp * api2_dic)
  za0 = -2._wp*api2_dic*api1_bor


                                    ! Taylor expansion around the minimum

  zd = za2*za2 - 3._wp*za3*za1     ! Discriminant of the quadratic equation
                                    ! for the minimum close to the root

  IF (zd > 0._wp) THEN              ! If the discriminant is positive, i.e.,
                                    ! if the cubic has two distinct extrema,
    zsqrtd = SQRT(zd)               ! locate the minimum.

    IF(za2 < 0) THEN
      zhmin = (-za2 + zsqrtd)/3._wp
    ELSE
      zhmin = -za1/(za2 + zsqrtd)
    ENDIF

    zpcb_hmin = za0 + zhmin*(za1 + zhmin*(za2 + zhmin*za3))

    IF (zpcb_hmin < 0._wp) THEN     ! If the minimum has a negative ordinate,
                                    ! it can be used to derive a H_0

      z_hini = zhmin + SQRT(-zpcb_hmin/zsqrtd)

                                    ! Check if this H_ini is compatible
                                    ! with the Alk_CB bounds. We have to
                                    ! check the sign of the auxiliary
                                    ! polynomial 
      za2 = 2._wp - zac + zbc
      za1 = 2._wp * api1_dic
      za0 = 2._wp * api2_dic

      zpcb_hini = za0 + z_hini*(za1 + z_hini*za2)

      IF (zpcb_hini > 0._wp) THEN
                                    ! H_ini *is* compatible with Alk_CB:
                                    ! only make sure it is within [p_hinf,p_hsup].
        z_hini = MAX(MIN(p_hsup, z_hini), p_hinf)

      ELSE                          ! H_ini *is not* compatble with Alk_CB:
                                    ! use the geometric mean of
                                    ! p_hinf and p_hsup instead.
        z_hini = SQRT(p_hinf*p_hsup)

      ENDIF

    ELSE

      z_hini = SQRT(p_hinf*p_hsup)

    ENDIF

  ELSE                              ! If the cubic does not have distinct
                                    ! extrema, use the geometric mean of
                                    ! p_hinf and p_hsup

    z_hini = SQRT(p_hinf*p_hsup)

  ENDIF

ENDIF

HINI_ACB_CO2 = z_hini


RETURN


!===============================================================================
 END FUNCTION HINI_ACB_CO2
!===============================================================================




!===============================================================================
 FUNCTION HINI_ACBW_HCO3(p_alkcbw, p_hco3, p_bortot, p_hinf, p_hsup)
!===============================================================================

! Function returns the root for the 2nd order approximation of the
! DIC -- B_T -- A_CB equation for [H+] (reformulated as a cubic polynomial)
! around the local minimum, if it exists.

! Returns * p_hsup if p_alkcbw <= 0
!         * the geometric mean of p_hinf and p_hsup if 0 < p_alkcb 
!           but the 2nd order approximation does not have a solution


USE MOD_CHEMCONST,                  ONLY : api1_dic, api2_dic, api1_bor,       &
                                           aphscale, api1_wat

IMPLICIT NONE


REAL(KIND=wp) :: HINI_ACBW_HCO3

!--------------------!
! Argument variables !
!--------------------!

REAL(KIND=wp), INTENT(IN) ::  p_alkcbw, p_hco3, p_bortot
REAL(KIND=wp), INTENT(IN) ::  p_hinf, p_hsup


!-----------------!
! Local variables !
!-----------------!

REAL(KIND=wp) ::  zgamma
REAL(KIND=wp) ::  zac, zbc
REAL(KIND=wp) ::  zd, zsqrtd
REAL(KIND=wp) ::  za3, za2, za1, za0
REAL(KIND=wp) ::  zhmin, zpcbw_hmin
REAL(KIND=wp) ::  zhmax, zpcbw_hmax

REAL(KIND=wp) ::  z_hini


!-------------------------------------------------------------------------------


IF (p_alkcbw <= 0._wp) THEN

  z_hini = p_hsup

ELSE

  zac = p_alkcbw / p_hco3
  zbc = p_bortot / p_hco3

                                    ! Coefficients of the cubic polynomial
  za3 = 1._wp / (aphscale * p_hco3)
  za2 = zac + api1_bor * za3 - 1._wp
  za0 = -(api1_wat / p_hco3 + 2._wp*api2_dic/api1_dic)
  za1 = api1_bor * (zac - zbc - 1._wp) + za0
  za0 = api1_bor * za0

                                    ! Taylor expansion around the minimum

  zd = za2*za2 - 3._wp*za3*za1      ! Discriminant of the quadratic equation
                                    ! for the extrema

  IF (zd > 0._wp) THEN              ! If the discriminant is positive, i.e.,
                                    ! if the cubic has two distinct extrema
    zsqrtd = SQRT(zd)

    IF(za2 < 0) THEN
      zhmin =        (-za2 + zsqrtd ) / (3._wp * za3)
    ELSE
      zhmin = -za1 / ( za2 + zsqrtd )
    ENDIF

                                    ! Here z_pcbw_hmin = P_CBW(H_min)
    zpcbw_hmin = za0 + zhmin * (za1 + zhmin * (za2 + zhmin*za3))

    IF (zpcbw_hmin < 0._wp) THEN

      z_hini = zhmin + SQRT(-zpcbw_hmin/zsqrtd)

    ELSE

      IF(za2 < 0._wp) THEN
        zhmax = -za1 / ( za2 - zsqrtd )
      ELSE
        zhmax =        (-za2 - zsqrtd ) / (3._wp * za3)
      ENDIF

                                    ! Here z_pcbw_hmax = P_CBW(H_max) - a_0
      zpcbw_hmax = zhmax * (za1 + zhmax * (za2 + zhmax*za3))

      z_hini = zhmax * (1._wp - SQRT((zpcbw_hmax + za0)/zpcbw_hmax))

    ENDIF
                                    ! Still have to make sure that
                                    ! z_hini is within [p_hinf,p_hsup]
      z_hini = MAX(MIN(p_hsup, z_hini), p_hinf)

  ELSE                              ! If the cubic does not have distinct
                                    ! extrema, use the geometric mean of
                                    ! H_min and p_max for H_ini

    z_hini = SQRT(p_hinf*p_hsup)

  ENDIF

ENDIF

HINI_ACBW_HCO3 = z_hini


RETURN


!===============================================================================
 END FUNCTION HINI_ACBW_HCO3
!===============================================================================




!===============================================================================
 FUNCTION HINI_ACBW_CO3(p_alkcbw, p_co3, p_bortot, p_hinf, p_hsup, k_nroots)
!===============================================================================

! Function returns the root for the 2nd order approximation of the
! DIC -- B_T -- A_CB equation for [H+] (reformulated as a cubic polynomial)
! around the local minimum, if it exists.

! Returns * p_hsup if p_alkcbw <= 0
!         * the geometric mean of p_hinf and p_hsup if 0 < p_alkcb 
!           but the 2nd order approximation does not have a solution


USE MOD_CHEMCONST,                  ONLY : api1_dic, api2_dic, api1_bor,       &
                                           aphscale, api1_wat

IMPLICIT NONE


REAL(KIND=wp), DIMENSION(2) :: HINI_ACBW_CO3

!--------------------!
! Argument variables !
!--------------------!

REAL(KIND=wp),               INTENT(IN) ::  p_alkcbw, p_co3, p_bortot
REAL(KIND=wp), DIMENSION(2), INTENT(IN) ::  p_hinf, p_hsup
INTEGER,                     INTENT(IN) ::  k_nroots


!-----------------!
! Local variables !
!-----------------!

REAL(KIND=wp) ::  zgamma
REAL(KIND=wp) ::  zac, zbc
REAL(KIND=wp) ::  za3, za2, za1, za0
REAL(KIND=wp) ::  zd, zsqrtd 
REAL(KIND=wp) ::  zhmin, zpcbw_hmin
REAL(KIND=wp) ::  zhmax, zpcbw_hmax
REAL(KIND=wp) ::  zhifl, zpcbw_hifl

REAL(KIND=wp) ::  z_hini1, z_hini2


!-------------------------------------------------------------------------------



IF (p_alkcbw <= 0._wp) THEN

! [XXX] Not sure about this !!

  z_hini1 = p_hsup(1)
  z_hini2 = p_hsup(2)

!~ ELSEIF (p_alkcb >= (2._wp*p_dictot + p_bortot)) THEN

!~   z_hini = p_hinf

ELSE

  zgamma = p_co3*(api1_dic/api2_dic) - 1._wp / aphscale

  zac = p_alkcbw / p_co3
  zbc = p_bortot / p_co3

                                    ! Coefficients of the cubic polynomial
  za3 = zgamma / p_co3
  za2 = -(api1_bor * za3 + zac - 2._wp)
  za0 = api1_wat / p_co3            ! ... provisionally
  za1 = -api1_bor * (zac - zbc - 2._wp) + za0
  za0 = api1_bor * za0              ! ... finally


  IF (zgamma < 0._wp) THEN          ! k_nroots = 1 always in this case

    zd = za2*za2 - 3._wp*za1*za3

    IF (zd <= 0._wp) THEN

      z_hini1 = SQRT(p_hinf(1)*p_hsup(1))
      z_hini2 = pp_hnan

    ELSE

      zsqrtd = SQRT(zd)

      zhmax = - ( za2 + zsqrtd) / (3._wp * za3)    ! Since za3 < 0, we
      zhmin =   (-za2 + zsqrtd) / (3._wp * za3)    ! have H_min < H_max 

      zpcbw_hmax = za0 + zhmax*(za1 + zhmax*(za2 + zhmax * za3))

      IF (zpcbw_hmax >= 0) THEN
        z_hini1 = zhmax + SQRT(zpcbw_hmax/zsqrtd)
      ELSE
                                    ! Here: zpcbw_hmin = P_CBW(H_min) - 
        zpcbw_hmin = zhmin * (za1 + zhmin*(za2 + zhmin * za3))
        z_hini1    = zhmin * (1._wp - SQRT(zpcbw_hmin/(zpcbw_hmin + za0)))
      ENDIF
                                    ! and make sure that z_hini1 is 
                                    ! within [p_hinf(1),p_hsup(1)]
      z_hini1 = MAX(MIN(p_hsup(1), z_hini1), p_hinf(1))

      z_hini2 = pp_hnan

    ENDIF


  ELSEIF (zgamma > 0._wp) THEN

    SELECT CASE(k_nroots)
    CASE(0)

      z_hini1 = pp_hnan
      z_hini2 = pp_hnan

    CASE(1)                         ! H = H_tan

      z_hini1 = p_hinf(1)           ! p_hinf(1) = p_hsup(1) = H_tan
      z_hini2 = pp_hnan             ! in this case

    CASE(2)

      zd = za2*za2 - 3._wp*za1*za3

#if defined(DEBUG_PHSOLVERS)
      PRINT*, '[HINI_ACBW_CO3] discriminant  :', zd
#endif

      IF (zd > 0._wp) THEN 

        zsqrtd = SQRT(zd)

        zhmax = -( za2 + zsqrtd) / (3._wp * za3)    ! Since za3 > 0, we
        zhmin =  (-za2 + zsqrtd) / (3._wp * za3)    ! have H_max < H_min

        zpcbw_hmin = za0 + zhmin*(za1 + zhmin*(za2 + zhmin * za3))

#if defined(DEBUG_PHSOLVERS)
        PRINT*, '[HINI_ACBW_CO3] zhmax  :', zhmax
        PRINT*, '[HINI_ACBW_CO3] zhmin  :', zhmin
        PRINT*, '[HINI_ACBW_CO3] zpcbw_hmin  :', zpcbw_hmin
#endif

        IF ((zhmin <= 0._wp) .OR. (zpcbw_hmin >= 0._wp)) THEN

          z_hini1 = SQRT(p_hinf(1) * p_hsup(1))
          z_hini2 = SQRT(p_hinf(2) * p_hsup(2))

        ELSE

          zhifl = -za2 / (3._wp * za3)
          zpcbw_hifl = za0 + zhifl*(za1 + zhifl*(za2 + zhifl * za3))

#if defined(DEBUG_PHSOLVERS)
          PRINT*, '[HINI_ACBW_CO3] zhifl  :', zhifl
          PRINT*, '[HINI_ACBW_CO3] zpcbw_hifl  :', zpcbw_hifl
#endif
          IF (zhifl > 0._wp) THEN

            IF (zpcbw_hifl > 0._wp) THEN

              z_hini1 = zhmin - (zhmin - zhifl) &
                                * SQRT(zpcbw_hmin/(zpcbw_hmin - zpcbw_hifl))

            ELSEIF (zpcbw_hifl < 0._wp) THEN

              zpcbw_hmax = za0 + zhmax*(za1 + zhmax*(za2 + zhmax * za3))

              z_hini1 = zhmax + (zhifl - zhmax) &
                                * SQRT(zpcbw_hmax/(zpcbw_hmax - zpcbw_hifl))

#if defined(DEBUG_PHSOLVERS)
              PRINT*, '[HINI_ACBW_CO3] zpcbw_hmax    :', zpcbw_hmax
              PRINT*, '[HINI_ACBW_CO3] z_hini1 (prov):', z_hini1
#endif

            ELSE

              z_hini1 = zhifl

            ENDIF

          ELSE

            z_hini1 = zhmin * (1._wp - SQRT(zpcbw_hmin/(zpcbw_hmin - za0)))

          ENDIF

          z_hini2 = zhmin + SQRT(-zpcbw_hmin/zsqrtd)

#if defined(DEBUG_PHSOLVERS)
          PRINT*, '[HINI_ACBW_CO3] z_hini2(prov) :', z_hini2
#endif

                                    ! and make sure that z_hini1 is within
                                    ! [p_hinf(1),p_hsup(1)]
          z_hini1 = MAX(MIN(p_hsup(1), z_hini1), p_hinf(1))
                                    ! and z_hini2 within [p_hinf(2),p_hsup(2)]
          z_hini2 = MAX(MIN(p_hsup(2), z_hini2), p_hinf(2))

#if defined(DEBUG_PHSOLVERS)
          PRINT*, '[HINI_ACBW_CO3] z_hini1 :', z_hini1
          PRINT*, '[HINI_ACBW_CO3] z_hini2 :', z_hini2
#endif

        ENDIF


      ELSE                          ! If the cubic does not have distinct
                                    ! extrema, use the respective geometric
                                    ! mean of H_inf and H_sup for H_ini
        z_hini1 = SQRT(p_hinf(1)*p_hsup(1))
        z_hini2 = SQRT(p_hinf(2)*p_hsup(2))

      ENDIF

    END SELECT

  ELSE  ! zgamma == 0._wp

    ! The cubic equation degenerates to a quadratic

    IF (k_nroots == 0) THEN

      z_hini1 = pp_hnan
      z_hini2 = pp_hnan

    ELSE  ! k_nroots = 1

      zd = za1*za1 - 4._wp*za2*za0

      IF (za1 > 0._wp) THEN
        z_hini1 =                -( za1 + SQRT(zd) ) / (za2 + za2)
      ELSE
        z_hini1 = - (za0 + za0) / ( za1 - SQRT(zd) )
      ENDIF
                                    ! Still have to make sure that z_hini1
                                    ! is within [p_hinf(1),p_hsup(1)]
      z_hini1 = MAX(MIN(p_hsup(1), z_hini1), p_hinf(1))

      z_hini2 = pp_hnan

    ENDIF

  ENDIF

ENDIF

HINI_ACBW_CO3(1) = z_hini1
HINI_ACBW_CO3(2) = z_hini2


RETURN


!===============================================================================
 END FUNCTION HINI_ACBW_CO3
!===============================================================================




!===============================================================================
 SUBROUTINE HINFSUPINI(p_alktot,    p_dicvar, p_bortot,                        &
                                    p_po4tot, p_siltot,                        &
                                    p_nh4tot, p_h2stot,                        &
                                    p_so4tot, p_flutot,                        &
                                    k_dicsel, k_nroots,                        &
                                    p_hinf,   p_hsup,   p_hini                 )
!===============================================================================


IMPLICIT NONE


!--------------------!
! Argument variables !
!--------------------!

REAL(KIND=wp), INTENT(IN)    :: p_alktot
REAL(KIND=wp), INTENT(IN)    :: p_dicvar
REAL(KIND=wp), INTENT(IN)    :: p_bortot
REAL(KIND=wp), INTENT(IN)    :: p_po4tot
REAL(KIND=wp), INTENT(IN)    :: p_siltot
REAL(KIND=wp), INTENT(IN)    :: p_nh4tot
REAL(KIND=wp), INTENT(IN)    :: p_h2stot
REAL(KIND=wp), INTENT(IN)    :: p_so4tot
REAL(KIND=wp), INTENT(IN)    :: p_flutot

INTEGER,       INTENT(IN)    :: k_dicsel
INTEGER,       INTENT(OUT)   :: k_nroots

REAL(KIND=wp), DIMENSION(2), INTENT(OUT)   :: p_hinf
REAL(KIND=wp), DIMENSION(2), INTENT(OUT)   :: p_hsup
REAL(KIND=wp), DIMENSION(2), INTENT(INOUT) :: p_hini


!-----------------!
! Local variables !
!-----------------!

!  None


!-------------------------------------------------------------------------------

niter_alktan = 0
ahtan = pp_hnan
aatan = HUGE(1._wp)

ahmin_gampos  = pp_hnan
almin_gampos  = HUGE(1._wp)
acrit2_gampos = HUGE(1._wp)
acrit0_gampos = HUGE(1._wp)

SELECT CASE(k_dicsel)

CASE(jp_dic)
  CALL HINFSUPINI_DIC(p_alktot, p_dicvar, p_bortot,       &
                                p_po4tot, p_siltot,       &
                                p_nh4tot, p_h2stot,       &
                                p_so4tot, p_flutot,       &
                                p_hinf,   p_hsup,         &
                                p_hini,   k_nroots)

CASE(jp_co2)
  CALL HINFSUPINI_CO2(p_alktot, p_dicvar, p_bortot,       &
                                p_po4tot, p_siltot,       &
                                p_nh4tot, p_h2stot,       &
                                p_so4tot, p_flutot,       &
                                p_hinf,   p_hsup,         &
                                p_hini,   k_nroots)

CASE(jp_hco3)
  CALL HINFSUPINI_HCO3(p_alktot, p_dicvar, p_bortot,      &
                                p_po4tot, p_siltot,       &
                                p_nh4tot, p_h2stot,       &
                                p_so4tot, p_flutot,       &
                                p_hinf,   p_hsup,         &
                                p_hini,   k_nroots)

CASE(jp_co3)
  CALL HINFSUPINI_CO3(p_alktot, p_dicvar, p_bortot,       &
                                p_po4tot, p_siltot,       &
                                p_nh4tot, p_h2stot,       &
                                p_so4tot, p_flutot,       &
                                p_hinf,   p_hsup,         &
                                p_hini,   k_nroots)

CASE DEFAULT

  k_nroots = 0

  p_hinf(:) = pp_hnan
  p_hsup(:) = pp_hnan
  p_hini(:) = pp_hnan

END SELECT


RETURN


!===============================================================================
 END SUBROUTINE HINFSUPINI
!===============================================================================




!===============================================================================
 SUBROUTINE HINFSUPINI_DIC(p_alktot, p_dictot, p_bortot,                       &
                                    p_po4tot, p_siltot,                        &
                                    p_nh4tot, p_h2stot,                        &
                                    p_so4tot, p_flutot,                        &
                                    p_hinf,   p_hsup,                          &
                                    p_hini,   k_nroots                         )
!===============================================================================

! Subroutine provides
!  - the infimum and the supremum of Alk_{nW}, obtained from ANW_INFSUP
!  - a valid initial value for any solver, already brought within brackets.
! If the given p_hini is not equal to pp_hnan, the calculation of h_ini from
! the ACB approximation is skipped and the provided value is only brought into
! [p_hinf, p_hsup] if necessary.


USE MOD_CHEMCONST, ONLY: api1_wat, aphscale


IMPLICIT NONE


!--------------------!
! Argument variables !
!--------------------!

REAL(KIND=wp), INTENT(IN)                  ::  p_alktot
REAL(KIND=wp), INTENT(IN)                  ::  p_dictot
REAL(KIND=wp), INTENT(IN)                  ::  p_bortot
REAL(KIND=wp), INTENT(IN)                  ::  p_po4tot
REAL(KIND=wp), INTENT(IN)                  ::  p_siltot
REAL(KIND=wp), INTENT(IN)                  ::  p_nh4tot
REAL(KIND=wp), INTENT(IN)                  ::  p_h2stot
REAL(KIND=wp), INTENT(IN)                  ::  p_so4tot
REAL(KIND=wp), INTENT(IN)                  ::  p_flutot

REAL(KIND=wp), DIMENSION(2), INTENT(OUT)   ::  p_hinf
REAL(KIND=wp), DIMENSION(2), INTENT(OUT)   ::  p_hsup
REAL(KIND=wp), DIMENSION(2), INTENT(INOUT) ::  p_hini

INTEGER,                     INTENT(OUT)   ::  k_nroots

!-----------------!
! Local variables !
!-----------------!

REAL(KIND=wp) ::  zalknw_inf, zalknw_sup
REAL(KIND=wp) ::  za3, za2, za1, za0
REAL(KIND=wp) ::  zd, zsqrtd, z_hcmin
REAL(KIND=wp) ::  z_hinf,  z_hsup


!-------------------------------------------------------------------------------


k_nroots  = 1


CALL ANW_INFSUP(p_dictot, p_bortot,   &
                p_po4tot, p_siltot,   &
                p_nh4tot, p_h2stot,   &
                p_so4tot, p_flutot,   &
                zalknw_inf, zalknw_sup)

za1 = p_alktot - zalknw_inf
zd  = za1**2 + 4._wp*api1_wat/aphscale

IF (za1 > 0) THEN
  z_hinf = 2._wp*api1_wat / ( za1 + SQRT(zd) )
ELSE
  z_hinf =         aphscale*(-za1 + SQRT(zd) ) / 2._wp
ENDIF


za1 = p_alktot - zalknw_sup
zd  = za1**2 + 4._wp*api1_wat/aphscale

IF (za1 > 0) THEN
  z_hsup = 2._wp*api1_wat / ( za1 + SQRT(zd) )
ELSE
  z_hsup =       aphscale * (-za1 + SQRT(zd) ) / 2._wp
ENDIF


#ifdef SAFEGEOMEAN_INIT

IF (      (p_hini(1) == pp_hnan) &          ! request to calculate H_ini
     .OR. (p_hini(1) < z_hinf)   &          ! the given H_ini is too low
     .OR. (p_hini(1) > z_hsup) ) THEN       ! the given H_ini too high

  p_hini(1) = SQRT(z_hsup*z_hinf)

ENDIF

#else

IF (p_hini(1) == pp_hnan) THEN      ! request to calculate H_ini

# if defined(DEBUG_PHSOLVERS)
  PRINT*, '[HINFSUPINI] using HINI_ACB_DIC to set H_ini'
# endif

  p_hini(1) = HINI_ACB_DIC(p_alktot, p_dictot, p_bortot, z_hinf, z_hsup)

ELSE                                ! H_ini given: only bring it into
                                    ! [z_hinf, z_hsup]
  p_hini(1) = MAX(MIN(z_hsup, p_hini(1)), z_hinf)

ENDIF

#endif


p_hinf(1) = z_hinf
p_hsup(1) = z_hsup


p_hinf(2) = pp_hnan
p_hsup(2) = pp_hnan
p_hini(2) = pp_hnan


RETURN


!===============================================================================
 END SUBROUTINE HINFSUPINI_DIC
!===============================================================================




!===============================================================================
 SUBROUTINE HINFSUPINI_CO2(p_alktot, p_co2,   p_bortot,                        &
                                    p_po4tot, p_siltot,                        &
                                    p_nh4tot, p_h2stot,                        &
                                    p_so4tot, p_flutot,                        &
                                    p_hinf,   p_hsup,                          &
                                    p_hini,   k_nroots                         )
!===============================================================================

! Subroutine provides
!  - the infimum and the supremum of Alk_{nW}, obtained from ANW_INFSUP
!  - a valid initial value for any solver, already brought within brackets.
! If the given p_hini is not equal to pp_hnan, the calculation of h_ini from
! the ACB approximation is skipped and the provided value is only brought into
! [p_hinf, p_hsup] if necessary.


USE MOD_CHEMCONST, ONLY: api1_wat, aphscale, api1_dic, api2_dic


IMPLICIT NONE


!--------------------!
! Argument variables !
!--------------------!

REAL(KIND=wp), INTENT(IN)                  :: p_alktot
REAL(KIND=wp), INTENT(IN)                  :: p_co2
REAL(KIND=wp), INTENT(IN)                  :: p_bortot
REAL(KIND=wp), INTENT(IN)                  :: p_po4tot
REAL(KIND=wp), INTENT(IN)                  :: p_siltot
REAL(KIND=wp), INTENT(IN)                  :: p_nh4tot
REAL(KIND=wp), INTENT(IN)                  :: p_h2stot
REAL(KIND=wp), INTENT(IN)                  :: p_so4tot
REAL(KIND=wp), INTENT(IN)                  :: p_flutot

REAL(KIND=wp), DIMENSION(2), INTENT(OUT)   :: p_hinf
REAL(KIND=wp), DIMENSION(2), INTENT(OUT)   :: p_hsup
REAL(KIND=wp), DIMENSION(2), INTENT(INOUT) :: p_hini

INTEGER,       INTENT(OUT)                 :: k_nroots

!-----------------!
! Local variables !
!-----------------!

REAL(KIND=wp) ::  zalknwc_inf, zalknwc_sup
REAL(KIND=wp) ::  z_dictot
REAL(KIND=wp) ::  za3, za2, za1, za0
REAL(KIND=wp) ::  zd, zsqrtd, z_hcmin
REAL(KIND=wp) ::  z_hinf,  z_hsup


!-------------------------------------------------------------------------------


k_nroots  = 1

                                    ! Get Alk_nWCinf and Alk_nWCinf:
z_dictot = 0._wp                    ! set C_T to 0 an call ANW_INFSUP

CALL ANW_INFSUP(z_dictot, p_bortot,     &
                p_po4tot, p_siltot,     &
                p_nh4tot, p_h2stot,     &
                p_so4tot, p_flutot,     &
                zalknwc_inf, zalknwc_sup)

! Lower root bracket
! would normally require the solution of a cubic equation.  It is 
! nevertheless sufficient to chose the location of the minimum of the
! cubic (written such that a_3 > 0).

                                    ! Coefficients of cubic polynomial
za3 = 1._wp/aphscale
za2 = (p_alktot - zalknwc_inf)
za1 = -(api1_dic * p_co2 + api1_wat)
! za0 = -2._wp * api2_dic * p_dicvar

                                    ! The derivative has as constant
                                    ! term za1 < 0 => one positive and
                                    ! one negative root.
                                    ! Determine the positive one,
                                    ! i.e., the greater one of the two
zd   = za2**2 - 3._wp*za1*za3

IF (za2 > 0) THEN
  z_hinf = -za1 / ( za2 + SQRT(zd) )
ELSE
  z_hinf =        (-za2 + SQRT(zd) ) / (3._wp*za3)
ENDIF


! Upper root bracket:
! would normally require the solution of a cubic equation.
! It is nevertheless sufficient to chose the greater of the roots of
! the quadratic expansion around the minimum of the cubic (written
! such that a_3 > 0)

                                    ! Coefficients of cubic polynomial
za3 = 1._wp/aphscale
za2 = (p_alktot - zalknwc_sup)
za1 = -(api1_dic*p_co2 + api1_wat)
za0 = -2._wp * api2_dic * p_co2

                                    ! The derivative has as constant
                                    ! term za1 < 0 => one positive and
                                    ! one negative root.
                                    ! Determine the positive one,
                                    ! i.e., the greater one of the two
zd     = za2**2 - 3._wp*za1*za3
zsqrtd = SQRT(zd)

IF (za2 > 0) THEN
  z_hcmin =     -za1 / ( za2 + zsqrtd )
ELSE
  z_hcmin =            (-za2 + zsqrtd ) / (3._wp*za3)
ENDIF

                                  ! 
z_hsup = z_hcmin + SQRT(-(za0 + z_hcmin*(za1 + z_hcmin*(za2 + z_hcmin*za3)))/zsqrtd)


#ifdef SAFEGEOMEAN_INIT

IF (      (p_hini(1) == pp_hnan) &          ! request to calculate H_ini
     .OR. (p_hini(1) < z_hinf)   &          ! the given H_ini is too low
     .OR. (p_hini(1) > z_hsup) ) THEN       ! the given H_ini too high

  p_hini(1) = SQRT(z_hsup*z_hinf)

ENDIF

#else

IF (p_hini(1) == pp_hnan) THEN      ! request to calculate H_ini

# if defined(DEBUG_PHSOLVERS)
  PRINT*, '[HINFSUPINI] using HINI_ACB_CO2 to set H_ini'
# endif

  p_hini(1) = HINI_ACB_CO2(p_alktot, p_co2, p_bortot, z_hinf, z_hsup)

ELSE                                ! H_ini given: only bring it into
                                    ! [z_hinf, z_hsup]
  p_hini(1) = MAX(MIN(z_hsup, p_hini(1)), z_hinf)

ENDIF

#endif


p_hinf(1) = z_hinf
p_hsup(1) = z_hsup


p_hinf(2) = pp_hnan
p_hsup(2) = pp_hnan
p_hini(2) = pp_hnan


RETURN


!===============================================================================
 END SUBROUTINE HINFSUPINI_CO2
!===============================================================================




!===============================================================================
 SUBROUTINE HINFSUPINI_HCO3(p_alktot, p_hco3, p_bortot,                        &
                                    p_po4tot, p_siltot,                        &
                                    p_nh4tot, p_h2stot,                        &
                                    p_so4tot, p_flutot,                        &
                                    p_hinf,   p_hsup,                          &
                                    p_hini,   k_nroots                         )
!===============================================================================

! Subroutine provides
!  - the infimum and the supremum of Alk_{nW}, obtained from ANW_INFSUP
!  - a valid initial value for any solver, already brought within brackets.
! If the given p_hini is not equal to pp_hnan, the calculation of h_ini from
! the ACB approximation is skipped and the provided value is only brought into
! [p_hinf, p_hsup] if necessary.


USE MOD_CHEMCONST, ONLY: api1_wat, aphscale, api1_dic, api2_dic


IMPLICIT NONE


!--------------------!
! Argument variables !
!--------------------!

REAL(KIND=wp), INTENT(IN)                  :: p_alktot
REAL(KIND=wp), INTENT(IN)                  :: p_hco3
REAL(KIND=wp), INTENT(IN)                  :: p_bortot
REAL(KIND=wp), INTENT(IN)                  :: p_po4tot
REAL(KIND=wp), INTENT(IN)                  :: p_siltot
REAL(KIND=wp), INTENT(IN)                  :: p_nh4tot
REAL(KIND=wp), INTENT(IN)                  :: p_h2stot
REAL(KIND=wp), INTENT(IN)                  :: p_so4tot
REAL(KIND=wp), INTENT(IN)                  :: p_flutot

REAL(KIND=wp), DIMENSION(2), INTENT(OUT)   :: p_hinf
REAL(KIND=wp), DIMENSION(2), INTENT(OUT)   :: p_hsup
REAL(KIND=wp), DIMENSION(2), INTENT(INOUT) :: p_hini

INTEGER,       INTENT(OUT)                 :: k_nroots

!-----------------!
! Local variables !
!-----------------!

REAL(KIND=wp) ::  zalknwc_inf, zalknwc_sup
REAL(KIND=wp) ::  z_dictot
REAL(KIND=wp) ::  za3, za2, za1, za0
REAL(KIND=wp) ::  zd, zsqrtd, z_hcmin
REAL(KIND=wp) ::  z_hinf,  z_hsup


!-------------------------------------------------------------------------------


k_nroots  = 1


z_dictot = 0._wp

CALL ANW_INFSUP(z_dictot, p_bortot,     &
                p_po4tot, p_siltot,     &
                p_nh4tot, p_h2stot,     &
                p_so4tot, p_flutot,     &
                zalknwc_inf, zalknwc_sup)

! za2 =  1._wp/aphscale
za1 =  p_alktot - zalknwc_inf - p_hco3
za0 = -(2._wp*(api2_dic/api1_dic)*p_hco3 + api1_wat)

zd  = za1**2 - 4._wp * za0 / aphscale

IF (za1 > 0) THEN
  z_hinf = -2._wp * za0 / ( za1 + SQRT(zd) )
ELSE
  z_hinf =       aphscale*(-za1 + SQRT(zd) ) / 2._wp
ENDIF


! za2 =  1._wc/aphscale
za1 =  p_alktot - zalknwc_sup - p_hco3
! za0 = -(2._wp*(api2_dic/api1_dic)*p_hco3 + api1_wat)

zd    = za1**2 - 4._wp * za0 / aphscale

IF (za1 > 0) THEN
  z_hsup = -2._wp * za0 / ( za1 + SQRT(zd) )
ELSE
  z_hsup =     aphscale * (-za1 + SQRT(zd) ) / 2._wp
ENDIF


# ifdef SAFEGEOMEAN_INIT

IF (      (p_hini(1) == pp_hnan) &          ! request to calculate H_ini
     .OR. (p_hini(1) < z_hinf)   &          ! the given H_ini is too low
     .OR. (p_hini(1) > z_hsup) ) THEN       ! the given H_ini too high

  p_hini(1) = SQRT(z_hsup*z_hinf)

ENDIF

# else

IF (p_hini(1) == pp_hnan) THEN       ! request to calculate H_ini
#   if defined(DEBUG_PHSOLVERS)
  PRINT*, '[HINFSUPINI] using HINI_ACBW_HCO3 to set H_ini'
#   endif

  p_hini(1) = HINI_ACBW_HCO3(p_alktot, p_hco3, p_bortot, z_hinf, z_hsup)

ELSE                                ! H_ini given: only bring it into
                                    ! [z_hinf, z_hsup]
  p_hini(1) = MAX(MIN(z_hsup, p_hini(1)), z_hinf)

ENDIF

# endif


p_hinf(1) = z_hinf
p_hsup(1) = z_hsup


p_hinf(2) = pp_hnan
p_hsup(2) = pp_hnan
p_hini(2) = pp_hnan


RETURN


!===============================================================================
 END SUBROUTINE HINFSUPINI_HCO3
!===============================================================================




!===============================================================================
 SUBROUTINE HINFSUPINI_CO3(p_alktot, p_co3,   p_bortot,                        &
                                    p_po4tot, p_siltot,                        &
                                    p_nh4tot, p_h2stot,                        &
                                    p_so4tot, p_flutot,                        &
                                    p_hinf,   p_hsup,                          &
                                    p_hini,   k_nroots                         )
!===============================================================================

! Subroutine provides
!  - the infimum and the supremum of Alk_{nW}, obtained from ANW_INFSUP
!  - a valid initial value for any solver, already brought within brackets.
! If the given p_hini is not equal to pp_hnan, the calculation of h_ini from
! the ACB approximation is skipped and the provided value is only brought into
! [p_hinf, p_hsup] if necessary.


USE MOD_CHEMCONST, ONLY: api1_wat, aphscale, api1_dic, api2_dic


IMPLICIT NONE


!--------------------!
! Argument variables !
!--------------------!

REAL(KIND=wp), INTENT(IN)                  :: p_alktot
REAL(KIND=wp), INTENT(IN)                  :: p_co3
REAL(KIND=wp), INTENT(IN)                  :: p_bortot
REAL(KIND=wp), INTENT(IN)                  :: p_po4tot
REAL(KIND=wp), INTENT(IN)                  :: p_siltot
REAL(KIND=wp), INTENT(IN)                  :: p_nh4tot
REAL(KIND=wp), INTENT(IN)                  :: p_h2stot
REAL(KIND=wp), INTENT(IN)                  :: p_so4tot
REAL(KIND=wp), INTENT(IN)                  :: p_flutot

REAL(KIND=wp), DIMENSION(2), INTENT(OUT)   :: p_hinf
REAL(KIND=wp), DIMENSION(2), INTENT(OUT)   :: p_hsup
REAL(KIND=wp), DIMENSION(2), INTENT(INOUT) :: p_hini

INTEGER,       INTENT(OUT)                 :: k_nroots

!-----------------!
! Local variables !
!-----------------!

REAL(KIND=wp) ::  zalknwc_inf, zalknwc_sup
REAL(KIND=wp) ::  zalknwc_asympt_coeff
REAL(KIND=wp) ::  z_dictot
REAL(KIND=wp) ::  z_gamma
REAL(KIND=wp) ::  z_hmin, z_lmin, z_amin, zalknwc_hmin
REAL(KIND=wp) ::  z_htan, z_atan, z_tol
REAL(KIND=wp) ::  za3, za2, za1, za0
REAL(KIND=wp) ::  zd, zsqrtd, zdiff
REAL(KIND=wp) ::  z_hcmin
REAL(KIND=wp) ::  z_pcbw_hmin, z_pcbw_hmax
REAL(KIND=wp) ::  z_hinf1, z_hsup1
REAL(KIND=wp) ::  z_hinf2, z_hsup2

REAL(KIND=wp), DIMENSION(2) ::  z_hinf, z_hsup, z_hini


!-------------------------------------------------------------------------------


z_dictot = 0._wp

CALL ANW_INFSUP(z_dictot, p_bortot,       &
                p_po4tot, p_siltot,       &
                p_nh4tot, p_h2stot,       &
                p_so4tot, p_flutot,       &
                zalknwc_inf, zalknwc_sup, &
                zalknwc_asympt_coeff      )


z_gamma = p_co3 * (api1_dic/api2_dic) - 1._wp/aphscale

#if defined(DEBUG_PHSOLVERS)
PRINT*, '[HINFSUPINI_CO3] z_gamma  :', z_gamma
#endif

IF (z_gamma < 0._wp) THEN !-----------------------------------------------------

  k_nroots  = 1

                                    ! H_inf(1) = abscissa of P_UL,
                                    ! which is the positive root of
                                    ! the following quadratic
  za2 = z_gamma
  za1 = - (p_alktot - (p_co3 + p_co3) - zalknwc_inf)
  za0 = api1_wat


  zd  = za1**2 - 4._wp * api1_wat * z_gamma

  IF (za1 > 0) THEN
    z_hinf1 =               -( za1 + SQRT(zd) ) / (za2 + za2)
  ELSE
    z_hinf1 = -(za0 + za0) / ( za1 - SQRT(zd) )
  ENDIF


                                    ! H_sup(1) = abscissa of P_LL,
                                    ! which is the positive root of
                                    ! the following quadratic
  za2 = z_gamma
  za1 = - (p_alktot - (p_co3 + p_co3) - zalknwc_sup)
  za0 = api1_wat

  zd  = za1**2 - 4._wp * za0 * za2

  IF (za1 > 0._wp) THEN
    z_hsup1 = -(za0 + za0) / ( za1 - SQRT(zd) )
  ELSE
    z_hsup1 =               -( za1 + SQRT(zd) ) / (za2 + za2)
  ENDIF

  z_hinf2 = pp_hnan
  z_hsup2 = pp_hnan

ELSEIF (z_gamma > 0._wp) THEN !-------------------------------------------------

  ! 0, 1, or 2 roots ?

  z_hmin = SQRT(api1_wat/z_gamma)
  z_lmin = 2._wp * SQRT(z_gamma * api1_wat) + (p_co3 + p_co3)


  zalknwc_hmin = ANW(z_hmin,                     &
                      p_co3,    p_bortot,        &
                      p_po4tot, p_siltot,        &
                      p_nh4tot, p_h2stot,        &
                      p_so4tot, p_flutot,  jp_co3)

  ahmin_gampos  = z_hmin
  almin_gampos  = z_lmin
  acrit2_gampos = z_lmin + zalknwc_hmin
  acrit0_gampos = z_lmin + zalknwc_inf


  IF (p_alktot >= (z_lmin + zalknwc_hmin)) THEN


    k_nroots = 2

                                    ! H_inf(1) = abscissa of P_UL, 
                                    ! which is the lower (positive)
                                    ! root of the following quadratic;
                                    ! H_sup(2) = abscissa of P_UR, 
                                    ! which is the greater (positive)
                                    ! root of the following quadratic;
    za2 =  z_gamma
    za1 = -(p_alktot - (p_co3 + p_co3) - zalknwc_inf)
    za0 =  api1_wat

    zd  = za1**2 - 4._wp * za0 * za2


    IF (za1 > 0) THEN
      z_hinf1 =               -( za1 + SQRT(zd) ) / (za2 + za2)
    ELSE
      z_hinf1 = -(za0 + za0) / ( za1 - SQRT(zd) )
    ENDIF


    IF (za1 > 0) THEN
      z_hsup2 = -(za0 + za0) / ( za1 + SQRT(zd) )
    ELSE
      z_hsup2 =                (-za1 + SQRT(zd) ) / (za2 + za2)
    ENDIF



    IF (p_alktot > (z_lmin + zalknwc_sup)) THEN

                                    ! H_sup(1) = abscissa of P_LL, 
                                    ! which is the lower (positive)
                                    ! root of the following quadratic;
                                    ! H_inf(2) = abscissa of P_LR, 
                                    ! which is the greater (positive)
                                    ! root of the following quadratic;
      za2 =  z_gamma
      za1 = -(p_alktot - (p_co3 + p_co3) - zalknwc_sup)
      za0 =  api1_wat

      zd  = za1**2 - 4._wp * za0 * za2

      IF (za1 > 0) THEN
        z_hsup1 =               -( za1 + SQRT(zd) ) / (za2 + za2)
      ELSE
        z_hsup1 = -(za0 + za0) / ( za1 - SQRT(zd) )
      ENDIF


      IF (za1 > 0) THEN
        z_hinf2 = -(za0 + za0) / ( za1 + SQRT(zd) )
      ELSE
        z_hinf2 =                (-za1 + SQRT(zd) ) / (za2 + za2)
      ENDIF


    ELSE

      z_hsup1 = z_hmin
      z_hinf2 = z_hmin

    ENDIF

  ELSEIF (p_alktot <= (z_lmin + zalknwc_inf)) THEN

    k_nroots = 0

    z_hinf1  = pp_hnan
    z_hsup1  = pp_hnan

    z_hinf2  = pp_hnan
    z_hsup2  = pp_hnan


  ELSE
                                    ! Alk_T is in the intermediate region
                                    ! where we need to determine H_tan and Alk_tan

                                    ! Brackets of the H_tan:
                                    ! (H_min, A_min) and (H_UR, A_UR)

                                    ! H_UR = abscissa of P_UR, 
                                    ! which is the greater (positive)
                                    ! root of the following quadratic:
                                    ! If Alk_T > Alk_tan, H_UR will also be
                                    ! H_sup(2)
    za2 =  z_gamma
    za1 = -(p_alktot - (p_co3 + p_co3) - zalknwc_inf)
    za0 =  api1_wat

    zd  = za1**2 - 4._wp * za0 * za2

    IF (za1 > 0) THEN
      z_hsup2 = -(za0 + za0) / ( za1 + SQRT(zd) )
    ELSE
      z_hsup2 =                (-za1 + SQRT(zd) ) / (za2 + za2)
    ENDIF



    z_tol = z_hmin * pp_rdel_ah_target      ! We assume that H_min is
                                            ! of the same order of magnitude
                                            ! as H_tan for fixing the absolute
                                            ! tolerance on H_tan

    z_atan = ALK_TAN(z_hmin, z_hsup2, z_tol, z_htan)

    ahtan = z_htan
    aatan = z_atan

    IF (p_alktot < z_atan) THEN

      k_nroots = 0

      z_hinf1  = pp_hnan
      z_hsup1  = pp_hnan

      z_hinf2  = pp_hnan
      z_hsup2  = pp_hnan
 
    ELSEIF (p_alktot > z_atan) THEN

      k_nroots = 2

                                    ! H_inf(1) = abscissa of P_UL, 
                                    ! which is the lower (positive)
                                    ! root of the following quadratic;
                                    ! H_sup(2) = abscissa of P_UR, which
                                    ! has already served as a bracket for H_tan
      za2 =  z_gamma
      za1 = -(p_alktot - (p_co3 + p_co3) - zalknwc_inf)
      za0 =  api1_wat

      zd = za1*za1 - 4._wp * za2 * za0

      IF (za1 > 0) THEN
        z_hinf1 = -(za0 + za0) / ( za1 - SQRT(zd) )
      ELSE
        z_hinf1 =               -( za1 + SQRT(zd) ) / (za2 + za2)
      ENDIF


      z_hsup1  = z_htan
      z_hinf2  = z_htan

    ELSE

      k_nroots = 1

      z_hinf1  = z_htan
      z_hsup1  = z_htan

      z_hinf2  = pp_hnan
      z_hsup2  = pp_hnan

    ENDIF

  ENDIF


ELSE ! z_gamma = 0 -----------------------------------------------------

  zdiff = p_alktot - (p_co3 + p_co3) - zalknwc_inf

  IF (zdiff > 0) THEN

    k_nroots = 1

    z_hinf1  = api1_wat / zdiff
    z_hsup1  = (api1_wat + zalknwc_asympt_coeff)  / zdiff

    z_hinf2  = pp_hnan
    z_hsup2  = pp_hnan

  ELSE

    k_nroots = 0
    
    z_hinf1  = pp_hnan
    z_hsup1  = pp_hnan

    z_hinf2  = pp_hnan
    z_hsup2  = pp_hnan

  ENDIF

ENDIF !-----------------------------------------------------------------
  

z_hinf(1) = z_hinf1
z_hsup(1) = z_hsup1

z_hinf(2) = z_hinf2
z_hsup(2) = z_hsup2


# ifdef SAFEGEOMEAN_INIT

IF (      (p_hini(1) == pp_hnan) &          ! request to calculate H_ini
     .OR. (p_hini(1) < z_hinf1)  &          ! the given H_ini is too low
     .OR. (p_hini(1) > z_hsup1) ) THEN      ! the given H_ini too high

  p_hini(1) = SQRT(z_hsup1*z_hinf1)

ENDIF

IF (      (p_hini(2) == pp_hnan) &          ! request to calculate H_ini
     .OR. (p_hini(2) < z_hinf2)  &          ! the given H_ini is too low
     .OR. (p_hini(2) > z_hsup2) ) THEN      ! the given H_ini too high

  p_hini(2) = SQRT(z_hsup2*z_hinf2)

ENDIF

#else

z_hini(:) = HINI_ACBW_CO3(p_alktot, p_co3, p_bortot, z_hinf, z_hsup, k_nroots)

IF (p_hini(1) == pp_hnan) THEN      ! request to calculate H_ini(1)
  p_hini(1) = z_hini(1)
ELSE                                ! H_ini(1) given: only bring it into
  p_hini(1) = MAX(MIN(z_hsup1, p_hini(1)), z_hinf1) ! [z_hinf1, z_hsup1]
ENDIF

IF (p_hini(2) == pp_hnan) THEN      ! request to calculate H_ini(1)
  p_hini(2) = z_hini(2)
ELSE                                ! H_ini(2) given: only bring it into
  p_hini(2) = MAX(MIN(z_hsup2, p_hini(2)), z_hinf2) ! [z_hinf2, z_hsup2]
ENDIF

#endif


p_hinf(:) = z_hinf(:)
p_hsup(:) = z_hsup(:)


RETURN


CONTAINS

!===============================================================================
 FUNCTION ALK_TAN(p_a, p_b, p_tol, p_h                                         )
!===============================================================================

!  This function determines the H^+ concentration H_tan where Alk attains
!  a minimum on the interval [H_a, H_b]. It also returns the Alk_tan value
!  in H_tan
!
!  Input:
!
!   - p_a   lower endpoint of the bracketing interval (H_a)
!   - p_b   greater endpoint of the bracketing interval (H_b)
!   - tol   imposed tolerance (bracketting interval length
!
!  Output:
!
!   - p_h   abcissa where Alk attains its minimum.
!   - ALK_TAN  Alk value at p_h
!
!  This function subprogram is a slightly modified version of the
!  Algol 60 procedure localmin presented in Richard Brent, Algorithms
!  for Minimization Without Derivatives, Prentice-Hall, Inc. (1973),
!  republished by Dover Publications (2002):

!  - translated from the Algol 60 version in section 5.8
!    of Brent (1973/2002) to Fortran 90 by Guy Munhoven;
!  - including corrections to errata reported by R. Brent on
!    https://maths-people.anu.edu.au/~brent/pub/pub011.html (date of
!    last modification not reported; retrieved 10th December 2020);
!  - including modifications by the author himself as in
!    https://www.netlib.org/go/fmin.f (date of last modification not
!    reported; retrieved 10th December 2020);
!  - verified against function local_min in
!    https://people.sc.fsu.edu/~jburkardt/f_src/brent/brent.f90
!    by John Burkardt (last reported modification: 01st July 2013;
!    retrieved 10th December 2020)
!
!  The algorithm combines the golden section search and successive
!  parabolic interpolation. The function minimised here (EQUATION_AT,
!  with dicvar = 0) has a continuous second derivative which is positive
!  at the minimum and this minimum lies strictly between H_a and H_b.
!  Convergence  may thus be expected to be of the order of about 1.324.
!  The function is never evaluated at two points closer together than
!  eps*abs(fmin)+(tol/3), where eps = SQRT(EPSILON(1._wp).
!  The approximation of the abscissa of the minimum may be expected to
!  present an error lower than  3*eps*abs(fmin)+tol.


USE MOD_PRECISION


IMPLICIT NONE


REAL(KIND=wp) ::  ALK_TAN


!--------------------!
! Argument variables !
!--------------------!

REAL(KIND=wp), INTENT(IN)  ::  p_a, p_b
REAL(KIND=wp), INTENT(IN)  ::  p_tol
REAL(KIND=wp), INTENT(OUT) ::  p_h


!-----------------!
! Local variables !
!-----------------!

REAL(KIND=wp) ::  z_a, z_b, z_m
REAL(KIND=wp) ::  z_h
REAL(KIND=wp) ::  z_fh
REAL(KIND=wp) ::  z_tol, z_2tol, z_tol_3
REAL(KIND=wp) ::  z_d, z_e, z_eps
REAL(KIND=wp) ::  z_u,  z_v,  z_w
REAL(KIND=wp) ::  z_fu, z_fv, z_fw
REAL(KIND=wp) ::  z_p, z_q, z_r
REAL(KIND=wp) ::  pp_alktot0 = 0._wp

!  pp_c is the squared inverse of the golden ratio:
!  pp_c = (2/(1 + sqrt(5))**2 = (3 - sqrt(5))/2
!  21-digit decimal approximation from
!  https://www.wolframalpha.com/input/?i=0.5*%283.0-SQRT%285.0%29%29
REAL(KIND=wp), PARAMETER :: pp_c = 0.381966011250105151795_wp

!  eps is approximately the square root of the relative machine
!  precision.


!-------------------------------------------------------------------------------


z_tol = EPSILON(1._wp) + 1._wp

z_eps = SQRT(EPSILON(1._wp))

z_a = p_a
z_b = p_b

z_h = z_a + pp_c * (z_b - z_a)

z_v = z_h
z_w = z_h

z_d = 0._wp
z_e = 0._wp

z_fh = EQUATION_AT(pp_alktot0,  z_h,      &
                      p_co3,    p_bortot, &
                      p_po4tot, p_siltot, &
                      p_nh4tot, p_h2stot, &
                      p_so4tot, p_flutot, &
                      jp_co3              )

z_fv = z_fh
z_fw = z_fh

z_tol_3 = p_tol/3._wp


DO

  niter_alktan = niter_alktan + 1

  z_m    = 0.5_wp * (z_a + z_b)

  z_tol  = z_eps*ABS(z_h) + z_tol_3
  z_2tol = z_tol + z_tol

                                    ! Check stopping criterion
  IF (ABS(z_h - z_m) <= (z_2tol - 0.5_wp*(z_b - z_a))) EXIT

  z_p = 0._wp
  z_q = 0._wp
  z_r = 0._wp

  IF (ABS(z_e) > z_tol) THEN        ! Fit a parabola

    z_r = (z_h - z_w) * (z_fh - z_fv)
    z_q = (z_h - z_v) * (z_fh - z_fw)
    z_p = (z_h - z_v) * z_q - (z_h - z_w) * z_r
    z_q = 2._wp * (z_q - z_r)

    IF (z_q > 0._wp) THEN
      z_p = -z_p
    ELSE
      z_q = -z_q
    ENDIF

    z_r = z_e
    z_e = z_d

  ENDIF


  IF ( (ABS(z_p) < ABS(0.5_wp*z_q*z_r)) &
       .AND. (z_p > z_q*(z_a - z_h))    &
       .AND. (z_p < z_q*(z_b - z_h))    ) THEN

                                    ! "Parabolic interpolation" step
    z_d = z_p / z_q
    z_u = z_h + z_d
                                    ! Function must not be evaluated
                                    ! too close to z_a or z_b
    IF ( ((z_u - z_a) < z_2tol) .OR. ((z_b - z_u) < z_2tol) ) THEN

      IF (z_h < z_m) THEN
        z_d =  z_tol
      ELSE
        z_d = -z_tol
      ENDIF

    ENDIF

  ELSE                              !  "Golden section" step

    IF (z_h < z_m) THEN
      z_e = z_b - z_h
    ELSE
      z_e = z_a - z_h
    ENDIF

    z_d = pp_c * z_e

  ENDIF

                                    ! New function evaluation at z_u;
  IF (ABS(z_d) >= z_tol) THEN       ! z_u must not be too close to z_h
    z_u = z_h + z_d
  ELSEIF (z_d > 0._wp) THEN
    z_u = z_h + z_tol
  ELSE
    z_u = z_h - z_tol
  ENDIF

  z_fu = EQUATION_AT(pp_alktot0, z_u,      &
                      p_co3,    p_bortot, &
                      p_po4tot, p_siltot, &
                      p_nh4tot, p_h2stot, &
                      p_so4tot, p_flutot, &
                      jp_co3              )

                                    ! Update z_a, z_b, z_v, z_w, and z_h
  IF (z_fu <= z_fh) THEN

    IF (z_u < z_h) THEN
      z_b = z_h
    ELSE
      z_a = z_h
    ENDIF

    z_v  = z_w
    z_fv = z_fw

    z_w  = z_h
    z_fw = z_fh

    z_h  = z_u
    z_fh = z_fu            

  ELSE

    IF (z_u < z_h) THEN
      z_a = z_u
    ELSE
      z_b = z_u
    ENDIF

    IF ((z_fu <= z_fw) .OR. (z_w == z_h)) THEN

      z_v  = z_w
      z_fv = z_fw

      z_w  = z_u
      z_fw = z_fu

    ELSEIF ((z_fu <= z_fv) .OR. (z_v == z_h) .OR. (z_v == z_w)) THEN

      z_v  = z_u
      z_fv = z_fu

    ENDIF

  ENDIF

ENDDO


p_h     = z_h                       ! Copy location of minimum into p_h.
ALK_TAN = z_fh                      ! Return the value of minimum as result


RETURN


!===============================================================================
 END FUNCTION ALK_TAN
!===============================================================================


!===============================================================================
 END SUBROUTINE HINFSUPINI_CO3
!===============================================================================




!===============================================================================
 FUNCTION SOLVE_AT_GENERAL(p_alktot, p_dictot, p_bortot,                       &
                           p_po4tot, p_siltot, p_nh4tot, p_h2stot,             &
                           p_so4tot, p_flutot,                                 &
                           p_hini,   p_val)
!===============================================================================

! Wrapper for compatibility with SolveSAPHE v. 1.0.x

IMPLICIT NONE

REAL(KIND=wp) :: SOLVE_AT_GENERAL


!--------------------!
! Argument variables !
!--------------------!

REAL(KIND=wp), INTENT(IN)            :: p_alktot
REAL(KIND=wp), INTENT(IN)            :: p_dictot
REAL(KIND=wp), INTENT(IN)            :: p_bortot
REAL(KIND=wp), INTENT(IN)            :: p_po4tot
REAL(KIND=wp), INTENT(IN)            :: p_siltot
REAL(KIND=wp), INTENT(IN)            :: p_nh4tot
REAL(KIND=wp), INTENT(IN)            :: p_h2stot
REAL(KIND=wp), INTENT(IN)            :: p_so4tot
REAL(KIND=wp), INTENT(IN)            :: p_flutot

REAL(KIND=wp), INTENT(IN),  OPTIONAL :: p_hini
REAL(KIND=wp), INTENT(OUT), OPTIONAL :: p_val


!-----------------!
! Local variables !
!-----------------!

REAL(KIND=wp), DIMENSION(2) ::  z_hini, z_val
REAL(KIND=wp), DIMENSION(2) ::  z_h

INTEGER ::  n_roots

IF (PRESENT(p_hini)) THEN

  z_hini(1) = p_hini
  z_hini(2) = pp_hnan

  IF (PRESENT(p_val)) THEN
    z_h(:) = SOLVE_AT_GENERAL2(p_alktot, p_dictot, p_bortot,                   &
                               p_po4tot, p_siltot, p_nh4tot, p_h2stot,         &
                               p_so4tot, p_flutot,                             &
                               jp_dic,   n_roots,                              &
                               z_hini,   z_val)
    p_val = z_val(1)
  ELSE
    z_h(:) = SOLVE_AT_GENERAL2(p_alktot, p_dictot, p_bortot,                   &
                               p_po4tot, p_siltot, p_nh4tot, p_h2stot,         &
                               p_so4tot, p_flutot,                             &
                               jp_dic,   n_roots,                              &
                               z_hini)
  ENDIF

ELSE

  IF (PRESENT(p_val)) THEN
    z_h(:) = SOLVE_AT_GENERAL2(p_alktot, p_dictot, p_bortot,                   &
                               p_po4tot, p_siltot, p_nh4tot, p_h2stot,         &
                               p_so4tot, p_flutot,                             &
                               jp_dic,   n_roots,                              &
                               p_val = z_val)
    p_val = z_val(1)
  ELSE
    z_h(:) = SOLVE_AT_GENERAL2(p_alktot, p_dictot, p_bortot,                   &
                               p_po4tot, p_siltot, p_nh4tot, p_h2stot,         &
                               p_so4tot, p_flutot,                             &
                               jp_dic,   n_roots)
  ENDIF
  
ENDIF


IF (n_roots == 1) THEN
  SOLVE_AT_GENERAL = z_h(1)
ELSE
  SOLVE_AT_GENERAL = pp_hnan
ENDIF

RETURN


!===============================================================================
 END FUNCTION SOLVE_AT_GENERAL
!===============================================================================





!===============================================================================
 FUNCTION SOLVE_AT_GENERAL2(p_alktot, p_dicvar, p_bortot,                      &
                            p_po4tot, p_siltot,                                &
                            p_nh4tot, p_h2stot,                                &
                            p_so4tot, p_flutot,                                &
                            k_dicsel, k_nroots,                                &
                            p_hini,   p_val                                    )
!===============================================================================

! Universal pH solver that converges from any given initial value,
! determines upper an lower bounds for the solution if required.
! p_dicvar stands for one of the four carbonate system related variables,
! depending on the value of k_dicsel:
! k_dicsel = jp_dic:  p_dicvar = DIC
! k_dicsel = jp_co2:  p_dicvar = [CO2]
! k_dicsel = jp_hco3: p_dicvar = [HCO3]
! k_dicsel = jp_co3:  p_dicvar = [CO3]


IMPLICIT NONE


!------------------------------!
! Function type and parameters !
!------------------------------!

REAL(KIND=wp), DIMENSION(2) :: SOLVE_AT_GENERAL2

                                        ! Threshold value for switching from
                                        ! pH space to [H^+] space iterations.
REAL(KIND=wp), PARAMETER :: pz_exp_threshold = 1.0_wp

                                        ! Greatest acceptable ratio of variation
                                        ! for a Newton iterate zh relative to its
                                        ! predecessor zh_prev:
                                        ! EXP(-pz_exp_upperlim)
                                        !    < zh/zh_prev
                                        !       < EXP(pz_exp_upperlim)
                                        ! i. e.,
                                        ! ABS(LOG(zh/zh_prev)) < pz_exp_upperlim
REAL(KIND=wp), PARAMETER :: pz_exp_upperlim  = 4.6_wp   ! EXP(4.6) = 100.

                                        ! Lowest limit for a Regula Falsi iterate
                                        ! zh relative to zh_min, as a fraction
                                        ! of the length of the [zh_min, zh_max] interval:
                                        ! zh_min + pz_rf_thetahmin * (zh_max - zh_min)
                                        !    <= zh <= zh_max - pz_rf_thetahmin * (zh_max - zh_min)
REAL(KIND=wp), PARAMETER :: pz_rf_thetamin   = 0.10_wp
REAL(KIND=wp), PARAMETER :: pz_rf_thetamax   = 1.00_wp - pz_rf_thetamin

                                        ! Maximum number of successive H_min/H_max changes
INTEGER,       PARAMETER :: jp_nmaxsucc      = 3


!--------------------!
! Argument variables !
!--------------------!

REAL(KIND=wp),               INTENT(IN)            :: p_alktot
REAL(KIND=wp),               INTENT(IN)            :: p_dicvar
REAL(KIND=wp),               INTENT(IN)            :: p_bortot
REAL(KIND=wp),               INTENT(IN)            :: p_po4tot
REAL(KIND=wp),               INTENT(IN)            :: p_siltot
REAL(KIND=wp),               INTENT(IN)            :: p_nh4tot
REAL(KIND=wp),               INTENT(IN)            :: p_h2stot
REAL(KIND=wp),               INTENT(IN)            :: p_so4tot
REAL(KIND=wp),               INTENT(IN)            :: p_flutot

INTEGER,                     INTENT(IN)            :: k_dicsel
INTEGER,                     INTENT(OUT)           :: k_nroots

REAL(KIND=wp), DIMENSION(2), INTENT(IN),  OPTIONAL :: p_hini
REAL(KIND=wp), DIMENSION(2), INTENT(OUT), OPTIONAL :: p_val


!-----------------!
! Local variables !
!-----------------!

REAL(KIND=wp), DIMENSION(2) ::  zh_inf, zh_sup, zh_ini

REAL(KIND=wp) :: zh_min,    zh_max,    zh_absmin
REAL(KIND=wp) :: zeqn_hmin, zeqn_hmax, zeqn_absmin
REAL(KIND=wp) :: zh, zh_prev, zh_lnfactor
REAL(KIND=wp) :: zh_delta
REAL(KIND=wp) :: zeqn, zdeqndh, zrf_hmin, zrf_hmax

INTEGER       :: i_root

INTEGER       :: nsucc_min, nsucc_max

LOGICAL       :: l_exitnow


!-------------------------------------------------------------------------------


IF (PRESENT(p_hini)) THEN
   zh_ini(:) = p_hini(:)
ELSE
   zh_ini(:) = pp_hnan
ENDIF


CALL HINFSUPINI(p_alktot, p_dicvar, p_bortot,       &
                          p_po4tot, p_siltot,       &
                          p_nh4tot, p_h2stot,       &
                          p_so4tot, p_flutot,       &
                          k_dicsel, k_nroots,       &
                          zh_inf,   zh_sup,   zh_ini)

ahmin_atgen_ini(:) = zh_inf(:)
ahmax_atgen_ini(:) = zh_sup(:)
ah_atgen_ini(:)    = zh_ini(:)

#if defined(DEBUG_PHSOLVERS)
PRINT*, '[SOLVE_AT_GENERAL2] n_roots  :', k_nroots
PRINT*, '[SOLVE_AT_GENERAL2] h_inf(:) :', zh_inf
PRINT*, '[SOLVE_AT_GENERAL2] h_sup(:) :', zh_sup
PRINT*, '[SOLVE_AT_GENERAL2] h_ini(:) :', zh_ini
#endif


DO i_root = 1, k_nroots

  niter_atgen(i_root)        = 0    ! Reset counters of iterations
  niter_atgen_limin(i_root)  = 0
  niter_atgen_limax(i_root)  = 0
  niter_atgen_limabs(i_root) = 0

  zh_min = zh_inf(i_root)
  zh_max = zh_sup(i_root)

  zeqn_hmin = EQUATION_AT(p_alktot, zh_min,    &
                          p_dicvar, p_bortot,  &
                          p_po4tot, p_siltot,  &
                          p_nh4tot, p_h2stot,  &
                          p_so4tot, p_flutot,  &
                          K_DICSEL = k_dicsel  )

  zeqn_hmax = EQUATION_AT(p_alktot, zh_max,    &
                          p_dicvar, p_bortot,  &
                          p_po4tot, p_siltot,  &
                          p_nh4tot, p_h2stot,  &
                          p_so4tot, p_flutot,  &
                          K_DICSEL = k_dicsel  )

  vhmin_atgen_ini(i_root) = zeqn_hmin
  vhmax_atgen_ini(i_root) = zeqn_hmax

#if defined(DEBUG_PHSOLVERS)
  PRINT*, '[SOLVE_AT_GENERAL2] zh_min / zeqn_hmin :', zh_min, zeqn_hmin
  PRINT*, '[SOLVE_AT_GENERAL2] zh_max / zeqn_hmax :', zh_max, zeqn_hmax
#endif

  zh = zh_ini(i_root)

  IF (ABS(zeqn_hmin) < ABS(zeqn_hmax)) THEN
    zh_absmin   = zh_min
    zeqn_absmin = zeqn_hmin
  ELSE
    zh_absmin   = zh_max
    zeqn_absmin = zeqn_hmax
  ENDIF
 

  zh_prev = zh

  zeqn = EQUATION_AT(p_alktot, zh,        &
                     p_dicvar, p_bortot,  &
                     p_po4tot, p_siltot,  &
                     p_nh4tot, p_h2stot,  &
                     p_so4tot, p_flutot,  &
                     K_DICSEL = k_dicsel, &
                     P_DERIVEQN = zdeqndh )

  nsucc_max = 0
  nsucc_min = 0

                                    ! The second root, if any, is on an
  IF (i_root == 2) THEN             ! increasing branch of the EQUATION_AT
                                    ! function. SOLVE_AT_GENERAL2 requires
    zeqn    = -zeqn                 ! that EQUATION_AT(..., zh_min, ...) > 0
    zdeqndh = -zdeqndh              ! and  EQUATION_AT(..., zh_max, ...) < 0.
                                    ! We therefore change the sign of the
  ENDIF                             ! function for the second root.


  DO

    IF (niter_atgen(i_root) > jp_maxniter_atgen) THEN
      zh = pp_hnan
      EXIT
    ENDIF


    IF (zeqn > 0._wp) THEN
      zh_min    = zh
      zeqn_hmin = zeqn
      nsucc_min = nsucc_min + 1
      nsucc_max = 0
#if defined(DEBUG_PHSOLVERS)
      PRINT*, '[SOLVE_AT_GENERAL2] increasing zh_min    :', zh_min, zeqn, nsucc_min
#endif
    ELSEIF (zeqn < 0._wp) THEN
      zh_max    = zh
      zeqn_hmax = zeqn
      nsucc_max = nsucc_max + 1
      nsucc_min = 0
#if defined(DEBUG_PHSOLVERS)
      PRINT*, '[SOLVE_AT_GENERAL2] decreasing zh_max    :', zh_max, zeqn, nsucc_max
#endif
    ELSE
      ! zh is the root; unlikely but, one never knows
      EXIT
    ENDIF


    ! Now determine the next iterate zh
    niter_atgen(i_root) = niter_atgen(i_root) + 1



    IF (ABS(zeqn) >= 0.5_wp*ABS(zeqn_absmin)) THEN

      ! If the function evaluation at the current point is not
      ! decreasing faster than expected with a bisection step
      ! (at least linearly) in absolute value take one regula falsi
      ! step, except if either the minimum or the maximum value has
      ! been modified more than three times (default - can be
      ! overridden by modifying the paraemeter value jp_nmaxsucc)
      ! in a row. This increases chances that the maximum, resp.
      ! minimum, is also revised from time to time.

      IF ((nsucc_min > jp_nmaxsucc) .OR. (nsucc_max > jp_nmaxsucc)) THEN

        ! Bisection step in pH-space:
        ! ph_new = (ph_min + ph_max)/2d0
        ! In terms of [H]_new:
        ! [H]_new = 10**(-ph_new)
        !         = 10**(-(ph_min + ph_max)/2d0)
        !         = SQRT(10**(-(ph_min + phmax)))
        !         = SQRT(zh_max * zh_min)
        zh = SQRT(zh_max * zh_min)
        nsucc_min = 0
        nsucc_max = 0

#if defined(DEBUG_PHSOLVERS)
        PRINT*, '[SOLVE_AT_GENERAL2] bisection nsucc > 3'
#endif

      ELSE

        ! Regula falsi step  on [H_min, H_max] (too expensive on [pH_min, pH_max])
        zrf_hmin = -zeqn_hmax/(zeqn_hmin - zeqn_hmax)
        zrf_hmax =  zeqn_hmin/(zeqn_hmin - zeqn_hmax)

        IF (zrf_hmin < pz_rf_thetamin) THEN
          zh = pz_rf_thetamin * zh_min + pz_rf_thetamax * zh_max
        ELSEIF (zrf_hmin >  pz_rf_thetamax) THEN
          zh = pz_rf_thetamax * zh_min + pz_rf_thetamin * zh_max
        ELSE
          zh = zrf_hmin*zh_min + zrf_hmax*zh_max
        ENDIF

#if defined(DEBUG_PHSOLVERS)
        PRINT*, '[SOLVE_AT_GENERAL2] regula falsi weights :', zrf_hmin, zrf_hmax
#endif

      ENDIF


#if defined(DEBUG_PHSOLVERS)
      zh_lnfactor = (zh - zh_prev)/zh_prev
      PRINT*, '[SOLVE_AT_GENERAL2] testing zh           :', zh, zh_lnfactor
#endif

      niter_atgen_limabs(i_root) = niter_atgen_limabs(i_root) + 1

    ELSE

      ! dzeqn/dpH = dzeqn/d[H] * d[H]/dpH
      !           = -zdeqndh * LOG(10) * [H]
      ! \Delta pH = -zeqn/(zdeqndh*d[H]/dpH) = zeqn/(zdeqndh*[H]*LOG(10))

      ! pH_new = pH_old + \deltapH

      ! [H]_new = 10**(-pH_new)
      !         = 10**(-pH_old - \Delta pH)
      !         = [H]_old * 10**(-zeqn/(zdeqndh*[H]_old*LOG(10)))
      !         = [H]_old * EXP(-LOG(10)*zeqn/(zdeqndh*[H]_old*LOG(10)))
      !         = [H]_old * EXP(-zeqn/(zdeqndh*[H]_old))

      zh_lnfactor = -zeqn/(zdeqndh*zh_prev)

      IF (ABS(zh_lnfactor) < pz_exp_threshold) THEN
        zh_delta    = zh_lnfactor*zh_prev
        zh          = zh_prev + zh_delta
      ELSEIF (ABS(zh_lnfactor) < pz_exp_upperlim) THEN
        zh          = zh_prev*EXP(zh_lnfactor)
      ELSE
        zh_lnfactor = SIGN(pz_exp_upperlim, zh_lnfactor)
        zh          = zh_prev*EXP(zh_lnfactor)
      ENDIF

#if defined(DEBUG_PHSOLVERS)
      PRINT*, '[SOLVE_AT_GENERAL2] testing Newton zh    :', zh, zh_lnfactor
#endif


      IF ( zh < zh_min ) THEN
        ! if [H]_new < [H]_min, then take
        ! one regula falsi step on [H_min, H_max]
        zrf_hmin = -zeqn_hmax/(zeqn_hmin - zeqn_hmax)
        zrf_hmax =  zeqn_hmin/(zeqn_hmin - zeqn_hmax)

        IF (zrf_hmin < pz_rf_thetamin) THEN
          zh = pz_rf_thetamin * zh_min + pz_rf_thetamax * zh_max
        ELSEIF (zrf_hmin >  pz_rf_thetamax) THEN
          zh = pz_rf_thetamax * zh_min + pz_rf_thetamin * zh_max
        ELSE
          zh = zrf_hmin*zh_min + zrf_hmax*zh_max
        ENDIF


#if defined(DEBUG_PHSOLVERS)
        zh_lnfactor       = (zh - zh_prev)/zh_prev
        PRINT*, '[SOLVE_AT_GENERAL2] increasing Newton zh :', zh, zh_lnfactor
#endif

        niter_atgen_limin(i_root) = niter_atgen_limin(i_root) + 1

      ENDIF

      IF ( zh > zh_max ) THEN
        ! if [H]_new < [H]_min, then take
        ! one regula falsi step on [H_min, H_max]
        zrf_hmin = -zeqn_hmax/(zeqn_hmin - zeqn_hmax)
        zrf_hmax =  zeqn_hmin/(zeqn_hmin - zeqn_hmax)

        IF (zrf_hmin < pz_rf_thetamin) THEN
          zh = pz_rf_thetamin * zh_min + pz_rf_thetamax * zh_max
        ELSEIF (zrf_hmin >  pz_rf_thetamax) THEN
          zh = pz_rf_thetamax * zh_min + pz_rf_thetamin * zh_max
        ELSE
          zh = zrf_hmin*zh_min + zrf_hmax*zh_max
        ENDIF

#if defined(DEBUG_PHSOLVERS)
        zh_lnfactor       = (zh - zh_prev)/zh_prev
        PRINT*, '[SOLVE_AT_GENERAL2] decreasing Newton zh:', zh, zh_lnfactor
#endif

        niter_atgen_limax(i_root) = niter_atgen_limax(i_root) + 1

      ENDIF


    ENDIF



    IF (ABS(zeqn_absmin) > ABS(zeqn)) THEN
#if defined(DEBUG_PHSOLVERS)
      PRINT*, '[SOLVE_AT_GENERAL2] adjusting absmin     :', zh_prev, zeqn
#endif
      zh_absmin   = zh_prev
      IF (i_root == 2) THEN
        zeqn_absmin = -zeqn
      ELSE
        zeqn_absmin =  zeqn
      ENDIF
    ENDIF


    zeqn = EQUATION_AT(p_alktot, zh,        &
                       p_dicvar, p_bortot,  &
                       p_po4tot, p_siltot,  &
                       p_nh4tot, p_h2stot,  &
                       p_so4tot, p_flutot,  &
                       K_DICSEL = k_dicsel, &
                       P_DERIVEQN = zdeqndh )

                                    ! Exit if the length of [H_min, H_max] is of
                                    ! the same order as the required precision
    IF ((zh_max - zh_min) < (0.5_wp*(zh_max + zh_min) * pp_rdel_ah_target)) THEN

                                    ! Check if the previously registered
                                    ! absolute minimum eqn value was lower than
                                    ! the current one - if so, return that one.
      IF (ABS(zeqn_absmin) < ABS(zeqn)) THEN
        zh   = zh_absmin
        zeqn = zeqn_absmin
      ENDIF

      EXIT                          ! Done!

    ENDIF

                                    ! Prepare the next iteration
    IF (i_root == 2) THEN 
      zeqn    = -zeqn               !  - revert equation signs if we
      zdeqndh = -zdeqndh            !    are searching for the second root
    ENDIF

    zh_prev = zh                    !  - save the previous iterate


  ENDDO


  SOLVE_AT_GENERAL2(i_root) = zh

#if defined(DEBUG_PHSOLVERS)
  IF (zh /= pp_hnan) THEN
    PRINT*, '[SOLVE_AT_GENERAL2] zh / zeqn :', zh, zeqn
  ELSE
    PRINT*, '[SOLVE_AT_GENERAL2] zh / zeqn :', zh, HUGE(1._wp)
  ENDIF
#endif

  IF (PRESENT(p_val)) THEN

    IF (zh /= pp_hnan) THEN 
      p_val(i_root) = zeqn
    ELSE
      p_val(i_root) = HUGE(1._wp)
    ENDIF

  ENDIF

ENDDO


                                    ! Finally, initialize the variables
                                    ! related to the non-used roots.
DO i_root = k_nroots+1, 2

  niter_atgen(i_root)               = 0
  niter_atgen_limin(i_root)         = 0
  niter_atgen_limax(i_root)         = 0
  niter_atgen_limabs(i_root)        = 0

  vhmin_atgen_ini(i_root)           = HUGE(1._wp)
  vhmax_atgen_ini(i_root)           = HUGE(1._wp)

  SOLVE_AT_GENERAL2(i_root)         = pp_hnan
  IF (PRESENT(p_val)) p_val(i_root) = HUGE(1._wp)

ENDDO

RETURN


!===============================================================================
 END FUNCTION SOLVE_AT_GENERAL2
!===============================================================================




!===============================================================================
 FUNCTION SOLVE_AT_GENERAL_SEC(p_alktot, p_dictot, p_bortot,                   &
                           p_po4tot, p_siltot, p_nh4tot, p_h2stot,             &
                           p_so4tot, p_flutot,                                 &
                           p_hini,   p_val)
!===============================================================================

! Wrapper for compatibility with SolveSAPHE v. 1.0.x and 1.1

IMPLICIT NONE

REAL(KIND=wp) :: SOLVE_AT_GENERAL_SEC


!--------------------!
! Argument variables !
!--------------------!

REAL(KIND=wp), INTENT(IN)            :: p_alktot
REAL(KIND=wp), INTENT(IN)            :: p_dictot
REAL(KIND=wp), INTENT(IN)            :: p_bortot
REAL(KIND=wp), INTENT(IN)            :: p_po4tot
REAL(KIND=wp), INTENT(IN)            :: p_siltot
REAL(KIND=wp), INTENT(IN)            :: p_nh4tot
REAL(KIND=wp), INTENT(IN)            :: p_h2stot
REAL(KIND=wp), INTENT(IN)            :: p_so4tot
REAL(KIND=wp), INTENT(IN)            :: p_flutot

REAL(KIND=wp), INTENT(IN),  OPTIONAL :: p_hini
REAL(KIND=wp), INTENT(OUT), OPTIONAL :: p_val


!-----------------!
! Local variables !
!-----------------!

REAL(KIND=wp), DIMENSION(2) ::  z_hini, z_val
REAL(KIND=wp), DIMENSION(2) ::  z_h

INTEGER ::  n_roots

IF (PRESENT(p_hini)) THEN

  z_hini(1) = p_hini
  z_hini(2) = pp_hnan

  IF (PRESENT(p_val)) THEN
    z_h(:) = SOLVE_AT_GENERAL2_SEC(p_alktot, p_dictot, p_bortot,               &
                               p_po4tot, p_siltot, p_nh4tot, p_h2stot,         &
                               p_so4tot, p_flutot,                             &
                               jp_dic,   n_roots,                              &
                               z_hini,   z_val)
    p_val = z_val(1)
  ELSE
    z_h(:) = SOLVE_AT_GENERAL2_SEC(p_alktot, p_dictot, p_bortot,               &
                               p_po4tot, p_siltot, p_nh4tot, p_h2stot,         &
                               p_so4tot, p_flutot,                             &
                               jp_dic,   n_roots,                              &
                               z_hini)
  ENDIF

ELSE

  IF (PRESENT(p_val)) THEN
    z_h(:) = SOLVE_AT_GENERAL2_SEC(p_alktot, p_dictot, p_bortot,               &
                               p_po4tot, p_siltot, p_nh4tot, p_h2stot,         &
                               p_so4tot, p_flutot,                             &
                               jp_dic,   n_roots,                              &
                               p_val = z_val)
    p_val = z_val(1)
  ELSE
    z_h(:) = SOLVE_AT_GENERAL2_SEC(p_alktot, p_dictot, p_bortot,               &
                               p_po4tot, p_siltot, p_nh4tot, p_h2stot,         &
                               p_so4tot, p_flutot,                             &
                               jp_dic,   n_roots)
  ENDIF
  
ENDIF


IF (n_roots == 1) THEN
  SOLVE_AT_GENERAL_SEC = z_h(1)
ELSE
  SOLVE_AT_GENERAL_SEC = pp_hnan
ENDIF

RETURN


!===============================================================================
 END FUNCTION SOLVE_AT_GENERAL_SEC
!===============================================================================





!===============================================================================
 FUNCTION SOLVE_AT_GENERAL2_SEC(p_alktot, p_dicvar, p_bortot,                  &
                                p_po4tot, p_siltot,                            &
                                p_nh4tot, p_h2stot,                            &
                                p_so4tot, p_flutot,                            &
                                k_dicsel, k_nroots,                            &
                                p_hini,   p_val                                )
!===============================================================================

! Universal pH solver that converges from any given initial value.
! Determines upper an lower bounds for the solution if no initial values are
! provided.

USE MOD_CHEMCONST, ONLY: api1_wat, aphscale


IMPLICIT NONE


REAL(KIND=wp), DIMENSION(2) :: SOLVE_AT_GENERAL2_SEC

                                        ! Lowest limit for a Regula Falsi iterate
                                        ! zh relative to zh_min, as a fraction
                                        ! of the length of the [zh_min, zh_max] interval:
                                        ! zh_min + pz_rf_thetahmin * (zh_max - zh_min)
                                        !    <= zh <= zh_max - pz_rf_thetahmin * (zh_max - zh_min)
REAL(KIND=wp), PARAMETER :: pz_rf_thetamin   = 0.10_wp
REAL(KIND=wp), PARAMETER :: pz_rf_thetamax   = 1.00_wp - pz_rf_thetamin

                                        ! Maximum number of successive H_min/H_max changes
INTEGER,       PARAMETER :: jp_nmaxsucc      = 3


!--------------------!
! Argument variables !
!--------------------!

REAL(KIND=wp),               INTENT(IN)            :: p_alktot
REAL(KIND=wp),               INTENT(IN)            :: p_dicvar
REAL(KIND=wp),               INTENT(IN)            :: p_bortot
REAL(KIND=wp),               INTENT(IN)            :: p_po4tot
REAL(KIND=wp),               INTENT(IN)            :: p_siltot
REAL(KIND=wp),               INTENT(IN)            :: p_nh4tot
REAL(KIND=wp),               INTENT(IN)            :: p_h2stot
REAL(KIND=wp),               INTENT(IN)            :: p_so4tot
REAL(KIND=wp),               INTENT(IN)            :: p_flutot

INTEGER,                     INTENT(IN)            :: k_dicsel
INTEGER,                     INTENT(OUT)           :: k_nroots

REAL(KIND=wp), DIMENSION(2), INTENT(IN),  OPTIONAL :: p_hini
REAL(KIND=wp), DIMENSION(2), INTENT(OUT), OPTIONAL :: p_val


!-----------------!
! Local variables !
!-----------------!

REAL(KIND=wp), DIMENSION(2) ::  zh_inf, zh_sup, zh_ini
REAL(KIND=wp) :: zh, zh_prev, zh_1, zh_2, zh_delta
REAL(KIND=wp) :: zh_min,    zh_max,    zh_absmin
REAL(KIND=wp) :: zeqn_hmin, zeqn_hmax, zeqn_absmin
REAL(KIND=wp) :: zeqn, zeqn_1, zeqn_2, zrf_hmin, zrf_hmax
REAL(KIND=wp) :: zdelta

INTEGER       :: i_root

INTEGER       :: nsucc_min, nsucc_max

LOGICAL       :: l_exitnow


!-------------------------------------------------------------------------------


IF (PRESENT(p_hini)) THEN
   zh_ini = p_hini
ELSE
   zh_ini = pp_hnan
ENDIF

CALL HINFSUPINI(p_alktot, p_dicvar, p_bortot,       &
                          p_po4tot, p_siltot,       &
                          p_nh4tot, p_h2stot,       &
                          p_so4tot, p_flutot,       &
                          k_dicsel, k_nroots,       &
                          zh_inf,   zh_sup,   zh_ini)

ahmin_atsec_ini(:) = zh_inf
ahmax_atsec_ini(:) = zh_sup
ah_atsec_ini(:)    = zh_ini

#if defined(DEBUG_PHSOLVERS)
   PRINT*, '[SOLVE_AT_GENERAL2_SEC] n_roots  :', k_nroots
   PRINT*, '[SOLVE_AT_GENERAL2_SEC] h_min(:) :', zh_inf
   PRINT*, '[SOLVE_AT_GENERAL2_SEC] h_sup(:) :', zh_sup
   PRINT*, '[SOLVE_AT_GENERAL2_SEC] h_ini(:) :', zh_ini
#endif


DO i_root = 1, k_nroots

                                    
  niter_atsec(i_root)        = 0    ! Reset counters of iterations
  niter_atsec_limin(i_root)  = 0
  niter_atsec_limax(i_root)  = 0
  niter_atsec_limabs(i_root) = 0

  zh_min = zh_inf(i_root)
  zh_max = zh_sup(i_root)

  zeqn_hmin = EQUATION_AT(p_alktot, zh_min,    &
                          p_dicvar, p_bortot,  &
                          p_po4tot, p_siltot,  &
                          p_nh4tot, p_h2stot,  &
                          p_so4tot, p_flutot,  &
                          K_DICSEL = k_dicsel  )

  zeqn_hmax = EQUATION_AT(p_alktot, zh_max,    &
                          p_dicvar, p_bortot,  &
                          p_po4tot, p_siltot,  &
                          p_nh4tot, p_h2stot,  &
                          p_so4tot, p_flutot,  &
                          K_DICSEL = k_dicsel  )

  vhmin_atsec_ini(i_root) = zeqn_hmin
  vhmax_atsec_ini(i_root) = zeqn_hmax

#if defined(DEBUG_PHSOLVERS)
  PRINT*, '[SOLVE_AT_GENERAL2_SEC] zh_min / zeqn_hmin   :', zh_min, zeqn_hmin
  PRINT*, '[SOLVE_AT_GENERAL2_SEC] zh_max / zeqn_hmax   :', zh_max, zeqn_hmax
#endif

  zh = zh_ini(i_root)

  IF (ABS(zeqn_hmin) < ABS(zeqn_hmax)) THEN
    zh_absmin   = zh_min
    zeqn_absmin = zeqn_hmin
  ELSE
    zh_absmin   = zh_max
    zeqn_absmin = zeqn_hmax
  ENDIF


  ! Prepare the secant iterations: two initial (zh, zeqn) pairs are required
  ! We have the starting value, that needs to be completed by the evaluation
  ! of the equation value it produces.

  ! Complete the initial value with its equation evaluation
  ! (will take the role of the $n-2$ iterate at the first secant evaluation)


  zh_2   = zh                       ! zh_2 is the initial value;

  zeqn_2 = EQUATION_AT(p_alktot, zh_2,      &
                       p_dicvar, p_bortot,  &
                       p_po4tot, p_siltot,  &
                       p_nh4tot, p_h2stot,  &
                       p_so4tot, p_flutot,  &
                       K_DICSEL = k_dicsel  )

  IF (ABS(zeqn_absmin) > ABS(zeqn_2)) THEN
#if defined(DEBUG_PHSOLVERS)
    PRINT*, '[SOLVE_AT_GENERAL2_SEC] adjusting absmin     :', zh_2, zeqn_2
#endif
    zh_absmin   = zh_2
    zeqn_absmin = zeqn_2
  ENDIF

  IF (i_root == 2) zeqn_2 = -zeqn_2 ! The second root, if any, is on an
                                    ! increasing branch of the EQUATION_AT
                                    ! function. SOLVE_AT_GENERAL2_SEC requires
                                    ! that EQUATION_AT(..., zh_min, ...) > 0
                                    ! and  EQUATION_AT(..., zh_max, ...) < 0.
                                    ! We therefore change the sign of the
                                    ! function for the second root.


  ! Adapt bracketing interval and heuristically set zh_1

  IF (zeqn_2 < 0._wp) THEN
                                    ! If zeqn_2 < 0, then we adjust zh_max:
                                    ! we can be sure that zh_min < zh_2 < zh_max.
    zh_max    = zh_2
    zeqn_hmax = zeqn_2
                                    ! for zh_1, try 25% (0.1 pH units) below the current zh_max,
                                    ! but stay above SQRT(zh_min*zh_max), which would be equivalent
                                    ! to a bisection step on [pH@zh_min, pH@zh_max]
    zh_1   = MAX(zh_max/1.25_wp, SQRT(zh_min*zh_max))

  ELSEIF (zeqn_2 > 0._wp) THEN
                                    ! If zeqn_2 < 0, then we adjust zh_min:
                                    ! we can be sure that zh_min < zh_2 < zh_max.
    zh_min    = zh_2
    zeqn_hmin = zeqn_2
                                    ! for zh_1, try 25% (0.1 pH units) above the current zh_min,
                                    ! but stay below SQRT(zh_min*zh_max) which would be equivalent
                                    ! to a bisection step on [pH@zh_min, pH@zh_max]
    zh_1   = MIN(zh_min*1.25_wp, SQRT(zh_min*zh_max))
   
  ELSE                              ! we have got the root; unlikely, but one never knows

    zh = zh_2

    IF (i_root == 2) THEN
      zeqn = -zeqn_2
    ELSE
      zeqn =  zeqn_2
    ENDIF

    EXIT

  ENDIF

                                    ! We now have the first pair completed (zh_2, zeqn_2).
                                    ! Define the second one (zh_1, zeqn_1), which is also
                                    ! the first iterate. zh_1 has already been set above.

  niter_atsec(i_root) = 1           ! Update counter of iterations


  zeqn_1 = EQUATION_AT(p_alktot, zh_1,      &
                       p_dicvar, p_bortot,  &
                       p_po4tot, p_siltot,  &
                       p_nh4tot, p_h2stot,  &
                       p_so4tot, p_flutot,  &
                       K_DICSEL = k_dicsel  )

  IF (i_root == 2) zeqn_1 = -zeqn_1 ! Second root: change sign of EQUATION_AT


                                    ! Adapt bracketing interval: we know
                                    ! that zh_1 <= zh <= zh_max (if zeqn_1 > 0)
                                    ! or zh_min <= zh <= zh_1 (if zeqn_1 < 0),
                                    ! so this can always be done.
  IF (zeqn_1 > 0._wp) THEN

    zh_min    = zh_1
    zeqn_hmin = zeqn_1
#if defined(DEBUG_PHSOLVERS)
    PRINT*, '[SOLVE_AT_GENERAL2_SEC] increasing zh_min    :', zh_min, zeqn_hmin
#endif

  ELSEIF (zeqn_1 < 0._wp) THEN

    zh_max    = zh_1
    zeqn_hmax = zeqn_1
#if defined(DEBUG_PHSOLVERS)
    PRINT*, '[SOLVE_AT_GENERAL2_SEC] decreasing zh_max    :', zh_max, zeqn_hmax
#endif

  ELSE                              ! zh_1 is the root

    zh = zh_1
    IF (i_root == 2) THEN
      zeqn = -zeqn_1
    ELSE
      zeqn =  zeqn_1
    ENDIF

    EXIT

  ENDIF


  IF  (ABS(zeqn_1) > ABS(zeqn_absmin)) THEN
                                    ! Swap zh_2 and zh_1 if ABS(zeqn_2) < ABS(zeqn_1)
                                    ! so that zh_2 and zh_1 lead to decreasing equation
                                    ! values (in absolute value)
    zh     = zh_1
    zeqn   = zeqn_1
    zh_1   = zh_2
    zeqn_1 = zeqn_2
    zh_2   = zh
    zeqn_2 = zeqn

  ELSE

    zh_absmin = zh_1
    IF (i_root == 2) THEN
      zeqn_absmin = -zeqn_1
    ELSE
      zeqn_absmin =  zeqn_1
    ENDIF

  ENDIF


                                    ! Pre-calculate the first secant iterate
                                    ! (this is the second iterate)

  niter_atsec(i_root) = 2

  zh_delta = -zeqn_1/((zeqn_2-zeqn_1)/(zh_2 - zh_1))

  zh = zh_1 + zh_delta


                                    ! Make sure that zh_min < zh < zh_max (if not,
                                    ! bisect around zh_1 which is the best estimate)

  IF (zh > zh_max) THEN             ! this can only happen if zh_2 < zh_1
                                    ! and zeqn_2 > zeqn_1 > 0
    zh = SQRT(zh_1*zh_max)

  ENDIF

  IF (zh < zh_min) THEN             ! this can only happen if zh_2 > zh_1
                                    ! and zeqn_2 < zeqn_1 < 0
    zh = SQRT(zh_1*zh_min)

  ENDIF


  zh_prev = zh

  zeqn = EQUATION_AT(p_alktot,   zh,        &
                       p_dicvar, p_bortot,  &
                       p_po4tot, p_siltot,  &
                       p_nh4tot, p_h2stot,  &
                       p_so4tot, p_flutot,  &
                       K_DICSEL = k_dicsel  )

  nsucc_max = 0
  nsucc_min = 0

  IF (i_root == 2) zeqn = -zeqn     ! Second root: change sign of EQUATION_AT


  DO

    IF (niter_atsec(i_root) > jp_maxniter_atsec) THEN

      zh = pp_hnan
      EXIT

    ENDIF


    ! Adapt bracketing interval: since initially, zh_min <= zh <= zh_max
    ! we are sure that zh will improve either bracket, depending on the sign
    ! of zeqn
    IF (zeqn > 0._wp) THEN
      zh_min    = zh
      zeqn_hmin = zeqn
      nsucc_min = nsucc_min + 1
      nsucc_max = 0
#if defined(DEBUG_PHSOLVERS)
      PRINT*, '[SOLVE_AT_GENERAL2_SEC] increasing zh_min    :', zh_min, zeqn
#endif
    ELSEIF (zeqn < 0._wp) THEN
      zh_max    = zh
      zeqn_hmax = zeqn
      nsucc_max = nsucc_max + 1
      nsucc_min = 0
#if defined(DEBUG_PHSOLVERS)
      PRINT*, '[SOLVE_AT_GENERAL2_SEC] increasing zh_max    :', zh_max, zeqn
#endif
    ELSE
      ! zh is the root
      EXIT
    ENDIF


    ! start calculation of next iterate

    niter_atsec(i_root) = niter_atsec(i_root) + 1

    zh_2   = zh_1
    zeqn_2 = zeqn_1
    zh_1   = zh
    zeqn_1 = zeqn


    IF (ABS(zeqn) >= 0.5_wp*ABS(zeqn_absmin)) THEN

      ! if the function evaluation at the current point
      ! is not decreasing faster in absolute value than
      ! we may expect for a bisection step, then take
      ! one bisection step on [ph_min, ph_max]
      ! ph_new = (ph_min + ph_max)/2d0
      ! In terms of [H]_new:
      ! [H]_new = 10**(-ph_new)
      !         = 10**(-(ph_min + ph_max)/2d0)
      !         = SQRT(10**(-(ph_min + phmax)))
      !         = SQRT(zh_max * zh_min)
     
      IF ((nsucc_min > jp_nmaxsucc) .OR. (nsucc_max > jp_nmaxsucc)) THEN

        zh = SQRT(zh_max * zh_min)
        nsucc_min = 0
        nsucc_max = 0

#if defined(DEBUG_PHSOLVERS)
        PRINT*, '[SOLVE_AT_GENERAL2_SEC] bisection nsucc > 3'
#endif

      ELSE

        zrf_hmin = -zeqn_hmax/(zeqn_hmin - zeqn_hmax)
        zrf_hmax =  zeqn_hmin/(zeqn_hmin - zeqn_hmax)

        IF (zrf_hmin < pz_rf_thetamin) THEN
          zh = pz_rf_thetamin * zh_min + pz_rf_thetamax * zh_max
        ELSEIF (zrf_hmin >  pz_rf_thetamax) THEN
          zh = pz_rf_thetamax * zh_min + pz_rf_thetamin * zh_max
        ELSE
          zh = zrf_hmin*zh_min + zrf_hmax*zh_max
        ENDIF

#if defined(DEBUG_PHSOLVERS)
        PRINT*, '[SOLVE_AT_GENERAL2_SEC] regula falsi weights :', zrf_hmin, zrf_hmax
#endif

      ENDIF


      zh_delta = zh - zh_1


#if defined(DEBUG_PHSOLVERS)
      PRINT*, '[SOLVE_AT_GENERAL2_SEC] testing zh           :', zh, zeqn, zh_delta
#endif

      niter_atsec_limabs(i_root) = niter_atsec_limabs(i_root) + 1

    ELSE

      ! \Delta H = -zeqn_1*(h_2 - h_1)/(zeqn_2 - zeqn_1) 
      ! H_new = H_1 + \Delta H

      zh_delta = -zeqn_1/((zeqn_2-zeqn_1)/(zh_2 - zh_1))

      zh       = zh_1 + zh_delta

#if defined(DEBUG_PHSOLVERS)
      PRINT*, '[SOLVE_AT_GENERAL2_SEC] testing zh           :', zh, zeqn, zh_delta
#endif


      IF ( zh < zh_min ) THEN
        ! if [H]_new < [H]_min, i.e., if ph_new > ph_max then
        ! take one bisection step on [ph_min, ph_max]
        ! ph_new = (ph_prev + ph_max)/2d0

        zrf_hmin = -zeqn_hmax/(zeqn_hmin - zeqn_hmax)
        zrf_hmax =  zeqn_hmin/(zeqn_hmin - zeqn_hmax)
  
        IF (zrf_hmin < pz_rf_thetamin) THEN
          zh = pz_rf_thetamin * zh_min + pz_rf_thetamax * zh_max
        ELSEIF (zrf_hmin >  pz_rf_thetamax) THEN
          zh = pz_rf_thetamax * zh_min + pz_rf_thetamin * zh_max
        ELSE
          zh = zrf_hmin*zh_min + zrf_hmax*zh_max
        ENDIF

#if defined(DEBUG_PHSOLVERS)
        PRINT*, '[SOLVE_AT_GENERAL2] regula falsi weights     :', zrf_hmin, zrf_hmax
#endif

        zh_delta  = zh - zh_1

#if defined(DEBUG_PHSOLVERS)
        PRINT*, '[SOLVE_AT_GENERAL2_SEC] increasing sec zh    :', zh, zeqn, zh_delta
#endif

        niter_atsec_limin(i_root) = niter_atsec_limin(i_root) + 1

      ENDIF

      IF ( zh > zh_max ) THEN
        ! if [H]_new > [H]_max, i.e., if ph_new < ph_min, then
        ! take one bisection step on [ph_min, ph_max]

        zrf_hmin = -zeqn_hmax/(zeqn_hmin - zeqn_hmax)
        zrf_hmax =  zeqn_hmin/(zeqn_hmin - zeqn_hmax)
  
        IF (zrf_hmin < pz_rf_thetamin) THEN
          zh = pz_rf_thetamin * zh_min + pz_rf_thetamax * zh_max
        ELSEIF (zrf_hmin >  pz_rf_thetamax) THEN
          zh = pz_rf_thetamax * zh_min + pz_rf_thetamin * zh_max
        ELSE
          zh = zrf_hmin*zh_min + zrf_hmax*zh_max
        ENDIF

#if defined(DEBUG_PHSOLVERS)
        PRINT*, '[SOLVE_AT_GENERAL2] regula falsi weights     :', zrf_hmin, zrf_hmax
#endif

        zh_delta  = zh - zh_1

#if defined(DEBUG_PHSOLVERS)
        PRINT*, '[SOLVE_AT_GENERAL2_SEC] decreasing sec zh    :', zh, zeqn, zh_delta
#endif

        niter_atsec_limax(i_root) = niter_atsec_limax(i_root) + 1

      ENDIF

    ENDIF


    IF (ABS(zeqn_absmin) > ABS(zeqn)) THEN

      zh_absmin   = zh_prev
      IF (i_root == 2) THEN
        zeqn_absmin = -zeqn
      ELSE
        zeqn_absmin =  zeqn
      ENDIF

#if defined(DEBUG_PHSOLVERS)
      PRINT*, '[SOLVE_AT_GENERAL2_SEC] adjusting absmin     :', zh_prev, zeqn
#endif

    ENDIF


    zeqn = EQUATION_AT(p_alktot, zh,        &
                       p_dicvar, p_bortot,  &
                       p_po4tot, p_siltot,  &
                       p_nh4tot, p_h2stot,  &
                       p_so4tot, p_flutot,  &
                       K_DICSEL = k_dicsel  )


                                    ! Exit if the length of [H_min, H_max] is of
                                    ! the same order as the required precision
    IF ((zh_max - zh_min) < (0.5_wp*(zh_max + zh_min) * pp_rdel_ah_target)) THEN

                                    ! Check if the previously registered
                                    ! absolute minimum eqn value was lower than
                                    ! the current one - if so, return that one.
      IF (ABS(zeqn_absmin) < ABS(zeqn)) THEN
        zh   = zh_absmin
        zeqn = zeqn_absmin
      ENDIF

      EXIT

    ENDIF

                                    ! Prepare the next iteration
    IF (i_root == 2) THEN 
      zeqn    = -zeqn               !  - revert equation signs if we
                                    !    are searching for the second root
    ENDIF

    zh_prev = zh                    !  - save the previous iterate


  ENDDO


  SOLVE_AT_GENERAL2_SEC(i_root) = zh

#if defined(DEBUG_PHSOLVERS)
  IF (zh /= pp_hnan) THEN
    PRINT*, '[SOLVE_AT_GENERAL2_SEC] zh / zeqn :', zh, zeqn
  ELSE
    PRINT*, '[SOLVE_AT_GENERAL2_SEC] zh / zeqn :', zh, HUGE(1._wp)
  ENDIF
#endif

  IF (PRESENT(p_val)) THEN

    IF(zh /= pp_hnan) THEN
      p_val(i_root) = zeqn
    ELSE
      p_val(i_root) = HUGE(1._wp)
    ENDIF

  ENDIF


ENDDO

                                    ! Finally, initialize the variables
                                    ! related to the non-used roots.
DO i_root = k_nroots+1, 2

  niter_atsec(i_root)               = 0
  niter_atsec_limin(i_root)         = 0
  niter_atsec_limax(i_root)         = 0
  niter_atsec_limabs(i_root)        = 0

  vhmin_atsec_ini(i_root)           = HUGE(1._wp)
  vhmax_atsec_ini(i_root)           = HUGE(1._wp)

  SOLVE_AT_GENERAL2_SEC(i_root)     = pp_hnan
  IF (PRESENT(p_val)) p_val(i_root) = HUGE(1._wp)

ENDDO




RETURN

!===============================================================================
 END FUNCTION SOLVE_AT_GENERAL2_SEC
!===============================================================================



END MODULE MOD_PHSOLVERS_LOGGING
