!
!    Copyright  2013, 2014, 2020, 2021 Guy Munhoven
!
!    This file is part of SolveSAPHE v. 2

!    SolveSAPHE is free software: you can redistribute it and/or modify
!    it under the terms of the GNU Lesser General Public License as published by
!    the Free Software Foundation, either version 3 of the License, or
!    (at your option) any later version.
!
!    SolveSAPHE is distributed in the hope that it will be useful,
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU Lesser General Public License for more details.
!
!    You should have received a copy of the GNU Lesser General Public License
!    along with SolveSAPHE.  If not, see <http://www.gnu.org/licenses/>.
!





! **********************
! Precompiler directives
! **********************

! CREATEFILES:
! -- if defined, result files are written
! -- if not defined, result files are *not* written

#define CREATEFILES


! CASE_TSP = 1: Surface cold
!            2: Surface warm
!            3: Deep cold
!            4: Surface cold, brackish

#define CASE_TSP 1


! CASE_SW  = 1: SW1 -- common present-day seawater
!            2: SW2 -- common present-day and future seawater
!            3: SW3 -- extreme seawater samples
!            4: SW4 -- brackish water and dilute seawater
!            5: SW5 -- slightly less extreme seawater

#define CASE_SW 2


! **************************
! End precompiler directives
! **************************

!==============================================================================================
PROGRAM DRIVER_AT_CARBONATE
!==============================================================================================

USE MOD_PRECISION
USE MOD_CHEMCONST
USE MOD_PHSOLVERS
USE MOD_CHEMSPECIATION


IMPLICIT NONE

REAL(KIND=wp)                       :: z_alktot, z_dictot, z_bortot
REAL(KIND=wp)                       :: z_po4tot, z_siltot, z_nh4tot, z_h2stot
REAL(KIND=wp)                       :: z_so4tot, z_flutot
REAL(KIND=wp)                       :: z_h, z_val
REAL(KIND=wp)                       :: z_co2, z_hco3, z_co3
REAL(KIND=wp)                       :: z_alkc

REAL                                :: z_cputime_start, z_cputime_end

INTEGER :: ji, ji_alk, ji_dic, ji_ph

INTEGER, PARAMETER :: jp_resunit = 1
REAL(KIND=wp), PARAMETER            :: z_zero    = 0._wp


! Temperature, salinity, pressure data
! ====================================

#if CASE_TSP == 1
REAL(KIND=wp), PARAMETER            :: z_t_k    = 275.15_wp
REAL(KIND=wp), PARAMETER            :: z_s      =  35.00_wp
REAL(KIND=wp), PARAMETER            :: z_p_bar  =   0.00_wp
 CHARACTER(LEN=*), PARAMETER        :: cp_tspid = 'sc'
#endif

#if CASE_TSP == 2
REAL(KIND=wp), PARAMETER            :: z_t_k    = 298.15_wp
REAL(KIND=wp), PARAMETER            :: z_s      =  35.00_wp
REAL(KIND=wp), PARAMETER            :: z_p_bar  =   0.00_wp
 CHARACTER(LEN=*), PARAMETER        :: cp_tspid = 'sw'
#endif

#if CASE_TSP == 3
REAL(KIND=wp), PARAMETER            :: z_t_k    = 275.15_wp
REAL(KIND=wp), PARAMETER            :: z_s      =  35.00_wp
REAL(KIND=wp), PARAMETER            :: z_p_bar  = 300.00_wp
 CHARACTER(LEN=*), PARAMETER        :: cp_tspid = 'dc'
#endif

#if CASE_TSP == 4
REAL(KIND=wp), PARAMETER            :: z_t_k   = 275.15_wp
REAL(KIND=wp), PARAMETER            :: z_s     =   3.50_wp
REAL(KIND=wp), PARAMETER            :: z_p_bar =   0.00_wp
 CHARACTER(LEN=*), PARAMETER        :: cp_tspid = 'sb'
#endif


! DIC and Alkalinity distributions (SWx cases)
! ============================================

#if CASE_SW == 1
! Settings for common present-day seawater samples
REAL(KIND=wp), PARAMETER            :: z_dictot_min =  1.85E-3_wp
REAL(KIND=wp), PARAMETER            :: z_dictot_max =  2.45E-3_wp
REAL(KIND=wp), PARAMETER            :: z_alktot_min =  2.2E-3_wp
REAL(KIND=wp), PARAMETER            :: z_alktot_max =  2.5E-3_wp
REAL(KIND=wp), PARAMETER            :: z_ph_min     =  6.0E+0_wp
REAL(KIND=wp), PARAMETER            :: z_ph_max     = 10.0E+0_wp
INTEGER, PARAMETER                  :: jp_ndic = 600
INTEGER, PARAMETER                  :: jp_nalk = 300
INTEGER, PARAMETER                  :: jp_nph  = 400
 CHARACTER(LEN=*), PARAMETER        :: cp_fileid = 'sw1-' // cp_tspid
#endif

#if CASE_SW == 2
! Settings for common present-day and future seawater
! (derived from SP750 simulation experiment with MBM-Medusa)
REAL(KIND=wp), PARAMETER            :: z_dictot_min =  1.85E-3_wp
REAL(KIND=wp), PARAMETER            :: z_dictot_max =  3.35E-3_wp
REAL(KIND=wp), PARAMETER            :: z_alktot_min =  2.2E-3_wp
REAL(KIND=wp), PARAMETER            :: z_alktot_max =  3.5E-3_wp
REAL(KIND=wp), PARAMETER            :: z_ph_min     =  6.0E+0_wp
REAL(KIND=wp), PARAMETER            :: z_ph_max     = 10.0E+0_wp
INTEGER, PARAMETER                  :: jp_ndic = 1500
INTEGER, PARAMETER                  :: jp_nalk = 1300
INTEGER, PARAMETER                  :: jp_nph  = 400
 CHARACTER(LEN=*), PARAMETER        :: cp_fileid = 'sw2-' // cp_tspid
#endif

#if CASE_SW == 3
! Settings for extreme seawater samples
REAL(KIND=wp), PARAMETER            :: z_dictot_min =  0.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_dictot_max =  6.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_alktot_min = -1.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_alktot_max =  5.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_ph_min     =  3.0E+0_wp
REAL(KIND=wp), PARAMETER            :: z_ph_max     = 12.0E+0_wp
INTEGER, PARAMETER                  :: jp_ndic = 600
INTEGER, PARAMETER                  :: jp_nalk = 600
INTEGER, PARAMETER                  :: jp_nph  = 900
 CHARACTER(LEN=*), PARAMETER        :: cp_fileid = 'sw3-' // cp_tspid
#endif

#if CASE_SW == 4
! Settings for dilute seawater samples
REAL(KIND=wp), PARAMETER            :: z_dictot_min =  0.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_dictot_max =  1.2E-3_wp
REAL(KIND=wp), PARAMETER            :: z_alktot_min =  0.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_alktot_max =  1.5E-3_wp
REAL(KIND=wp), PARAMETER            :: z_ph_min     =  3.0E+0_wp
REAL(KIND=wp), PARAMETER            :: z_ph_max     = 12.0E+0_wp
INTEGER, PARAMETER                  :: jp_ndic = 120
INTEGER, PARAMETER                  :: jp_nalk = 150
INTEGER, PARAMETER                  :: jp_nph  = 900
 CHARACTER(LEN=*), PARAMETER        :: cp_fileid = 'sw4-' // cp_tspid
#endif

#if CASE_SW == 5
! Settings for extreme seawater samples
REAL(KIND=wp), PARAMETER            :: z_dictot_min =  0.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_dictot_max =  4.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_alktot_min = -1.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_alktot_max =  3.0E-3_wp
REAL(KIND=wp), PARAMETER            :: z_ph_min     =  3.0E+0_wp
REAL(KIND=wp), PARAMETER            :: z_ph_max     = 12.0E+0_wp
INTEGER, PARAMETER                  :: jp_ndic = 400
INTEGER, PARAMETER                  :: jp_nalk = 400
INTEGER, PARAMETER                  :: jp_nph  = 900
 CHARACTER(LEN=*), PARAMETER        :: cp_fileid = 'sw5-' // cp_tspid
#endif


! Work variables and arrays
! =========================

REAL(KIND=wp), DIMENSION(jp_ndic)         :: z_dictot_arr
REAL(KIND=wp), DIMENSION(jp_nalk)         :: z_alktot_arr
REAL(KIND=wp), DIMENSION(jp_nph)          :: z_ph_arr
REAL(KIND=wp)                             :: z_dictot_del
REAL(KIND=wp)                             :: z_alktot_del
REAL(KIND=wp)                             :: z_ph_del


REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk) :: z_h_atgen
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk) :: z_val_atgen

REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk) :: z_co2_atgen
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk) :: z_hco3_atgen
REAL(KIND=wp), DIMENSION(jp_ndic,jp_nalk) :: z_co3_atgen

REAL(KIND=wp), DIMENSION(jp_nph, jp_nalk) :: z_co2_ph_at
REAL(KIND=wp), DIMENSION(jp_nph, jp_nalk) :: z_hco3_ph_at
REAL(KIND=wp), DIMENSION(jp_nph, jp_nalk) :: z_co3_ph_at


INTEGER, DIMENSION(jp_ndic,jp_nalk)       :: jndivg_atgen
INTEGER, DIMENSION(jp_ndic,jp_nalk)       :: jniter_div_atgen





 PRINT*
 PRINT*, 'T(K)          : ', z_t_k
 PRINT*, 'Salinity      : ', z_s
 PRINT*, 'Pressure(bar) : ', z_p_bar


z_bortot =  A_BTOT_SALIN(z_s)
z_po4tot =  0.5E-6_wp
z_siltot =  5.E-6_wp
z_nh4tot =  0.E-3_wp
z_h2stot =  0.E-3_wp
z_so4tot =  A_SO4TOT_SALIN(z_s)
z_flutot =  A_FTOT_SALIN(z_s)


 PRINT*
 PRINT*, 'SigB          : ', z_bortot
 PRINT*, 'SigPO4        : ', z_po4tot
 PRINT*, 'SigSil        : ', z_siltot
 PRINT*, 'SigAmmonium   : ', z_nh4tot
 PRINT*, 'SigSulfides   : ', z_h2stot
 PRINT*, 'SigSulfates   : ', z_so4tot
 PRINT*, 'SigF          : ', z_flutot


 CALL SETUP_API4PHSWS(z_t_k, z_s, z_p_bar)

 PRINT*
 PRINT*, 'Pi_1 DIC : ', api1_dic
 PRINT*, 'Pi_2 DIC : ', api2_dic
 PRINT*, 'Pi_1 BT  : ', api1_bor
 PRINT*, 'Pi_1 PO4 : ', api1_po4
 PRINT*, 'Pi_2 PO4 : ', api2_po4
 PRINT*, 'Pi_3 PO4 : ', api3_po4
 PRINT*, 'Pi_1 Sil : ', api1_sil
 PRINT*, 'Pi_1 NH4 : ', api1_nh4
 PRINT*, 'Pi_1 H2S : ', api1_h2s
 PRINT*, 'Pi_1 HSO4: ', api1_so4
 PRINT*, 'Pi_1 HF  : ', api1_flu
 PRINT*, 'Pi_1 H2O : ', api1_wat
 PRINT*, 'pHscale  : ', aphscale

 z_dictot_del = (z_dictot_max-z_dictot_min)/REAL(jp_ndic,KIND=wp)
 z_alktot_del = (z_alktot_max-z_alktot_min)/REAL(jp_nalk,KIND=wp)
 z_ph_del     = (z_ph_max    -z_ph_min    )/REAL(jp_nph,KIND=wp)

 PRINT*
 PRINT*, 'Running variant "'//cp_fileid//'"'

 PRINT*
 PRINT*, 'DIC Interval  : ', z_dictot_min, z_dictot_max
 PRINT*, 'ALK Interval  : ', z_alktot_min, z_alktot_max
 PRINT*, 'pH Interval   : ', z_ph_min,     z_ph_max
 PRINT*, 'DIC Step      : ', z_dictot_del
 PRINT*, 'ALK Step      : ', z_alktot_del
 PRINT*, 'pH  Step      : ', z_ph_del

 DO ji_dic = 1, jp_ndic
  z_dictot_arr(ji_dic) =  z_dictot_min+(REAL(ji_dic,KIND=wp)-0.5_wp) * z_dictot_del
 ENDDO

 DO ji_alk = 1, jp_nalk
  z_alktot_arr(ji_alk) =  z_alktot_min+(REAL(ji_alk,KIND=wp)-0.5_wp) * z_alktot_del
 ENDDO

 DO ji_ph = 1, jp_nph
  z_ph_arr(ji_ph) =  z_ph_min+(REAL(ji_ph,KIND=wp)-0.5_wp) * z_ph_del
 ENDDO

 PRINT*, 'DIC First/Last: ', z_dictot_arr(1), z_dictot_arr(jp_ndic)
 PRINT*, 'ALK First/Last: ', z_alktot_arr(1), z_alktot_arr(jp_nalk)
 PRINT*, 'pH  First/Last: ', z_ph_arr(1),     z_ph_arr(jp_nph)



 PRINT*
 PRINT*, 'Calling SOLVE_AT_GENERAL'
 PRINT*, '------------------------'


 jndivg_atgen(:,:) = 0

 CALL CPU_TIME(z_cputime_start)


DO ji_alk = 1, jp_nalk

  z_alktot = z_alktot_arr(ji_alk)

  DO ji_dic = 1, jp_ndic

    z_dictot = z_dictot_arr(ji_dic)

    z_h = SOLVE_AT_GENERAL(z_alktot, z_dictot, z_bortot,                       &
                           z_po4tot, z_siltot, z_nh4tot, z_h2stot,             &
                           z_so4tot, z_flutot, p_val=z_val)

    z_h_atgen(ji_dic,ji_alk)   = z_h
    z_val_atgen(ji_dic,ji_alk) = z_val

    IF(z_h == pp_hnan) &
      jndivg_atgen(ji_dic,ji_alk)      = 1

    CALL SPECIATION_DIC(z_dictot, z_h, z_co2, z_hco3, z_co3)

    z_co2_atgen(ji_dic,ji_alk)  = z_co2
    z_hco3_atgen(ji_dic,ji_alk) = z_hco3
    z_co3_atgen(ji_dic,ji_alk)  = z_co3

  ENDDO


  DO ji_ph = 1, jp_nph

    z_h = 10._wp**(-z_ph_arr(ji_ph))

    z_alkc = -EQUATION_AT(z_alktot, z_h,       z_zero,   z_bortot,           &
                          z_po4tot, z_siltot,  z_nh4tot, z_h2stot,           &
                          z_so4tot, z_flutot)

    IF ( z_alkc < 0._wp ) THEN
      z_co2  = -1._wp
      z_hco3 = -1._wp
      z_co3  = -1._wp
    ELSE
      z_co2  = z_alkc * (    (z_h*z_h)   / (api1_dic*z_h + api2_dic+api2_dic) )
      z_hco3 = z_alkc * ( (api1_dic*z_h) / (api1_dic*z_h + api2_dic+api2_dic) )
      z_co3  = z_alkc * (       api2_dic / (api1_dic*z_h + api2_dic+api2_dic) )
    ENDIF

    z_co2_ph_at(ji_ph,ji_alk)  = z_co2
    z_hco3_ph_at(ji_ph,ji_alk) = z_hco3
    z_co3_ph_at(ji_ph,ji_alk)  = z_co3

  ENDDO


ENDDO



 CALL CPU_TIME(z_cputime_end)

 PRINT*, '[DRIVER_AT_CARBONATE] elapsed time [s]           : ', z_cputime_end - z_cputime_start
 PRINT*, '[DRIVER_AT_CARBONATE] total number of calls      : ', jp_nalk*jp_ndic
 PRINT*, '[DRIVER_AT_CARBONATE] total number of divergences: ', SUM(jndivg_atgen)


#if defined(CREATEFILES)
 OPEN(UNIT=jp_resunit, FILE='carbonate_'//cp_fileid//'.res',FORM='UNFORMATTED')

 WRITE(UNIT=jp_resunit) jp_ndic, jp_nalk, jp_nph

 WRITE(UNIT=jp_resunit) z_dictot_min
 WRITE(UNIT=jp_resunit) z_dictot_max

 WRITE(UNIT=jp_resunit) z_alktot_min
 WRITE(UNIT=jp_resunit) z_alktot_max

 WRITE(UNIT=jp_resunit) z_ph_min
 WRITE(UNIT=jp_resunit) z_ph_max

 WRITE(UNIT=jp_resunit) z_dictot_arr
 WRITE(UNIT=jp_resunit) z_alktot_arr
 WRITE(UNIT=jp_resunit) z_ph_arr

 WRITE(UNIT=jp_resunit) z_bortot
 WRITE(UNIT=jp_resunit) z_po4tot
 WRITE(UNIT=jp_resunit) z_siltot
 WRITE(UNIT=jp_resunit) z_nh4tot
 WRITE(UNIT=jp_resunit) z_h2stot
 WRITE(UNIT=jp_resunit) z_so4tot
 WRITE(UNIT=jp_resunit) z_flutot

 WRITE(UNIT=jp_resunit) ((z_h_atgen(ji_dic,ji_alk),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk)
 WRITE(UNIT=jp_resunit) ((z_val_atgen(ji_dic,ji_alk),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk)

 WRITE(UNIT=jp_resunit) ((z_co2_atgen(ji_dic,ji_alk),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk)
 WRITE(UNIT=jp_resunit) ((z_hco3_atgen(ji_dic,ji_alk),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk)
 WRITE(UNIT=jp_resunit) ((z_co3_atgen(ji_dic,ji_alk),ji_dic=1,jp_ndic),ji_alk=1,jp_nalk)

 WRITE(UNIT=jp_resunit) ((z_co2_ph_at(ji_ph,ji_alk),ji_ph=1,jp_nph),ji_alk=1,jp_nalk)
 WRITE(UNIT=jp_resunit) ((z_hco3_ph_at(ji_ph,ji_alk),ji_ph=1,jp_nph),ji_alk=1,jp_nalk)
 WRITE(UNIT=jp_resunit) ((z_co3_ph_at(ji_ph,ji_alk),ji_ph=1,jp_nph),ji_alk=1,jp_nalk)

 CLOSE(UNIT=jp_resunit)
#endif




!==============================================================================================
END PROGRAM
!==============================================================================================

