%% A MULTIPLE TIMESCALE APPROACH OF BISPECTRAL CORRELATION
% Authors: V. Denol, M. Esposito Marzino and M. Geuzaine
% Journal of Wind Engineering and Industrial Aerodynamics
%
% This code aims to reproduce the results presented in the illustrations of
% the article cited above. A model of a simple bridge with three spans and
% one point load is constructed. Variances and skewnesses are computed in
% time domain, in the nodal basis and in the modal basis, considering nodal
% recombinations with or without correlations. These values are compared to
% the outputs obtained in the frequency domain. There are two methods. The
% first one is based on the numerical integration of the spectra at 2nd and
% 3rd orders. The second one is based on the formulas derived in the paper
% with the help of the Multiple Timescale Spectral Analysis.
%
% Proper reference should be made to the paper cited hereabove if the code
% provided below and in the associated folder is used, in part or in full. 
%
%%

clc
clear
close all

doTimeDomainNodal = 1;
doTimeDomainModal = 1;
doFreqDomainBispe = 1;

%% Define finite element model
i_xi = 1 % 0.05:0.05:1;

for ii_xi=1:length(i_xi)

l1 = 100; 
l2 = 45.8; %46; 45.5;
n = 25;
n_span = 3;

xi2 = 1e-2*i_xi(ii_xi) %/ 4 ;
xi3 = 1e-2*i_xi(ii_xi) %/ 4 ;


[XNOD,YNOD,FIXNOD,NNode,ELEMNOA,ELEMNOB,ELEMLEN,ELEMGEO,ELEMMAT,ELEMTYP,NElem,APPU,CORRES] = ...
    StructurePredefinie_Poutre (0,n_span,[l2 l1 l2],n);
NDOF=3*NNode;
xx=1:NDOF; xx(1:3:end)=0; xx([2:3*n:end])=0; CORRES= xx(xx~=0);

GEOMSEC = 1;
GEOMINE = 1e12;
MATEYOU = 1;
MATEMAS = 100e3;
[K,M,~,Ke] = Assemblage(NNode,NElem,XNOD,YNOD,ELEMGEO,ELEMMAT,ELEMLEN,ELEMNOA,ELEMNOB,ELEMTYP,...
    GEOMSEC,GEOMINE,MATEYOU,MATEMAS);

K=K(CORRES,CORRES);
M=M(CORRES,CORRES);

%% DEFINITION OF LOAD
iload = 16;  % Number of the DOF where the only load is applied

U   = 30;
sigu= 5;
L   = 100;
c0 = 1;
c1 = 2;
c2 = 0;
coef = 5e4;

dt = 0.01;
N  = 2e6; %2e6;

Su = @(w__)   4*L/U*sigu^2 ./ (1+70.7*(w__*L/(2*pi*U)).^2).^(5/6) / (4*pi);
Sf = @(w__)  coef^2 * c1^2 * U^2 * Su(w__);

Bf = @(w1__,w2__) coef^3 * U^2 * c1^2*(2*c0+c2) * ( Su(w1__+w2__).*Su(w1__) + Su(w1__).*Su(w2__) + Su(w1__+w2__).*Su(w2__) );

mom3F = coef^3 * U^2 * c1^2*(2*c0+c2) * 3*sigu^4;

%% MODAL BASIS
disp ('Modal analysis')
[V, D]= eig(K,M);
omega = sqrt(diag(D));
V = NormalizeMode(V);
V(:,1) = -V(:,1);

for imod = 1:4
    subplot(2,2,imod)
    AMPL = 1;
    PlotDefo (V(:,imod),CORRES,NDOF,AMPL,XNOD,YNOD,ELEMNOA,ELEMNOB,ELEMLEN)
    axis normal
    grid on
    title (sprintf('Mode %d, omega=%0.3f',imod,omega(imod)))
end

Nmodes = 4;
V = V(:,1:Nmodes);

V(iload,:) % = [ 0.1305    0.5869    0.8835    0.9166 ]
% Rayleigh damping
% NB: impose damping in 2 and 3, in order to have slightly larger damping
% in mode 1
[alpha,beta] = RayleighDamping(omega(2),omega(3),xi2,xi3);
C = alpha*M + beta*K;


% Modal matrices & loads
Kgen = V'*K*V;
Mgen = V'*M*V;
Cgen = V'*C*V;

% Damping and system memory
xi = diag(Cgen./sqrt(Kgen.*Mgen))/2;

T1_3 = 2*pi./omega(1:3);
Tr = T1_3./2./xi(1:3); % all the same if mass proportional damping
nr = round(Tr(1)/dt);

%% TIME DOMAIN ANALYSIS
nRun = 1;
for iRun=1:nRun
    
    % Generation of load
    [u,t] = generateSamples(Su,N,dt);
    f = coef * U^2 * ( c0 + c1*u/U + (c0+c2/2)*(u/U).^2);
    
    %figure , plot(f), title (sprintf('Skewness %0.3f',skewness(f)))
    P = zeros(length(K),N);
    P(iload,:) = f;

    
    fprintf ('\nMONTE CARLO SIMULATION : Run #%d out of %d',iRun,nRun)
    if doTimeDomainNodal
        fprintf ('\n  Time stepping in nodal basis')
        tic
        Sol_Nodal{iRun} = solutionNodal(M,K,C,P,dt,N,t,nr,NDOF,CORRES,NElem,Ke);
        Sol_Nodal{iRun} = doStats(Sol_Nodal{iRun});
        timeNodal(iRun) = toc;
    end
    
    if doTimeDomainModal
        fprintf ('\n  Time stepping in modal basis')
        Pgen = V'*P;
        tic
        Sol_Modal{iRun} = solutionModal(Mgen,Kgen,Cgen,Pgen,V,dt,N,t,nr,NDOF,CORRES,NElem,Ke);
        Sol_Modal{iRun}  = doStats(Sol_Modal{iRun});
        timeModal(iRun) = toc;
    end
end

%% SPECTRAL AND BISPECTRAL ANALYSIS

Spectral = solutionSpectral(Mgen,Kgen,Cgen,xi,omega,U/L,V,Sf,Bf,mom3F,iload,NDOF,CORRES,NElem,Ke,doFreqDomainBispe);

%save (sprintf('save_%d',ii_xi))
end
%% PLOTS
cmp = get(groot,'DefaultAxesColorOrder');

%%
fig_modal_mom4 = figure; % 3rd order moments of modal responses
figure(fig_modal_mom4)

plotMom3(real(Spectral.m3Q_MTSA),   'color','k','marker','s','MarkerFaceColor','k');   hold on
plotMom3(real(Spectral.m3Q_MTSA_B), 'color',cmp(1,:),'marker','s','MarkerFaceColor','k');   hold on
plotMom3(real(Spectral.m3Q_MTSA_BR),'color',cmp(2,:),'marker','s','MarkerFaceColor','k');   hold on
plotMom3(real(Spectral.m3Q_MTSA_TR),'color',cmp(3,:),'marker','s','MarkerFaceColor','k');   hold on

legend ('MTSA (sum)','Background','Biresonant','Triresonant')
grid on


fig_modal_mom5 = figure; % 3rd order moments of modal responses
figure(fig_modal_mom5)

plotMom3(real(Spectral.m3Q_MTSA_B)./real(Spectral.m3Q_MTSA), 'color',cmp(1,:));   hold on
plotMom3(real(Spectral.m3Q_MTSA_BR)./real(Spectral.m3Q_MTSA),'color',cmp(2,:));   hold on
plotMom3(real(Spectral.m3Q_MTSA_TR)./real(Spectral.m3Q_MTSA),'color',cmp(3,:));   hold on

legend ('B','B+BR','B+BR+TR')
grid on
%%
disp ('Skewness of modal responses')
Spectral.momQ(:,3)
Sol_Modal{iRun}.momQ(:,3)
Sol_Modal{iRun}.momQ_qs(:,3)

fig_modal1 = figure; % Variance of modal responses
fig_modal2 = figure; % plotTF of modal responses


figure(fig_modal1),
plot(1:Nmodes, Spectral.momQ(:,2),'ko','MarkerFaceColor','k'), hold on, title ('Modal responses : STD')
for iRun=1:nRun
    plot(1:Nmodes,Sol_Modal{iRun}.momQ(:,2))
end
legend ('Spectral','Monte Carlo')



figure(fig_modal2);
iRun = 1;
plotTF(t(nr:end), Sol_Modal{iRun}.q,2048*8,[],2); title ('Modal responses')
for i1=1:Nmodes
    semilogy(Spectral.w/2/pi,squeeze(Spectral.Sq(i1,i1,:)),'color',cmp(i1,:)); hold on, xlabel ('Frequence [Hz]')
end

%%
figure
subplot(121),
for i1=1:Nmodes
    semilogy(Spectral.w/2/pi,squeeze(Spectral.Sq(i1,i1,:))); hold on, xlabel ('Frequence [Hz]')
end
subplot(222), bar(sqrt(Spectral.varq)), title ('STD of modal responses')
subplot(224), DessinCorrel(Spectral.corr), title ('Correlation of modal responses')

%%
fig_modal_mom3 = figure; % 3rd order moments of modal responses

figure(fig_modal_mom3)

plotMom3(Spectral.m3Q,           'color','k','marker','o','MarkerFaceColor','k');   hold on
plotMom3(real(Spectral.m3Q_MTSA),'color','k','marker','s','MarkerFaceColor','k');   hold on

for iRun=1:nRun
    plotMom3(Sol_Modal{iRun}.m3Q,'color',cmp(1,:)); hold on    
end
legend ('Bispectral analysis','MTSA','Monte Carlo')
grid on
%%
fig_modal_mom4 = figure; % 3rd order moments of modal responses
figure(fig_modal_mom4)

plotMom3(real(Spectral.m3Q_MTSA),   'color','k','marker','s','MarkerFaceColor','k');   hold on
plotMom3(real(Spectral.m3Q_MTSA_B), 'color',cmp(1,:),'marker','s','MarkerFaceColor','k');   hold on
plotMom3(real(Spectral.m3Q_MTSA_BR),'color',cmp(2,:),'marker','s','MarkerFaceColor','k');   hold on
plotMom3(real(Spectral.m3Q_MTSA_TR),'color',cmp(3,:),'marker','s','MarkerFaceColor','k');   hold on

legend ('MTSA (sum)','Background','Biresonant','Triresonant')
grid on


fig_modal_mom5 = figure; % 3rd order moments of modal responses
figure(fig_modal_mom5)

plotMom3(real(Spectral.m3Q_MTSA_B)./real(Spectral.m3Q_MTSA), 'color',cmp(1,:));   hold on
plotMom3(real(Spectral.m3Q_MTSA_BR)./real(Spectral.m3Q_MTSA),'color',cmp(2,:));   hold on
plotMom3(real(Spectral.m3Q_MTSA_TR)./real(Spectral.m3Q_MTSA),'color',cmp(3,:));   hold on

legend ('B','B+BR','B+BR+TR')
grid on
%% plot stats of displacement everywhere
figure
for i=2:3
    subplot(2,1,i-1)
    plot(XNOD, Spectral.momX(:,i),     'ko','MarkerFaceColor','k','MarkerSize',3); grid on; hold on
    plot(XNOD, Spectral.momX_CRSC(:,i),'ko','MarkerFaceColor','w','MarkerSize',3); grid on; hold on
    plot(XNOD, Spectral.momX_MTSA(:,i),     'ks','MarkerFaceColor','k','MarkerSize',3); grid on; hold on
    plot(XNOD, Spectral.momX_MTSA_CRSC(:,i),'ks','MarkerFaceColor','w','MarkerSize',3); grid on; hold on
    
    title (sprintf ('Statistical moment of transverse displacement, order %d',i))
end
for iRun = 1:nRun
    for i=2:3
        subplot(2,1,i-1)
        plot(XNOD, Sol_Nodal{iRun}.momX(:,i)     ,'color',[0.7 0.7 0.7]); grid on; hold on
        plot(XNOD, Sol_Modal{iRun}.momX(:,i)     ,'color',cmp(1,:)); grid on; hold on
        plot(XNOD, Sol_Modal{iRun}.momX_CRSC(:,i),'color',cmp(2,:)); grid on; hold on
        plot(XNOD, Sol_Modal{iRun}.momX_qs(:,i),  'color',cmp(4,:)); grid on; hold on
        title (sprintf ('Statistical moment of transverse displacement, order %d',i))
    end
end
%% plot stats of bending moments everywhere
figure
for i=2:3
    subplot(2,1,i-1)
    plot(XNOD, Spectral.momM(:,i),     'ko','MarkerFaceColor','k','MarkerSize',3); grid on; hold on
    plot(XNOD, Spectral.momM_CRSC(:,i),'ko','MarkerFaceColor','w','MarkerSize',3); grid on; hold on
    plot(XNOD, Spectral.momM_MTSA(:,i),     'ks','MarkerFaceColor','k','MarkerSize',3); grid on; hold on
    plot(XNOD, Spectral.momM_MTSA_CRSC(:,i),'ks','MarkerFaceColor','w','MarkerSize',3); grid on; hold on
    
    title (sprintf ('Statistical moment of bending moment, order %d',i))
end
for iRun = 1:nRun
    for i=2:3
        subplot(2,1,i-1)
        plot(XNOD, Sol_Nodal{iRun}.momM(:,i)     ,'color',[0.7 0.7 0.7]); grid on; hold on
        plot(XNOD, Sol_Modal{iRun}.momM(:,i)     ,'color',cmp(1,:)); grid on; hold on
        plot(XNOD, Sol_Modal{iRun}.momM_CRSC(:,i),'color',cmp(2,:)); grid on; hold on
        plot(XNOD, Sol_Modal{iRun}.momM_qs(:,i),  'color',cmp(4,:)); grid on; hold on
        title (sprintf ('Statistical moment of bending moment, order %d',i))
    end
end

%%
figure
w1=Spectral.w1;
w2=Spectral.w2;

Bfgen = @(w1__,w2__,i,j,k) V(iload,i)*V(iload,j)*V(iload,k)*Bf(w1__,w2__);
H = @(w__,i) 1./(-Mgen(i,i)*w__.^2+1i*w__*Cgen(i,i)+Kgen(i,i));
Bq = @(w1__,w2__,i,j,k)  Bfgen(w1__,w2__,i,j,k).*H(w1__,i).*H(w2__,j).*conj(H(w1__+w2__,k));

subplot(131)
i=1;j=1;k=1;
plotBisp(w1,w2,real(Bq(w1,w2,i,j,k)),500); colorbar

subplot(132)
i=1;j=2;k=2;
plotBisp(w1,w2,real(Bq(w1,w2,i,j,k)),500); colorbar

subplot(133)
i=1;j=2;k=3;
plotBisp(w1,w2,real(Bq(w1,w2,i,j,k)),500); colorbar
