%
% State-space matrices of shunts, decentralized or centralized networks for
% multimodal piezoelectric damping
%
% (c) G. Raze, J. Dietrich, B. Lossouarn and G. Kerschen
% Mar 2022
%
% Input:
%   * Omoc: matrix of open-circuit resonance frequencies
%   * B: piezoelectric structure state input matrix
%   * C: piezoelectric structure state output matrix
%   * modes: targeted modes; this should be a Ng x 1 cell vector, whose
%   gth entry contains the indices of the modes targeted by group g.
%   * patches: patches of the groups; this should be a Ng x 1 cell vector,
%   whose gth entry contains the indices of the patches in group g
%   * w: control weight; this should be a Ng x 1 cell vector, whose gth
%   entry has the same size as modes{g} and should represent the relative
%   weight attributed to the targeted modes.
%   * chargeModel: set to true if the state-space represents a
%   voltage-to-charge relation of the electrical circuit, set to false
%   (default) if it represents the voltage-to-current relation (i.e., the
%   admittance).
% 
% Output:
%   * An: state evolution matrix of the circuit
%   * Bn: state input matrix of the circuit
%   * Cn: state output matrix of the circuit
%   * Kcn: estimate of the modal electromechanical coupling factors between
%   the targeted modes and the whole circuit
% 
% See the associated scripts for examples
%

function [An,Bn,Cn,Kcn] = makeDecentralizedNetwork(Omoc,B,C,D,modes,patches,w,chargeModel)


  if nargin < 8
    chargeModel = false;
    if nargin < 7
      if nargin < 6
        modes = 1:1:length(Omoc);
      end
      W = eye(length(modes));
    end
  end

  Nm = length(Omoc);    % Number of modes
  Ng = length(modes);   % Number of groups
  Np = length(D);       % Number of patches


  
  
  %% Electrical mode shapes
  Nt = numel(cell2mat(modes));        % Total number of electrical modes
  PhiIn = zeros(Np,Nt);               % Input mode shapes matrix
  PhiOut = zeros(Np,Nt);              % Output mode shapes matrix
  kGlobal = cell(length(modes),1);    % Global electrical mode indices
  modesUnique = [];                   % Targeted modes
  DB = zeros(size(D));                % Block-diagonal part of D
  
  Ntp = 0;                            % Temporary index
  for gg = 1 : length(modes)
    Wg = diag(w{gg});
    kGlobal{gg} = Ntp + (1 : 1 : length(modes{gg}))';
    Ntp = Ntp + length(modes{gg});
    modesUnique = unique(sort([modesUnique;modes{gg}(:)]));
    
    DB(patches{gg},patches{gg}) = D(patches{gg},patches{gg});
    [UD,SD,VD] = svd(D(patches{gg},patches{gg}));
    SqrtSDI = (eye(length(SD))/SD).^0.5;
    [UM,SM,VM] = svd(Wg*(Omoc(modes{gg},modes{gg})\B(modes{gg},patches{gg}))*VD*SqrtSDI,'econ');
  
    PhiIng = UD*SqrtSDI*VM*UM.';
    PhiOutg = VD*SqrtSDI*VM*UM.';
    
    PhiIn(patches{gg},kGlobal{gg}) = PhiIng;
    PhiOut(patches{gg},kGlobal{gg}) = PhiOutg;
    
  end
  
  BPhiOut = B*PhiOut;
  PhiInTC = PhiIn'*C;
  Ns = length(modesUnique);
  
  %% Electrical resonance frequencies and damping ratios
  Ome = zeros(Nt);                              % Electrical resonance frequencies
  Ze2Ome = zeros(Nt);                           % Electrical damping
  Kcn = zeros(Ns,1);                            % Modal electromechanical coupling factor
  Delta = eye(Nt) + PhiIn.'*(D-DB)*PhiOut;      % Modal elastance matrix
  
  for kk = 1 : Ns
    % Resonant and non-resonant modes indices
    rr = modesUnique(kk);
    indRSup = (rr+1):1:Nm;
    
    kk2 = [];
    indKSup = [];
    for gg = 1 : Ng
      kk2 = [kk2 ; kGlobal{gg}(modes{gg} == rr)];
      indKSup = [indKSup ; kGlobal{gg}(modes{gg} > rr)];
    end
    
    % Effective electromechanical system
    ST = [Omoc(rr,rr)^2 ,             BPhiOut(rr,kk2) ,       zeros(1,length(indRSup)) ,  BPhiOut(rr,indKSup) ;
          PhiInTC(kk2,rr) ,           Delta(kk2,kk2) ,        PhiInTC(kk2,indRSup) ,      Delta(kk2,indKSup) ;
          zeros(length(indRSup),1) ,  BPhiOut(indRSup,kk2) ,  Omoc(indRSup,indRSup)^2 ,   BPhiOut(indRSup,indKSup) ; 
          PhiInTC(indKSup,rr) ,       Delta(indKSup,kk2) ,    PhiInTC(indKSup,indRSup) ,  Delta(indKSup,indKSup)];
    Nr = 1 + length(kk2);
        
    sysM = ST(1:Nr,1:Nr) - ST(1:Nr,Nr+1:end)*(ST(Nr+1:end,Nr+1:end)\ST(Nr+1:end,1:Nr));
    bk = -sysM(1,2:end);
    dk = sysM(2:end,2:end);
    ck = -sysM(2:end,1);
    
    % Effective characteristics
    woch2 = sysM(1,1);
    wsch2 = woch2 - bk*(dk\ck);
    Kcn(kk) = sqrt(woch2/wsch2-1);
     
    % Absorber frequency and damping ratio
    [Rh,Lh,zeta,delta2] = ikegameRLShunt(1,sqrt(wsch2),sqrt(woch2));
    omegaek2 = delta2*woch2;
    
    wem2 = diag(ck)\(dk*ck)/omegaek2;
    we = sqrt(1./wem2);
    Ome(kk2,kk2) = diag(we);
    Ze2Ome(kk2,kk2) = 2*zeta*Ome(kk2,kk2);
   
  end
  
  %% Impedance state-space model
  dA = zeros(Nt);
  for gg = 1 : Ng
    dA(kGlobal{gg},kGlobal{gg}) = -Ome(kGlobal{gg},kGlobal{gg})*...
      (eye(length(kGlobal{gg})) - PhiIn(patches{gg},kGlobal{gg}).'*D(patches{gg},patches{gg})*PhiOut(patches{gg},kGlobal{gg}))*...
      Ome(kGlobal{gg},kGlobal{gg});
  end
  
  An = [zeros(Nt) , eye(Nt) ; dA , -Ze2Ome];
  Bn = [zeros(Nt,Np) ; Ome*(PhiIn.')];
  if chargeModel
    Cn = [PhiOut*Ome , zeros(Np,Nt)];
  else
    Cn = [zeros(Np,Nt),PhiOut*Ome];
    
    Tr = null(null(An)');
    An = Tr.'*An*Tr;
    Bn = Tr.'*Bn;
    Cn = Cn*Tr;
  end

end