%
% Damping of a free-free beam with RL shunts, decentralized and centralized
% electrical networks
%
% (c) G. Raze, J. Dietrich, B. Lossouarn and G. Kerschen
% Mar 2022

%% Load model

load('FreeFreeBeamModel');
N = length(beamss.A)/2;

Omoc2 = -beamss.A((N+1):end,1:N);
Omoc = diag(sqrt(diag(Omoc2(4:end,4:end))));
B = beamss.B((N+4):end,2:end);
C = beamss.C(2:end,4:N);
D = -beamss.D(2:end,2:end);

%% Case 1: Multi-resonant shunts

modes = cell(20,1);
patches = cell(20,1);
w = cell(20,1);
for nn = 1 : 20
  modes{nn} = 1:1:4;
  patches{nn} = nn;
  w{nn} = ones(1,4);
end

% Compute shunts sate-space matrices
[An,Bn,Cn,Kcn] = makeDecentralizedNetwork(Omoc,B,C,D,modes,patches,w,true);

% Connect the beam to the shunts  
sysShunt = feedback(beamss,-ss(An,Bn,Cn,zeros(20)),2:21,2:21);
   

%% Case 2: Four decentralized networks, adjacent distribution

modes = {1:1:4 ; 1:1:4 ; 1:1:4 ; 1:1:4};
patches = {1:1:5 ; 6:1:10 ; 11:1:15 ; 16:1:20};
w = {ones(4,1) ; ones(4,1) ; ones(4,1) ; ones(4,1)};

% Compute network sate-space matrices
[An,Bn,Cn,Kcn] = makeDecentralizedNetwork(Omoc,B,C,D,modes,patches,w,true);

% Connect the beam to the network
sysDecentralizedNetworkAdjacent = feedback(beamss,-ss(An,Bn,Cn,zeros(20)),2:21,2:21);



%% Case 3: Four decentralized networks, alternate distribution

modes = {1:1:4 ; 1:1:4 ; 1:1:4 ; 1:1:4};
patches = {1:4:20 ; 2:4:20 ; 3:4:20 ; 4:4:20};
w = {ones(4,1) ; ones(4,1) ; ones(4,1) ; ones(4,1)};

% Compute network sate-space matrices
[An,Bn,Cn,Kcn] = makeDecentralizedNetwork(Omoc,B,C,D,modes,patches,w,true);

% Connect the beam to the network
sysDecentralizedNetworkAlternate = feedback(beamss,-ss(An,Bn,Cn,zeros(20)),2:21,2:21);


%% Case 4: Fully centralized network

modes = {1:1:4};
patches = {1:20};
w = {ones(4,1)};

% Compute network sate-space matrices
[An,Bn,Cn,Kcn] = makeDecentralizedNetwork(Omoc,B,C,D,modes,patches,w,true);

% Connect the beam to the network
sysCentralizedNetwork = feedback(beamss,-ss(An,Bn,Cn,zeros(20)),2:21,2:21);


%% Compare results
omegav = 2*pi*linspace(80,1200,1e4);
omegav = omegav(:);

figure('Units','normalized','Position',[0.1,0.25,0.8,0.5])
hold on
plot(omegav/2/pi,db(omegav.*squeeze(freqresp(beamss(1,1),omegav))),'-k')
plot(omegav/2/pi,db(omegav.*squeeze(freqresp(sysShunt(1,1),omegav))),'linewidth',2)
plot(omegav/2/pi,db(omegav.*squeeze(freqresp(sysDecentralizedNetworkAdjacent(1,1),omegav))),'-','linewidth',2)
plot(omegav/2/pi,db(omegav.*squeeze(freqresp(sysDecentralizedNetworkAlternate(1,1),omegav))),'-','linewidth',2)
plot(omegav/2/pi,db(omegav.*squeeze(freqresp(sysCentralizedNetwork(1,1),omegav))),'-.','linewidth',2)
set(gca,'xscale','log')
xlabel('Frequency (Hz)')
ylabel('Mobility (dB, ref. m/s/N)')
set(gca,'fontsize',14)
box on
xlim([omegav(1),omegav(end)]/2/pi)
ylim([-60,-15])

%% AutoMAC matrices: adjacent distribution

modes = {1:1:4 ; 1:1:4 ; 1:1:4 ; 1:1:4};
patches = {1:1:5 ; 6:1:10 ; 11:1:15 ; 16:1:20};

for gg = 1 : 4
  % SVD
  [UD,SD,VD] = svd(D(patches{gg},patches{gg}));
  
  % Optimal dimensionless mode shapes
  Phi = B(modes{gg},patches{gg})*VD*sqrt(SD);
  
  % AutoMAC
  MAC = diag(1./diag(Phi*Phi'))*(Phi*Phi').^2*diag(1./diag(Phi*Phi'));
  
  % Plot
  figure
  imagesc(MAC)
  colormap(gray)
  colorbar
  caxis([0 1])
  box on
  set(gca,'YDir','normal')
  set(gca,'fontsize',20)
  xlabel('Mode number (-)')
  ylabel('Mode number (-)')
end


%% AutoMAC matrices: alternate distribution

modes = {1:1:4 ; 1:1:4 ; 1:1:4 ; 1:1:4};
patches = {1:4:20 ; 2:4:20 ; 3:4:20 ; 4:4:20};

for gg = 1 : 4
  % SVD
  [UD,SD,VD] = svd(D(patches{gg},patches{gg}));
  
  % Optimal dimensionless mode shapes
  Phi = B(modes{gg},patches{gg})*VD*sqrt(SD);
  
  % AutoMAC
  MAC = diag(1./diag(Phi*Phi'))*(Phi*Phi').^2*diag(1./diag(Phi*Phi'));
  
  % Plot
  figure
  imagesc(MAC)
  colormap(gray)
  colorbar
  caxis([0 1])
  box on
  set(gca,'YDir','normal')
  set(gca,'fontsize',20)
  xlabel('Mode number (-)')
  ylabel('Mode number (-)')
end
