%
% Damping of a clamped-free beam with multi-resonant shunts, decentralized
% and centralized networks
%
% (c) G. Raze, J. Dietrich, B. Lossouarn and G. Kerschen
% Mar 2022

%% Load model

load('ClampedFreeBeamModel');
N = length(beamss.A)/2;

Omoc2 = -beamss.A((N+1):end,1:N);
Omoc = diag(sqrt(diag(Omoc2)));
B = beamss.B((N+1):end,2:end);
C = beamss.C(2:end,1:N);
D = -beamss.D(2:end,2:end);

wModes = ones(4,1);

%% Case 1: Multi-resonant shunts

modes = cell(5,1);
patches = cell(5,1);
w = cell(5,1);
for nn = 1 : 5
  modes{nn} = 1:1:4;
  patches{nn} = nn;
  w{nn} = wModes;
end

% Compute shunts sate-space matrices
[An,Bn,Cn,Kcn] = makeDecentralizedNetwork(Omoc,B,C,D,modes,patches,w,true);

% Connect the beam to the shunts  
sysShunt = feedback(beamss,-ss(An,Bn,Cn,zeros(5)),2:6,2:6);
   

%% Case 2: Three decentralized networks

modes = {1:1:4 ; 1:1:4 ; 1:1:4};
patches = {1:2 ; 3:4 ; 5};
w = {wModes ; wModes ; wModes};

% Compute network sate-space matrices
[An,Bn,Cn,Kcn] = makeDecentralizedNetwork(Omoc,B,C,D,modes,patches,w,true);

% Connect the beam to the network
sysDecentralizedNetwork = feedback(beamss,-ss(An,Bn,Cn,zeros(5)),2:6,2:6);


%% Case 3: Fully centralized network

modes = {1:1:4};
patches = {1:5};
w = {wModes};

% Compute network sate-space matrices
[An,Bn,Cn,Kcn] = makeDecentralizedNetwork(Omoc,B,C,D,modes,patches,w,true);

% Connect the beam to the network
sysCentralizedNetwork = feedback(beamss,-ss(An,Bn,Cn,zeros(5)),2:6,2:6);


%% Compare results
omegav = 2*pi*linspace(20,1000,1e4);
omegav = omegav(:);

figure('Units','normalized','Position',[0.1,0.25,0.8,0.5])
hold on
plot(omegav/2/pi,db(squeeze(freqresp(beamss(1,1),omegav))),'-k')
plot(omegav/2/pi,db(squeeze(freqresp(sysShunt(1,1),omegav))),'linewidth',2)
plot(omegav/2/pi,db(squeeze(freqresp(sysDecentralizedNetwork(1,1),omegav))),'-','linewidth',2)
plot(omegav/2/pi,db(squeeze(freqresp(sysCentralizedNetwork(1,1),omegav))),'-.','linewidth',2)
set(gca,'xscale','log')
xlabel('Frequency (Hz)')
ylabel('Receptance (dB, ref. m/N)')
set(gca,'fontsize',14)
box on
xlim([20,1e3])
set(gca,'XTick',[20,50,100,200,500,1000])
%ylim([-60,-15])

