%
% Plot the experimental FRFs of a clamped-free piezoelectric beam with a
% thin lamina controlled by shunts or newtorks.
%
% Robustness against frequency and mode shapes detuning.
%
% (c) G. Raze, J. Dietrich, B. Lossouarn and G. Kerschen
% Mar 2022

expFolder1 = '.\Data\2p\Robustness_Freq\';       % Data folder 1
expFolder2 = '.\Data\2p\Robustness_Shape\';      % Data folder 2


FRFFactor = 1/4.44822;         % Factor for the FRF (there was an error in the sensitivity of a sensor)

% Open circuit
[froc,hoc] = TestLabFRFData('.\Data\OC2');
hoc = abs(hoc./(2*pi*froc).^2)*FRFFactor;


% Frequency range for the peaks
frRange1 = [25,35];
frRange2 = [120,160];

% Weight vector (w1/w2 without correction, see readme.txt)
w =  [logspace(-3,-1,5) , logspace(-1,1,11) , logspace(1,3,5)];
w([5,16]) = [];

% Peak amplitudes (open circuit)
h1oc = max(hoc(and(froc>frRange1(1),froc<frRange1(2))));
h2oc = max(hoc(and(froc>frRange2(1),froc<frRange2(2))));

% Frequency detuning
fv = linspace(0.9,1.1,21);
fvn = {'0-9','0-91','0-92','0-93','0-94','0-95','0-96','0-97','0-98','0-99',...
  '1','1-01','1-02','1-03','1-04','1-05','1-06','1-07','1-08','1-09','1-1'};

% Mode shapes detuning
thetav = linspace(0,pi,21);

col = get(groot,'defaultAxesColorOrder');


% Peak values - frequency detuning

h1n_f = zeros(size(fv));            % Mode 1, network
h2n_f = zeros(size(fv));            % Mode 2, newtork
h1d_f = zeros(size(fv));            % Mode 1, shunts
h2d_f = zeros(size(fv));            % Mode 2, shunts

for ii = 1 : length(fv)
  [fr,h] = TestLabFRFData([expFolder1,'Network_f_',fvn{ii}]);
  hx = abs(h./(2*pi*fr).^2)*FRFFactor;
  
  h1n_f(ii) = max(hx(and(fr>frRange1(1),fr<frRange1(2))));
  h2n_f(ii) = max(hx(and(fr>frRange2(1),fr<frRange2(2))));
  
  [fr,h] = TestLabFRFData([expFolder1,'Shunts_f_',fvn{ii}]);
  hx = abs(h./(2*pi*fr).^2)*FRFFactor;
  
  h1d_f(ii) = max(hx(and(fr>frRange1(1),fr<frRange1(2))));
  h2d_f(ii) = max(hx(and(fr>frRange2(1),fr<frRange2(2))));

end

% Peak values - mode shape detuning

h1n_s = zeros(size(thetav));            % Mode 1, networks
h2n_s = zeros(size(thetav));            % Mode 2, networks
h1d_s = zeros(size(thetav));            % Mode 1, shunts
h2d_s = zeros(size(thetav));            % Mode 2, shunts


for ii = 1 : length(thetav)
  
  [fr,h] = TestLabFRFData([expFolder2,'Network_t_',num2str(ii)]);
  hx = abs(h./(2*pi*fr).^2)*FRFFactor;
  
  h1n_s(ii) = max(hx(and(fr>frRange1(1),fr<frRange1(2))));
  h2n_s(ii) = max(hx(and(fr>frRange2(1),fr<frRange2(2))));
  
  
  [fr,h] = TestLabFRFData([expFolder2,'Shunts_t_',num2str(ii)]);
  hx = abs(h./(2*pi*fr).^2)*FRFFactor;
  
  h1d_s(ii) = max(hx(and(fr>frRange1(1),fr<frRange1(2))));
  h2d_s(ii) = max(hx(and(fr>frRange2(1),fr<frRange2(2))));
end

%% Plot - frequency detuning robustness
indMidf = ceil(length(fv)/2);       % Reference value

figure
hold on
plot([fv(1),fv(end)],[0,0],'-','Color',[0.5,0.5,0.5])
plot(fv,db(h1n_f/h1n_f(indMidf)),'-o','Markersize',10,'linewidth',2,'Color',col(1,:))
plot(fv,db(h2n_f/h2n_f(indMidf)),'-o','Markersize',10,'linewidth',2,'Color',col(2,:))
plot(fv,db(h1d_f/h1d_f(indMidf)),'-x','Markersize',15,'linewidth',2,'Color',col(3,:))
plot(fv,db(h2d_f/h2d_f(indMidf)),'-x','Markersize',15,'linewidth',2,'Color',col(4,:))
set(gca,'xscale','log')
box on
xlabel('Frequency factor (-)')
ylabel('Maximum amplitude ratio (dB)')
set(gca,'fontsize',14)


%% Plot - mode shape detuning robustness
indMids = ceil(length(thetav)/2);   % Reference value

figure
hold on
plot([thetav(1),thetav(end)],[0,0],'-','Color',[0.5,0.5,0.5])
plot(thetav,db(h1n_s/h1n_s(1)),'-o','Markersize',10,'linewidth',2,'Color',col(1,:))
plot(thetav,db(h2n_s/h2n_s(1)),'-o','Markersize',10,'linewidth',2,'Color',col(2,:))
plot(thetav,db(h1d_s/h1d_s(1)),'-x','Markersize',15,'linewidth',2,'Color',col(3,:))
plot(thetav,db(h2d_s/h2d_s(1)),'-x','Markersize',15,'linewidth',2,'Color',col(4,:))
box on
xlim([0,pi])
set(gca,'XTick',[0,pi/4,pi/2,3*pi/4,pi])
set(gca,'XTickLabel',{'0','\pi/4','\pi/2','3\pi/4','\pi'})
xlabel('\theta (rad)')
ylabel('Maximum amplitude ratio (dB)')
set(gca,'fontsize',14)