%
% Plot the experimental FRFs of a clamped-free piezoelectric beam with a
% thin lamina controlled by shunts or newtorks.
%
% Results showing the effect of weights on performance.
%
% (c) G. Raze, J. Dietrich, B. Lossouarn and G. Kerschen
% Mar 2022

expFolder = '.\Data\2p\Performance\';       % Data folder


FRFFactor = 1/4.44822;         % Factor for the FRF (there was an error in the sensitivity of a sensor)

% Open-circuit
[froc,hoc] = TestLabFRFData('.\Data\OC2');
hoc = abs(hoc./(2*pi*froc).^2)*FRFFactor;

% Cases (w1/w2 without correction, see readme.txt, experimental)
cases = {'1e-3','3-2e-3','1e-2','3-2e-2','1e-1','1-6e-1','2-5e-1','4e-1',...
  '6-3e-1','1e0','1-6e0','2-5e0','4e0','6-3e0','1e1','3-2e1','1e2',...
  '3-2e2','1e3'};
% Cases (w1/w2 without correction, see readme.txt, theroetical)
casesv = logspace(-3,3,1e2);

S = diag([1,4.814968809985699]);    % Correction factor for the weights



% Frequency range for the peaks
frRange1 = [25,35];
frRange2 = [120,160];

% Weight vector (w1/w2 without correction, see readme.txt)
w =  [logspace(-3,-1,5) , logspace(-1,1,11) , logspace(1,3,5)];
w([5,16]) = [];

% Peaks (open circuit)
h1oc = max(hoc(and(froc>frRange1(1),froc<frRange1(2))));
h2oc = max(hoc(and(froc>frRange2(1),froc<frRange2(2))));

% Modal damping rations (open circuit)
xioc = zeros(2,1);
xioc(1) = halfPowerDamping(hoc(and(froc>frRange1(1),froc<frRange1(2))),2*pi*froc(and(froc>frRange1(1),froc<frRange1(2))));
xioc(2) = halfPowerDamping(hoc(and(froc>frRange2(1),froc<frRange2(2))),2*pi*froc(and(froc>frRange2(1),froc<frRange2(2))));


%% Experimental attenuation

h1n = zeros(size(w));
h2n = zeros(size(w));
h1d = zeros(size(w));
h2d = zeros(size(w));


for ii = 1 : length(cases)
  [fr,h] = TestLabFRFData([expFolder,'Network_w',cases{ii}]);
  hx = abs(h./(2*pi*fr).^2)*FRFFactor;
  
  h1n(ii) = max(hx(and(fr>frRange1(1),fr<frRange1(2))));
  h2n(ii) = max(hx(and(fr>frRange2(1),fr<frRange2(2))));
end

for ii = 1 : length(cases)
  [fr,h] = TestLabFRFData([expFolder,'Shunts_w',cases{ii}]);
  hx = abs(h./(2*pi*fr).^2)*FRFFactor;
  
  h1d(ii) = max(hx(and(fr>frRange1(1),fr<frRange1(2))));
  h2d(ii) = max(hx(and(fr>frRange2(1),fr<frRange2(2))));
end


%% Theoretical attenuation

load('PerformanceCoupling')

Adbn1 = ThomasReduction(Kc1n_th,xioc(1));
Adbn2 = ThomasReduction(Kc2n_th,xioc(2));
Adbd1 = ThomasReduction(Kc1d_th,xioc(1));
Adbd2 = ThomasReduction(Kc2d_th,xioc(2));

%% Plot results
col = get(groot,'defaultAxesColorOrder');

% Update weights (see readme.txt)
casesv = casesv*S(2,2);
w = w*S(2,2);

figure
hold on
plot(w,db(h1oc./h1n),'-o','Markersize',10,'linewidth',2,'Color',col(1,:))
plot(casesv,Adbn1,'--','Markersize',10,'linewidth',2,'Color',0.55*col(1,:))
plot(w,db(h2oc./h2n),'-o','Markersize',10,'linewidth',2,'Color',col(2,:))
plot(casesv,Adbn2,'--','Markersize',10,'linewidth',2,'Color',0.55*col(2,:))
plot(w,db(h1oc./h1d),'-x','Markersize',15,'linewidth',2,'Color',col(3,:))
plot(casesv,Adbd1,'--','Markersize',10,'linewidth',2,'Color',0.55*col(3,:))
plot(w,db(h2oc./h2d),'-x','Markersize',15,'linewidth',2,'Color',col(4,:))
plot(casesv,Adbd2,'--','Markersize',10,'linewidth',2,'Color',0.55*col(4,:))
%xlim([1e-3,1e3])
xlim([1e-2,1e3])
set(gca,'XTick',logspace(-2,3,6))
ylim([0 30])
set(gca,'xscale','log')
box on
xlabel('w_1/w_2 (-)')
ylabel('Attenuation (dB)')
set(gca,'fontsize',14)