#=============================================================
#
# Présentation du package qgisprocess
# 
# Auteur  : Philippe Lejeune
# Version :  2022/10/05
#
#=============================================================


rm(list=ls()) # réinitialisation de la mémoire

# 1. Installation et chargement des librairies --------------------

library(terra)
library(dplyr)
library(sf)
library(remotes)

remotes::install_github("paleolimbot/qgisprocess")
library(qgisprocess)

qgis_configure()

# forcer l'utilisation d'une version QGIS (nécessite de relancer RStudio)
#Sys.setenv(R_QGISPROCESS_PATH = 
#          "C:/Program Files/QGIS 3.16.11/bin/qgis_process-qgis-ltr.bat")

# 2. Chemin vers les données ---------
path0="C:/PL/01_COURS/tthr_2022/R_GIS_03"
path_in="C:/PL/01_COURS/tthr_2022/R_GIS_03/data_in"
path_out="C:/PL/01_COURS/tthr_2022/R_GIS_03/output"

# 3. liste des algorithmes reconnus par qgisprocess -------
df=qgis_algorithms()
names(df)

table(df$provider)

# liste des algo QGIS
df_qgis=df[df$provider=="qgis",]
df_qgis$algorithm

# liste des algo natifs (extension processing)
df_native=df[df$provider=="native",]
df_native$algorithm


# liste des algo liés à GDAL
df_gdal=df[df$provider=="gdal",]
df_gdal$algorithm

# recherche d'algorithme sur base d'un mot clé
df$algorithm[grepl(pattern = "buffer", df$algorithm)]


# 4. Exemple 1 - commande "Tampons multi-anneaux" --------

algo="native:multiringconstantbuffer"
qgis_description(algo)
qgis_show_help(algo)

f_in=paste0(path_in,"/localites.shp")
f_out=paste0(path_out,"/buffers_5x1km.shp")
result = qgis_run_algorithm(
  algorithm=algo,
  INPUT = f_in,
  RINGS = 5,
  DISTANCE = 1000,
  OUTPUT = f_out,
  .quiet = FALSE)

buff=st_read(f_out)
loc=st_read(f_in)
loc
plot(buff$geometry)
plot(loc$geometry,add=T)

# 5. Exemple 2 - Créer un polygone minimum concave --------

df$algorithm[grepl(pattern = "concave", df$algorithm)]

algo="qgis:knearestconcavehull"
qgis_show_help(algo)

f_in=paste0(path_in,"/localites.shp")
f_out=paste0(path_out,"/concave_hull.gpkg")
result = qgis_run_algorithm(
  algorithm=algo,
  INPUT = f_in,
  OUTPUT = f_out,
  .quiet = T)

# Affichage du résultat
hull=st_read(f_out)
plot(hull$geom)
loc=st_read(f_in)
plot(loc$geometry,add=T)

# 6. Exemple 3 - Calculer la pente du terrain (GDAL) --------

algo="gdal:slope"
qgis_show_help(algo)

f_in=paste0(path_in,"/mnt_2m.tif")
f_out=paste0(path_out,"/pente_pct.tif")
result = qgis_run_algorithm(
  algorithm=algo,
  INPUT = f_in,
  BAND = 1,
  AS_PERCENT = TRUE,
  OUTPUT = f_out,
  .quiet = TRUE)

slope=rast(f_out)
plot(slope)

loc=st_read(f_in)
plot(buff$geometry)
plot(loc$geometry,add=T)

# créer 1 couche de classes de pente
rcl = matrix(c(0, 15, 1, 15, 30, 2, 30, Inf, 3), 
             ncol = 3, byrow = TRUE)
rcl
f_out=paste0(path_out,"/classe_pente.tif")
cl_pente = classify(slope, rcl = rcl,filename=f_out,overwrite=T)
plot(cl_pente)

# 7. Exemple 4 - Tamiser un raster (GDAL Sieve) -----------

# appliquer 1 tamisage
algo="gdal:sieve"
qgis_show_help(algo)

f_in=paste0(path_out,"/classe_pente.tif")
f_out=paste0(path_out,"/classe_pente_tam20.tif")
result = qgis_run_algorithm(
  algorithm=algo,
  GRD_DEM = f_in,
  THRESHOLD = 20,
  OUTPUT = f_out,
  .quiet = TRUE)

# 8. Viewshed (extension WhiteboxTools) ---------------------

df_wbt=df[df$provider=="wbt",]
df_wbt$algorithm

algo="wbt:Viewshed"
qgis_show_help(algo)

f_pdv=paste0(path_in,"/point_de_vue.gpkg")
f_mnt=paste0(path_in,"/mnt_2m.tif")
f_out=paste0(path_out,"/viewshed.tif")

result <- qgis_run_algorithm(
  algo,
  dem = f_mnt,
  stations = f_pdv,
  height = 1.6,
  output = f_out,
  .quiet = TRUE)

mnt=rast(f_mnt)
plot(mnt)
plot(point$geometry,add=T,col="red",lwd=5)

vsh=rast(f_out)
point=st_read(f_pdv)
screen(2) ; plot(vsh)
plot(point$geometry,add=T,col="red",lwd=5)

# 9. Fonctions hydrologiques de l'extension Whitetoolbox --------------------

# Input and output data
f_in=paste0(path_in,"/mnt_2m.tif") # MNT
f_exutoire=paste0(path_in,"/exutoire.shp") # exutoire du BV
f_fill=paste0(path_out,"/mnt_fill.tif") # MNT sans dépression
f_fl_dir=paste0(path_out,"/fl_dir.tif") # Direction d'écoulement
f_fl_acc=paste0(path_out,"/fl_acc.tif") # Acculumation d'écoulement
f_watershed=paste0(path_out,"/watershed.tif") # bassin versant
f_str=paste0(path_out,"/stream.tif") # axes d'écoulement
f_str_vect=paste0(path_out,"/stream.shp") # axes d'écoulement vectoriels

# Délimitation d'un bassin versant

# step 1 : Fill depressions (remplissage des dépressions)

algo="wbt:FillDepressionsPlanchonAndDarboux"
result <- qgis_run_algorithm(
  algorithm="wbt:FillDepressionsPlanchonAndDarboux",
  dem = f_in,
  fix_flats= 1,
  output = f_fill,
  .quiet = TRUE)

# step 2 : D8 pointer (direction d'écoulement)
result <- qgis_run_algorithm(
  algorithm="wbt:D8Pointer",
  dem = f_fill,
  output = f_fl_dir,
  .quiet = TRUE)

# step 3 : Watreshed (délimitation du bassin versant)

result <- qgis_run_algorithm(
  algorithm="wbt:Watershed",
  d8_pntr= f_fl_dir,
  pour_pts = f_exutoire,
  output = f_watershed,
  .quiet = TRUE)

r=rast(f_watershed)
plot(r)
exut=st_read(f_exutoire)
plot(exut,add=T,col="red",lwd=5)

# Extraction des axes d'écoulement

# step 1 : Accumulation d'écoulement
result <- qgis_run_algorithm(
  algorithm="wbt:D8FlowAccumulation",
  input = f_fill,
  output = f_fl_acc,
  .quiet = TRUE)

# step 2 : Extraction des axes d'écoulement
result <- qgis_run_algorithm(
  algorithm="wbt:ExtractStreams",
  flow_accum = f_fl_acc,
  threshold = 50000,
  output = f_str,
  .quiet = TRUE)

# step 3 : Vectorisation des axes d'écoulement
qgis_show_help("wbt:RasterStreamsToVector")
result <- qgis_run_algorithm(
  algorithm="wbt:RasterStreamsToVector",
  streams = f_str,
  d8_pntr = f_fl_dir,
  output = f_str_vect,
  .quiet = TRUE)

# affichage du résulat
str=st_read(f_str_vect)
plot(str$geometry,add=T)
