%
% Damping of a plate
% (c) G. Raze, J. Dietrich, B. Lossouarn and G. Kerschen
% Nov 2020
%
% Comparison with Giorgio et al 2008
% 
% See
%
% Giorgio, I., Culla, A. and Del Vescovo, D. (2009) 
% Multimode vibration control using several piezoelectric transducers shunted with a multiterminal network,
% Archive of Applied Mechanics, 79(9), pp. 859879. doi: 10.1007/s00419-008-0258-x.
%

load('PlateModel')


N = length(platess.A)/2;

% System matrices
Om2 = -platess.A(N+1:end,1:N);
Om = sqrt(Om2);
ZOm = -platess.A(N+1:end,N+1:end);
Gam = platess.B(N+1:end,2:end);
Cp = -platess.D(2:end,2:end);



%% Giorgio et al 2008
Cp = 0.5*(Cp+Cp');
[V,D] = eig(Cp);
sqCpepsth = V*sqrt(D)*(V.');
Gampar = Om\Gam/sqCpepsth;                              % Giorgio et al 2008 coupling matrix       
[Bt,Gt,U] = giorgio2008(Gampar(1:5,:),Om(1:5,1:5));     % Giorgio et al 2008 method, with the first five modes

% Physical reluctance and conductance matrices
Bg = sqCpepsth*Bt*sqCpepsth;
Gg = sqCpepsth*Gt*sqCpepsth;
Bg = 0.5*(Bg+Bg.');
Gg = 0.5*(Gg+Gg.');

% State-space model
Ang = -Gg\Bg;
Bng = Gg\eye(5);
Cng = eye(5);
Dng = zeros(5);

% Controlled system
sysng = tf('s')*ss(Ang,Bng,Cng,Dng)*tf('s');
platecg = feedback(platess,-sysng,2:6,2:6);


%% Modal-based synthesis

Ns = 5;
modes = 1:1:Ns;
d = ones(Ns,1);
[C,CN,G,B,Phie,Om2e,ZOme,Ep] = networkTuning(modes,d,Cp,Gam,Om2);

Np = length(Cp);
An = [zeros(size(C)) , eye(size(C)) ; -C\B , -C\G];
Bn = [zeros(size(C)) ; C\Ep];
Cn = [zeros(size(C)) , Ep.'];
Dn = zeros(Np,Np);

sysn = tf('s')*ss(An,Bn,Cn,Dn);
platessNC = ss(platess.A,platess.B,platess.C,zeros(size(platess.D)));   % Model without piezoelectric capacitance (incorporated in the network)
platecn = feedback(platessNC,-sysn,2:6,2:6);







%% FRFs
wv = linspace(2*pi*100,2*pi*1000,10000);
h0 = squeeze(freqresp(tf('s')*platess(1,1),wv));
hc = squeeze(freqresp(tf('s')*platecg(1,1),wv));
hcn = squeeze(freqresp(tf('s')*platecn(1,1),wv));
figure
hold on
plot(wv/2/pi,db(abs(h0)),'-k','linewidth',1)
plot(wv/2/pi,db(abs(hc)),'-','linewidth',2)
plot(wv/2/pi,db(abs(hcn)),'-.','linewidth',2)
xlim([100,1000])
box on
set(gca,'fontsize',14)
xlabel('Frequency (Hz)')
ylabel('Velocity FRF (dB, ref. m/s/N)')
