%
% Compute the electrical matrices of a network for piezoelectric damping
% (c) G. Raze, J. Dietrich, B. Lossouarn and G. Kerschen
% Nov 2020
%
% Tuning: see
% Yamada, K. et al. (2010) 
% Optimum tuning of series and parallel LR circuits for passive vibration suppression using piezoelectric elements,
% Journal of Sound and Vibration. Elsevier, 329(24), pp. 50365057. doi: 10.1016/j.jsv.2010.06.021.
% 
%
% Input:
%   * modes: list of targeted modes
%   * d: associated weights (recommended: d = ones(size(modes)) )
%   * Cp: piezoelectric capacitance matrix
%   * PhiTGamma = Phi.'*Gamma: modal piezoelectric coupling matrix
%   * Om2: diagonal matrix with squared structural frequencies
%
% Output:
%   * C: capacitance matrix (overall network)
%   * CN: capacitance matrix (interconnecting network)
%   * G: conductance matrix
%   * B: reluctance matrix
%   * Phie: electrical mode shapes
%   * Om2e: diagonal matrix with squared electrical frequencies
%   * ZOme: diagonal modal damping matrix (2*xi*omega)
%   * Ep: piezoelectric localization matrix
%   

function [C,CN,G,B,Phie,Om2e,ZOme,Ep] = networkTuning(modes,d,Cp,PhiTGamma,Om2)

  P = size(PhiTGamma,2);
  Ns = length(modes);
  
  chiPOpt = zeros(P,Ns);
  
  [Vc,Dc] = eig(Cp);
  sqrtCp = Vc*sqrt(Dc)*(Vc.');
  
  % Electrical mode shapes
  for ii = 1 : Ns
    Gi = PhiTGamma(modes(ii),:);
    chiPOpt(:,ii) = 1/sqrt(Gi*(Cp\(Gi.')))*(sqrtCp\(Gi.'));
  end
  
  Dpb = diag(d);
  lMax = max(eig(chiPOpt*(Dpb^2)*(chiPOpt.')));
  Dp = 1/sqrt(lMax)*Dpb;
  
  Phip = sqrtCp\(chiPOpt*Dp);
  
  % Electrical frequencies and damping ratios: Yamada et al 2010
  we = zeros(Ns,1);
  zetae = zeros(Ns,1);
  for ii = 1 : Ns
    Kcrk2 = (PhiTGamma(modes(ii),:)*Phip(:,ii))^2/Om2(modes(ii),modes(ii));
    we(ii) = sqrt((2-Kcrk2)*Om2(modes(ii),modes(ii))/2);
    zetae(ii) = sqrt(3)/2*sqrt(Kcrk2/(2-Kcrk2));
  end
  
  
% Additional internal degrees of freedom
  if rank(Phip) == Ns
    Phie = Phip;
    Ne = P;
    Ep = eye(P);
  else
    W = null(Phip);
    Phie = [Phip ; W.'];
    Ne = size(Phie,1);
    Ep = eye(Ne,P);
  end
  
  % Electrical matrices
  if rank(Phie) == P
    
    PhiePI = eye(Ne)/Phie;
    Gsq = diag(sqrt(zetae.*we))/Phie;
    Bsq = diag(we)/Phie;
    
    C = PhiePI.'*PhiePI;
    G = 2*(Gsq.'*Gsq);
    B = Bsq.'*Bsq;
  else
    V = null(Phie.');
    Dv = V.'*Cp*V;
    
    PhiePI = pinv(Phie);
    CV = V*Dv*V.';
    Gsq = diag(sqrt(zetae.*we))*PhiePI;
    Bsq = diag(we)*PhiePI;
    
    C = PhiePI.'*PhiePI + 0.5*(CV+CV.');
    G = 2*(Gsq.'*Gsq);
    B = Bsq.'*Bsq;
  end
  
  
  Om2e = diag(we.^2);
  ZOme = diag(2*zetae.*we);
  CN = C - Ep*Cp*(Ep.');
end