%
% Giorgio et al 2008 method 
%
% See
%
% Giorgio, I., Culla, A. and Del Vescovo, D. (2009) 
% Multimode vibration control using several piezoelectric transducers shunted with a multiterminal network,
% Archive of Applied Mechanics, 79(9), pp. 859879. doi: 10.1007/s00419-008-0258-x.
%
% and
% 
% Wu, S. (1996) 
% Piezoelectric shunts with a parallel R-L circuit for structural damping and vibration control,
% in Johnson, C. D. (ed.) Spie, pp. 259269. doi: 10.1117/12.239093.
%
% Input:
%   * Gamma: piezoelectric coupling matrix to be diagonalized
%   * Om: Resonance frequencies matrix
%
% Output:
%   * B: reluctance matrix
%   * G: conductance matrix
%   * U: transformation matrix
function [B,G,U] = giorgio2008(Gamma,Om)

  % Initial guess polar decomposition
  [U,S,V] = svd(Gamma.');
  P = V*S*(V');
  U0 = U*(V');

  u0 = U0(:);

  uOpt = fsolve(@(u) objFun(u,Gamma),u0,...
    optimoptions(@fsolve,'FunctionTolerance',1e-12,'OptimalityTolerance',1e-9,...
    'StepTolerance',1e-9,'MaxFunctionEvaluations',1e4));


  U = reshape(uOpt,size(Gamma));

  disp(['Initial cost function norm: ',num2str(norm(objFun(u0,Gamma)))])
  disp(['Optimal cost function norm: ',num2str(norm(objFun(uOpt,Gamma)))])
  disp(['Initial Frobenius norm: ',num2str(norm(diag(Gamma*U0),'fro'))])
  disp(['Optimal Frobenius norm: ',num2str(norm(diag(Gamma*U),'fro'))])


  GU = Gamma*U;

  % Wu 1996 tuning
  Lm = 1./((1-diag(GU).^2/2).*diag(Om).^2);
  Rm = 1./(sqrt(2)*abs(diag(GU)).*diag(Om));


  B = U*diag(1./Lm)*(U.');
  G = U*diag(1./Rm)*(U.');

  function f = objFun(u,Gamma)

    % Orthogonal transformation matrix
    UU = reshape(u,size(Gamma));

    % Transformed coupling matrix and its diagonal part
    GUU = Gamma*UU;
    DGU = diag(diag(GUU));

    % Symmetric system of equations
    F1 = (GUU.')*DGU - DGU*GUU;
    F2 = UU*(UU.') - eye(size(UU));

    % Non-redundant system of equations
    F = triu(F1,1) + tril(F2);
    f = F(:);
  end
end