%
% Damping of a beam
% (c) G. Raze, J. Dietrich, B. Lossouarn and G. Kerschen
% Nov 2020
%
% Comparison with Lossouarn et al 2015
%
% See
% 
% Lossouarn, B., De, J. F. and Aucejo, M. (2015) 
% Multimodal vibration damping of a beam with a periodic array of piezoelectric patches connected to a passive electrical network,
% Smart Materials and Structures. IOP Publishing, 24(11), p. 115037. doi: 10.1088/0964-1726/24/11/115037.

load('beamModel')
N = length(beamss.A)/2;

% System matrices
Om2 = -beamss.A(N+1:end,1:N);
Om = sqrt(Om2);
ZOm = -beamss.A(N+1:end,N+1:end);
Gam = beamss.B(N+1:end,2:end);
Cp = -beamss.D(2:end,2:end);


%% Lossouarn 2015
% Cell characteristics
Ncells = 20;
al = 1;
Rl = 57.5;
Ll = 161.105e-3;
Cpl = Cp(1,1);
[Bl,Gl,Cl] = beamAnalogModel(al/2,Rl,Ll,Cpl,0,Ncells);

% Grounded ports to mimic free-free conditions
indC = [21,22,length(Cl)-1,length(Cl)];
Bl(indC,:) = [];
Bl(:,indC) = [];
Gl(indC,:) = [];
Gl(:,indC) = [];
Cl(indC,:) = [];
Cl(:,indC) = [];

Epl = [eye(Ncells) ; zeros(length(Cl)-Ncells,Ncells)];
Np = length(Cp);
indP = 1:Np;
indI = (Np+1):length(Cl);
Ni = length(indI);

% State space model
Assl = -[Cl(indP,indP) , Gl(indP,indI) ; zeros(Ni,Np) , Gl(indI,indI)]\[Gl(indP,indP) , zeros(Np,Ni) ; Gl(indI,indP) , Bl(indI,indI)];
Bssl = [Cl(indP,indP) , Gl(indP,indI) ; zeros(Ni,Np) , Gl(indI,indI)]\Epl;
Cssl = Epl.';
Dssl = zeros(Np,Np);

sysl = ss(Assl,Bssl,Cssl,Dssl)*tf('s');
beamNC = ss(beamss.A,beamss.B,beamss.C,zeros(size(beamss.D)));   % Model without piezoelectric capacitance (incorporated in the network)
beamcl = feedback(beamNC,-sysl,2:21,2:21);



%% Modal-based synthesis
modes = 4:1:7;
Ns = length(modes);
d = ones(Ns,1);

[C,CN,G,B,Phie,Om2e,ZOme,Ep] = networkTuning(modes,d,Cp,Gam,Om2);

An = [zeros(size(C)) , eye(size(C)) ; -C\B , -C\G];
Bn = [zeros(size(C)) ; C\Ep];
Cn = [zeros(size(C)) , Ep.'];
Dn = zeros(Np,Np);

sysn = tf('s')*ss(An,Bn,Cn,Dn);
beamcn = feedback(beamNC,-sysn,2:21,2:21);


%% FRF
wv = linspace(2*pi*60,2*pi*1200,1000);
h0 = squeeze(freqresp(tf('s')*beamss(1,1),wv));
hcl = squeeze(freqresp(tf('s')*beamcl(1,1),wv));
hcn = squeeze(freqresp(tf('s')*beamcn(1,1),wv));

figure
hold on
plot(wv/2/pi,db(abs(h0)),'-k','linewidth',1)
plot(wv/2/pi,db(abs(hcl)),'-','linewidth',2)
plot(wv/2/pi,db(abs(hcn)),'-','linewidth',2)
box on
set(gca,'fontsize',14)
xlabel('Frequency (Hz)')
ylabel('Velocity FRF (dB, ref. m/s/N)')
set(gca,'xscale','log')
xlim([60,max(wv/2/pi)])
ylim([-60,-10])

