%
% Electrical matrices of the beam electrical analog network
% (c) G. Raze, J. Dietrich, B. Lossouarn and G. Kerschen
% Nov 2020
%
% Input:
%   * a: transformer ratio
%   * R: transformer resistnace
%   * L: inductance
%   * Cp: piezoelectric capacitance at constant strain
%   * C0: capacitance for regularization
%	* N: number of cells
%
% Output:
%   * B: reluctance matrix 
%   * G: conductance matrix
%   * C: capacitance matrix
%   
function [B,G,C] = beamAnalogModel(a,R,L,Cp,C0,N)


  [Be,Ge,Ce] = cellAdmittance(a,R,L,Cp,C0);
  
  B = zeros(3*(N-1)+5);
  G = zeros(3*(N-1)+5);
  C = zeros(3*(N-1)+5);

  % Assembly
  for ii = 1 : N
    ind = [1,2,3,4,5] + (ii-1)*3;
    B(ind,ind) = B(ind,ind) + Be;
    G(ind,ind) = G(ind,ind) + Ge;
    C(ind,ind) = C(ind,ind) + Ce;
  end

  % Dofs reordering
  indP = 3:3:length(B);
  indI = setdiff(1:1:length(B),indP);
  
  B = B([indP,indI],[indP,indI]);
  G = G([indP,indI],[indP,indI]);
  C = C([indP,indI],[indP,indI]);
  
  % Reluctance, conductance and capacitance matrices of the cell
  function [Be,Ge,Ce] = cellAdmittance(a,R,L,Cp,C0)
    GT = [1/R     -1/R      -a/R;
          -1/R    1/R       a/R;
          -a/R    a/R       a^2/R];
    GT2 = [1/R     -1/R      a/R;
          -1/R    1/R       -a/R;
          a/R    -a/R       a^2/R];

    Be  = zeros(5);
    Be([1,4],[1,4]) = 1/L*[1,-1;-1,1];

    Ge = zeros(5);
    Ge([2,3,1],[2,3,1]) = GT;
    Ge([5,3,4],[5,3,4]) = Ge([5,3,4],[5,3,4]) + GT2;

    Ce = zeros(5);
    Ce(3,3) = Cp;
    Ce(2,2) = C0;
    Ce(5,5) = C0;
  end

end