% Tuning of a Current Flowing shunt
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Mar 2022
%
% Input:
%   * ri: residues vector of length Ns+1
%   * zi: zeros of Thevenin's impedance (vector of length Ns)
%   * zetai: electrical damping ratio (vector of length Ns)
%   * Cp: piezoelectric capacitance at constant strain
% 
% Output:
%   * Y: CF shunt admittance
%   * L: shunt inductance vector of length Ns
%   * R: shunt resistance vector of length Ns
%   * C: shunt capacitance vector of length Ns
%
function [Y,L,R,C] = CFTuning(ri,zi,zetai,Cp)

  % Poles of the Norton equivalent admittance
  Nnorm = round(length(zi)/2);
  ri = ri(:);
  MP = zeros(length(zi));
  pol0 = ri(1);
  for ii = 1 : length(zi)
    pol = 1;
    for jj = 1 : length(zi)
      if jj ~= ii
        pol = conv(pol,[1,(zi(jj)/zi(Nnorm)).^2]);
      end
    end
    pol0 = conv(pol0,[1,(zi(ii)/zi(Nnorm)).^2]);
    MP(ii,:) = pol;
  end
  pol = ri(2:end).'*MP;
  rr = roots([pol,0]+pol0);
  pi = sort(sqrt(-rr))*zi(Nnorm);
  
  
  % Undamped circuit tuning
  L = ones(size(zi))/Cp;
  
  for ii = 1 : length(L)
    indni = setdiff(1:1:length(pi),ii);
    
    L(indni) = L(indni).*(pi(ii)^2-pi(indni).^2)./(zi(ii)^2-pi(indni).^2);
    L(ii) = L(ii)./(zi(ii)^2-pi(ii).^2);
     
  end
  C = 1./(L.*pi.^2);
  

  % Resistances tuning
  EE = diag(1./C) + 1/Cp*ones(length(L));
  LL = diag(L);
  [PhiE,Om2E] = eig(EE,LL);
  R = ((PhiE.'.^2)\(2*zetai.*zi));

  
  if any(R<0)
    warning('There is at least one negative resistance in the circuit')
  end
  
  Y = admittance(L,R,C);
  
  % Admittance 
  function Y = admittance(L,R,C)
    s = zpk(0,[],1);
    Y = 0;
    for nn = 1 : length(L)
      Y = Y + s/L(nn)/(s^2 + R(nn)/L(nn)*s + 1/(C(nn)*L(nn)));
    end
  end
  
end