% Change the electrical input for state-space models of piezoelectric
% structures by its output (from voltage to charge and vice versa).
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Mar 2022
%
% Input:
%   * sys: state-space model
%   * indChange: index of inputs (and outputs) to change (NB: the indices
%   must be identical for inputs and outputs, e.g., index i for charge
%   input corresponds to index i for charge output)
% 
% Output:
%   * sys2: modified state-space model
%   * A2,B2,C2,D2: modified state-space matrices
%
function [sys2,A2,B2,C2,D2] = changeElectricalStateSpace(sys,indChange)

  % Original system
  [A,B,C,D] = ssdata(sys);
  indKeepIn = setdiff(1:1:size(B,2),indChange);
  indKeepOut = setdiff(1:1:size(C,1),indChange);
  
  % Inverse capacitance/elastance matrix
  Dcci = D(indChange,indChange)\eye(length(indChange));
  
  % Modified state-space matrices
  A2 = A - B(:,indChange)*Dcci*C(indChange,:);
  B2(:,[indKeepIn,indChange]) = [B(:,indKeepIn) - B(:,indChange)*Dcci*D(indChange,indKeepIn) , B(:,indChange)*Dcci];
  C2([indKeepOut,indChange],:) = [C(indKeepOut,:) - D(indKeepOut,indChange)*Dcci*C(indChange,:) ; -Dcci*C(indChange,:)];
  D2([indKeepOut,indChange],[indKeepIn,indChange]) = ...
    [D(indKeepOut,indKeepIn)-D(indKeepOut,indChange)*Dcci*D(indChange,indKeepIn), D(indKeepIn,indChange)*Dcci ;
    -Dcci*D(indChange,indKeepIn)                                        , Dcci];
  
  % Modified state-space model
  sys2 = ss(A2,B2,C2,D2);
  
  for ii = 1 : length(indChange)
    sys2.InputName{indChange(ii)} = sys.OutputName{indChange(ii)};
    sys2.OutputName{indChange(ii)} = sys.InputName{indChange(ii)};
  end
  
  for ii = 1 : length(indKeepIn)
    sys2.InputName{indKeepIn(ii)} = sys.InputName{indKeepIn(ii)};
  end
  
  for ii = 1 : length(indKeepOut)
    sys2.OutputName{indKeepOut(ii)} = sys.OutputName{indKeepOut(ii)};
  end
end