%
% Plot of the FRF of multi-resonant shunts (Hollkamp's, Second Foster
% Canonical Form, Current Flowing and Series-Parallel Impedance Structure)
% controlling a clamped-free piezoelectric beam.
%
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Mar 2022


addpath('IdealCircuitsCodes\')
addpath('CircuitsTuning\')

%% Beam model

load('BeamModel');                                  % Beam with open-circuited patch
beamssSC = changeElectricalStateSpace(beamss,2);    % Beam with short-circuited patch

% Dynamic elastance/capacitance identification
[wsc,woc,Cpe,r] = dynamicCapacitanceResidues(1/beamss(2,2));
[woc2,wsc2,Epe,r2] = dynamicCapacitanceResidues(beamss(2,2));

% Laplace variable
s = zpk(0,[],1);

% Plots
xl = [10;250];                        % X axis limits (Hz)
yl = [-150,-70];                      % Y axis limits (dB)
w = linspace(10*2*pi,2*pi*250,1e4);   % Frequency vector (rad/s)

% Controlled modes
Nmodes = 2;
modes = 1:1:Nmodes;

% Residues
r1 = [0.9,0.5,0.1];       % Hollkamp's and SFCF shunts 
r0 = 0.1;
r01 = [0.8,0.45,0.2];     % CF and SPIS shunts 

% FRFs
m = bode(beamss(1,1),w);        % Open-circuit



%% Impedance-based approach


% Hollkamp's shunt - FRF
figure
hold on
plot(w/2/pi,db(squeeze(m)),'-k')
for ii = 1:length(r1)
  % Residues
  ri =[r1(ii),1-r1(ii)];
  ri = ri/sum(ri);
  
  % Specification procedure
  [zi,zetai,wsch,woch] = impedanceSCOCFrequencies(woc2,-r2/Epe,modes,ri,'series');
  
  % Circuit tuning
  [Ych,Lch,Rch,Cch] = HollkampTuning(ri,zi,zetai,Cpe);

  % Controlled system FRF
  beamssch = feedback(beamss,-minreal(Ych/s),2,2);
  mch = bode(beamssch(1,1),w);
  plot(w/2/pi,db(squeeze(mch)),'-','linewidth',2)

  if ii == 3
    save('HollkampParameters','Cch','Lch','Rch','beamss');
  end
end
box on
set(gca,'fontsize',14)
set(gca,'xscale','log')
xlabel('Frequency (Hz)')
ylabel('Receptance (dB, ref. m/N)')
xlim(xl)
ylim(yl)
set(gca,'XTick',[5e1,1e2,2e2])



% Current flowing shunt - FRF
figure
hold on
plot(w/2/pi,db(squeeze(m)),'-k')
for ii = 1:length(r01)
  % Residues
  ri =[r0,r01(ii),1-r0-r01(ii)];
  ri = ri/sum(ri);

  % Specification procedure
  [zi,zetai,wsch,woch] = impedanceSCOCFrequencies(woc2,-r2/Epe,modes,ri,'series');
  
  % Circuit tuning
  [Ych,Lch,Rch,Cch] = CFTuning(ri,zi,zetai,Cpe);

  % Controlled system FRF
  beamssch = feedback(beamss,-minreal(Ych/s),2,2);
  mch = bode(beamssch(1,1),w);
  plot(w/2/pi,db(squeeze(mch)),'-','linewidth',2)
end
box on
set(gca,'fontsize',14)
set(gca,'xscale','log')
xlabel('Frequency (Hz)')
ylabel('Receptance (dB, ref. m/N)')
xlim(xl)
ylim(yl)
set(gca,'XTick',[5e1,1e2,2e2])

%% Admittance-based approaches

% Second Foster Canonical Form shunt - FRF
figure
hold on
plot(w/2/pi,db(squeeze(m)),'-k')
for ii = 1:length(r1)
  % Residues
  ri =[r1(ii),1-r1(ii)];
  ri = ri/sum(ri);

  % Specification procedure
  [zi,zetai,wsch,woch] = impedanceSCOCFrequencies(woc2,-r2/Epe,modes,ri,'parallel');
  
  % Circuit tuning
  [Zch,Lch,Rch,Cch] = SFCFTuning(ri,zi,zetai,Cpe);

  % Controlled system FRF
  beamssch = feedback(beamss,-minreal(1/(s*Zch)),2,2);
  mch = bode(beamssch(1,1),w);
  plot(w/2/pi,db(squeeze(mch)),'-','linewidth',2)
end
box on
set(gca,'fontsize',14)
set(gca,'xscale','log')
xlabel('Frequency (Hz)')
ylabel('Receptance (dB, ref. m/N)')
xlim(xl)
ylim(yl)
set(gca,'XTick',[5e1,1e2,2e2])


% Series-Parallel Impedance Structure - FRF
figure
hold on
plot(w/2/pi,db(squeeze(m)),'-k')
for ii = 1:length(r01)
  % Residues
  ri =[r0,r01(ii),1-r0-r01(ii)];
  ri = ri/sum(ri);

  % Specification procedure
  [zi,zetai,wsch,woch] = impedanceSCOCFrequencies(woc2,-r2/Epe,modes,ri,'parallel');
  
  % Circuit tuning
  [Zch,Lch,Rch,Cch] = SPISTuning(ri,zi,zetai,Cpe);

  % Controlled system FRF
  beamssch = feedback(beamss,-minreal(1/(Zch*s)),2,2);
  mch = bode(beamssch(1,1),w);
  plot(w/2/pi,db(squeeze(mch)),'-','linewidth',2)
end
box on
set(gca,'fontsize',14)
set(gca,'xscale','log')
xlabel('Frequency (Hz)')
ylabel('Receptance (dB, ref. m/N)')
xlim(xl)
ylim(yl)
set(gca,'XTick',[5e1,1e2,2e2])
