%
% Characteristics of an ideal Thevenin impedance for multimodal
% piezoelectric shunt damping
%
% Input:
%   * woc: open-circuit resonance frequencies
%   * tpc: modal coupling vector (Phi_oc^T*theta_p)
%   * modes: modes indices to be controlled
%   * r: residues associated with the targeted modes
%   * shuntType: baseline case 'series' or 'parallel'
%
% Output:
%   * z: zeros frequencies
%   * zetai: electrical damping ratios
%   * wsch: effective short-circuit resonance frequencies
%   * woch: effective open-circuit resonance frequencies
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Sep 2021
%
function [z,zetai,wsch,woch] = impedanceSCOCFrequencies(woc,tpc,modes,r,...
  shuntType)

  woc = woc(:);
  tpc = tpc(:);
  modes = modes(:);
  r = r(:);

  if length(r) > length(modes)
    r0 = r(1);
    r = r(2:end);
  else
    r0 = 0;
  end
  
  % Linearized approximation
  wsch = zeros(size(modes));
  woch = zeros(size(modes));
  z = zeros(size(modes));
  zetai = zeros(size(z));
  
  for ii = 1 : 1 : length(modes)
    indMechSup = (modes(ii)+1) : length(tpc);
    indInf = 1:(ii-1);
    indSupOC = (ii+1):length(modes);
    indSupSC = ii:length(modes);
    
    % Residual contributions
    kr = tpc(modes(ii))/woc(modes(ii))^2;
    knr = sum(tpc(indMechSup)./woc(indMechSup).^2);
    fl = sum(r(indInf).*(z(indInf).^2)/woc(modes(ii))^2);
    fhOC = r0 + sum(r(indSupOC));
    fhSC = r0 + sum(r(indSupSC));
    
    % Quadratic problem coefficients
    aOC = 1 - knr*fhOC;
    aSC = 1 - knr*fhSC;
    bOC = 1 - (kr+knr)*fhOC - knr*fl;
    bSC = 1 - (kr+knr)*fhSC - knr*fl;
    c = -(kr+knr)*fl;
    
    % Effective SC and OC frequencies
    wsch(ii) = woc(modes(ii))*sqrt(bSC/(2*aSC) + sqrt((bSC/(2*aSC))^2-c/aSC));
    woch(ii) = woc(modes(ii))*sqrt(bOC/(2*aOC) + sqrt((bOC/(2*aOC))^2-c/aOC));
    
    % Zero of the undamped admittance
    [~,~,zetai(ii),delta] = ikegameRLShunt(1,wsch(ii),woch(ii),shuntType);
    if strcmp(shuntType,'series')
      we = delta*woch(ii);
    else
      we = delta*wsch(ii);
    end
    z(ii) = we/sqrt(1-r(ii)/(1/knr + fl*woch(ii)^2/we^2 -fhOC));
    
  end
  
end