% Tuning of a Series-Parallel Impedance Structure
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Mar 2022
%
% Input:
%   * ri: residues vector of length Ns+1
%   * zi: zeros of Norton's admittance (vector of length Ns)
%   * zetai: electrical damping ratio (vector of length Ns)
%   * Cp: piezoelectric capacitance at constant strain
% 
% Output:
%   * Z: SPIS shunt impedance
%   * L: shunt inductance vector of length Ns
%   * R: shunt resistance vector of length Ns
%   * C: shunt capacitance vector of length Ns
%
function [Z,L,R,C] = SPISTuning(ri,zi,zetai,Cp)

  % Poles of the Thevenin equivalent impedance
  Nnorm = round(length(zi)/2);
  ri = ri(:);
  MP = zeros(length(zi));
  pol0 = ri(1);
  for ii = 1 : length(zi)
    pol = 1;
    for jj = 1 : length(zi)
      if jj ~= ii
        pol = conv(pol,[(zi(Nnorm)/zi(jj)).^2,1]);
      end
    end
    MP(ii,:) = pol;
    pol0 = conv(pol0,[(zi(Nnorm)/zi(ii)).^2,1]);
  end
  pol = ri(2:end).'*MP;
  rr = roots([0,pol]+pol0);
  pi = sort(sqrt(-rr))*zi(Nnorm);
  
  
  % Undamped circuit tuning
  L = -1./(Cp*(pi.^2));
  for ii = 1 : length(L)
    indni = setdiff(1:1:length(pi),ii);
    L(indni) = L(indni).*(1-(pi(indni)/zi(ii)).^2)./(1-(pi(indni)/pi(ii)).^2);
    L(ii) = L(ii).*(1 - (pi(ii)/zi(ii))^2);
  end
  
  C = 1./(L.*pi.^2);
  
  % Resistances tuning
  CC = diag(C) + Cp*ones(size(L));
  LL = diag(1./L);
  [PhiE,Om2E] = eig(LL,CC);
  R = 1./((PhiE.'.^2)\(2*zetai.*zi));
  
  
  if any(R<0)
    warning('There is at least one negative resistance in the circuit')
  end
  
  % Admittance 
  s = zpk(0,[],1);
  Z = 0;
  for ii = 1 : length(L)
    Z = Z + 1/C(ii)*s/(s^2 + s/(R(ii)*C(ii)) + 1/(C(ii)*L(ii)));
  end
end