% Tuning of a Current Blocking shunt with parallel RL shunts
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Mar 2022
%
% Input:
%   * ri: residues vector of length Ns
%   * zi: zeros of Norton's admittance (vector of length Ns)
%   * zetai: electrical damping ratio (vector of length Ns)
%   * Cp: piezoelectric capacitance at constant strain
% 
% Output:
%   * Z: CB shunt impedance
%   * L: shunt inductance vector of length Ns
%   * R: shunt resistance vector of length Ns
%   * Cf: filter capacitance vector of length Ns-1
%   * Lf: filter inductance vector of length Ns-1
%
function [Z,L,R,Cf,Lf] = CBPTuning(ri,zi,zetai,Cp)

  % Filter characteristics
  Cf = zeros(length(ri)-1,1);
  for ii = 1 : length(Cf)
    Cti = sum(1./Cf(1:(ii-1)));
    rit = sum(ri(1:ii));
    Cf(ii) = (1 - rit)/(rit*(1/Cp+Cti) - Cti);
  end
  Lf = 1./(Cf.*zi(1:end-1).^2);
  
    
  % Shunt branches tuning
  L = zeros(length(zi),1);
  R = zeros(length(zi),1);
  for ii = 1 : length(L)
    % Damped circuit
    sz = 1i*zi(ii);
    H = [1 , 0 ; sz*Cp , 1];
    
    for jj = 1 : ii-1
      % Damped circuit
      Ys = 1/(L(jj)*sz) + 1/R(jj);
      Zf = sz/Cf(jj)/(zi(jj)^2 + sz^2);
      H = H*[1, Zf ; Ys , 1+Zf*Ys];
    end
    
    % Ideal admittance
    Yn = 0;
    for jj = 1 : length(ri)
      Yn = Yn + ri(jj)*sz^2/(zi(jj)^2 + 2*zetai(jj)*zi(jj)*sz + sz^2);
    end
    Yn = sz*Cp/Yn;
    Yk = ( H(2,1) - H(1,1)* Yn )/( H(1,2)* Yn - H(2,2) );
    
    % Inductance tuning
    L(ii) = -1/(zi(ii)*imag(Yk));
    
    % Resistance tuning
    R(ii) = 1/real(Yk);
    
  end
  
  % Impedance 
  s = zpk(0,[],1);
  Z = 1/(1/(L(end)*s) + 1/R(end));
  for ii = (length(L)-1) : -1 : 1
    Ys = 1/(L(ii)*s) + 1/R(ii);
    Zf = s/Cf(ii)/(s^2 + 1/(Lf(ii)*Cf(ii)));
    Z = 1/( Ys + 1/( Zf + Z));
  end

end