%
% Plot of the FRF of multi-resonant shunts (Current Flowing and
% Series-Parallel Impedance Structure) controlling a clamped-free
% piezoelectric beam.
%
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Mar 2022

plotParameters = false;       % Set to true to plot electrical parameters as functions of r0 (warning: long computation)

addpath('IdealCircuitsCodes\')
addpath('CircuitsTuning\')

%% Beam model

load('BeamModel');                                  % Beam with open-circuited patch
beamssSC = changeElectricalStateSpace(beamss,2);    % Beam with short-circuited patch

% Dynamic elastance/capacitance identification
[wsc,woc,Cpe,r] = dynamicCapacitanceResidues(1/beamss(2,2));
[woc2,wsc2,Epe,r2] = dynamicCapacitanceResidues(beamss(2,2));

% Laplace variable
s = zpk(0,[],1);

% Plots
xl = [10;250];                        % X axis limits (Hz)
yl = [-150,-70];                      % Y axis limits (dB)
w = linspace(10*2*pi,2*pi*250,1e4);   % Frequency vector (rad/s)

% Controlled modes
Nmodes = 2;
modes = 1:1:Nmodes;

% Residues
ri = [0.5,0.5];       % Hollkamp's and SFCF shunts 
r0v = [0.1,0.5,0.9];  % CF and SPIS shunts


col = get(groot,'defaultAxesColorOrder');


% Ideal circuits characteristics
[zi,zetai,wsch,woch] = impedanceSCOCFrequencies(woc2,-r2/Epe,modes,ri,'series');
[zi2,zetai2,wsch2,woch2] = admittanceSCOCFrequencies(wsc,r/Cpe,modes,ri,'parallel');


%% Impedance-based approach


% Hollkamp's shunt
[Ych,Lch,Rch,Cch] = HollkampTuning(ri,zi,zetai,Cpe);
beamssch = feedback(beamss,-minreal(Ych/s),2,2);

% FRFs
m = bode(beamss(1,1),w);        % Open-circuit
mch = bode(beamssch(1,1),w);    % Hollkamp's shunt

% Circuit parameters
LCFV = zeros(2,length(r0v));    % Inductances (H)
RCFV = zeros(2,length(r0v));    % Resistances (Ohms)
CCFV = zeros(2,length(r0v));    % Capacitances (F)

% FRF plot
figure();
hold on
plot(w/2/pi,db(squeeze(m)),'-k')
plot(w/2/pi,db(squeeze(mch)),'-','linewidth',2,'Color',0.5*[1,1,1])
for ii = 1 : length(r0v)
  % Residues
  ri0 = [r0v(ii) , (1-r0v(ii))/2 , (1-r0v(ii))/2];
  
  % Specification procedure
  [zi0,zetai0,wsch0,woch0] = admittanceSCOCFrequencies(wsc,r/Cpe,modes,ri0,'series');
  
  % Circuit tuning
  [Ycf,LCFV(:,ii),RCFV(:,ii),CCFV(:,ii)] = CFTuning(ri0,zi0,zetai0,Cpe);
  
  % Controlled system FRF
  beamsscf = feedback(beamss,-minreal(Ycf/s),2,2);
  mcf = bode(beamsscf(1,1),w);
  
  if ii ~= 1
    plot(w/2/pi,db(squeeze(mcf)),'-','linewidth',2,'Color',col(ii,:))
  else
    plot(w/2/pi,db(squeeze(mcf)),'--','linewidth',2,'Color',col(ii,:))
  end
end
box on
set(gca,'fontsize',14)
set(gca,'xscale','log')
xlabel('Frequency (Hz)')
ylabel('Receptance (dB, ref. m/N)')
xlim(xl)
ylim(yl)
set(gca,'XTick',[5e1,1e2,2e2])



%% Admittance-based approach


% Second Foster Canonical form
[Zsf,Lsf,Rsf,Csf] = SFCFTuning(ri,zi2,zetai2,Cpe);
beamsssf = feedback(beamss,-minreal(1/(s*Zsf)),2,2);

% FRFs
msc = bode(beamssSC(1,1),w);      % Short-circuit
msf = bode(beamsssf(1,1),w);      % SFCF shunt

% Circuit parameters
LSPV = zeros(2,length(r0v));      % Inductances (H)
RSPV = zeros(2,length(r0v));      % Resistances (Ohms)
CSPV = zeros(2,length(r0v));      % Capacitances (F)

% FRF plot
figure()
hold on
plot(w/2/pi,db(squeeze(msc)),'-k')
plot(w/2/pi,db(squeeze(msf)),'-','linewidth',2,'Color',0.5*[1,1,1]) 
for ii = 1 : length(r0v)
  % Residues
  ri0 = [r0v(ii) , (1-r0v(ii))/2 , (1-r0v(ii))/2];
  
  % Specification procedure
  [zi20,zetai20,wsch20,woch20] = impedanceSCOCFrequencies(woc2,-r2/Epe,modes,ri0,'parallel');
  
  % Circuit tuning
  [Zsp,LSPV(:,ii),RSPV(:,ii),CSPV(:,ii)] = SPISTuning(ri0,zi20,zetai20,Cpe);
  
  % Controlled system FRF
  beamsssp = feedback(beamss,-minreal(1/(s*Zsp)),2,2);
  msp = bode(beamsssp(1,1),w);
  
  if ii ~= 1
    plot(w/2/pi,db(squeeze(msp)),'-','linewidth',2,'Color',col(ii,:))
  else
    plot(w/2/pi,db(squeeze(msp)),'--','linewidth',2,'Color',col(ii,:))
  end
end
box on
set(gca,'fontsize',14)
set(gca,'xscale','log')
xlabel('Frequency (Hz)')
ylabel('Receptance (dB, ref. m/N)')
xlim(xl)
ylim(yl)
set(gca,'XTick',[5e1,1e2,2e2])



%% Circuits parameters as functions of r0

if plotParameters
  % Residues
  r0v = linspace(0,1,1000);
  r0v = r0v(2:end-1);
  
  % CF circuit parameters
  LCFV_parameters = zeros(2,length(r0v));     % Inductances (H)
  RCFV_parameters = zeros(2,length(r0v));     % Resistances (Ohms)
  CCFV_parameters = zeros(2,length(r0v));     % Capacitances (F)

  LSPV_parameters = zeros(2,length(r0v));     % Inductances (H)
  RSPV_parameters = zeros(2,length(r0v));     % Resistances (Ohms)
  CSPV_parameters = zeros(2,length(r0v));     % Capacitances (F)

  for ii = 1 : length(r0v)
    % Residues
    ri0 = [r0v(ii) , (1-r0v(ii))/2 , (1-r0v(ii))/2];

    % Specification procedure
    [zi0,zetai0,wsch0,woch0] = admittanceSCOCFrequencies(wsc,r/Cpe,modes,ri0,'series');
    [zi20,zetai20,wsch20,woch20] = impedanceSCOCFrequencies(woc2,-r2/Epe,modes,ri0,'parallel');

    % Circuits tuning
    [Ycf,LCFV_parameters(:,ii),RCFV_parameters(:,ii),CCFV_parameters(:,ii)] = CFTuning(ri0,zi0,zetai0,Cpe);
    [Zsp,LSPV_parameters(:,ii),RSPV_parameters(:,ii),CSPV_parameters(:,ii)] = SPISTuning(ri0,zi20,zetai20,Cpe);

  end

  % Inductances plot
  figure
  hold on
  plot(r0v,LCFV_parameters(1,:),'Color',col(1,:),'linewidth',2)
  plot(r0v,LCFV_parameters(2,:),'Color',0.4*col(1,:),'linewidth',2)
  plot(r0v,LSPV_parameters(1,:),'Color',col(2,:),'linewidth',2)
  plot(r0v,LSPV_parameters(2,:),'Color',0.4*col(2,:),'linewidth',2)
  box on
  set(gca,'yscale','log')
  ylim([1e-2,1e6])
  set(gca,'fontsize',14)
  xlabel('r_0 (-)')
  ylabel('Inductance (H)')

  % Resistances plot
  figure
  hold on
  plot(r0v,RCFV_parameters(1,:),'Color',col(1,:),'linewidth',2)
  plot(r0v,RCFV_parameters(2,:),'Color',0.4*col(1,:),'linewidth',2)
  plot(r0v,RSPV_parameters(1,:),'Color',col(2,:),'linewidth',2)
  plot(r0v,RSPV_parameters(2,:),'Color',0.4*col(2,:),'linewidth',2)
  box on
  set(gca,'yscale','log')
  set(gca,'fontsize',14)
  xlabel('r_0 (-)')
  ylabel('Resistance (\Omega)')


  % Capacitances plot
  figure
  hold on
  plot(r0v,CCFV_parameters(1,:),'Color',col(1,:),'linewidth',2)
  plot(r0v,CCFV_parameters(2,:),'Color',0.4*col(1,:),'linewidth',2)
  plot(r0v,CSPV_parameters(1,:),'Color',col(2,:),'linewidth',2)
  plot(r0v,CSPV_parameters(2,:),'Color',0.4*col(2,:),'linewidth',2)
  box on
  set(gca,'yscale','log')
  set(gca,'fontsize',14)
  xlabel('r_0 (-)')
  ylabel('Capacitance (F)')
  ylim([1e-10,1e-4])


  % Undamped frequencies plot
  figure
  hold on
  plot(r0v,1./(CCFV_parameters(1,:).*LCFV_parameters(1,:)).^(0.5)/2/pi,'Color',col(1,:),'linewidth',2)
  plot(r0v,1./(CCFV_parameters(2,:).*LCFV_parameters(2,:)).^(0.5)/2/pi,'Color',0.4*col(1,:),'linewidth',2)
  plot(r0v,woc(1).*sqrt(Cpe./(CCFV_parameters(1,:)+Cpe))/2/pi,'-.','Color',[0.5,0.5,0.5],'linewidth',2)
  plot(r0v,woc(2).*sqrt(Cpe./(CCFV_parameters(2,:)+Cpe))/2/pi,'-.','Color',[0.5,0.5,0.5],'linewidth',2)
  plot(r0v,1./(LSPV_parameters(1,:).*CSPV_parameters(1,:)).^(0.5)/2/pi,'Color',col(2,:),'linewidth',2)
  plot(r0v,1./(LSPV_parameters(2,:).*CSPV_parameters(2,:)).^(0.5)/2/pi,'Color',0.4*col(2,:),'linewidth',2)
  plot(r0v,woc(1).*sqrt(1+Cpe./CSPV_parameters(1,:))/2/pi,'-.','color',col(3,:),'linewidth',2)
  plot(r0v,woc(2).*sqrt(1+Cpe./CSPV_parameters(2,:))/2/pi,'-.','color',col(3,:),'linewidth',2)
  box on
  set(gca,'yscale','log')
  set(gca,'fontsize',14)
  xlabel('r_0 (-)')
  ylabel('Undamped frequency (Hz)')

end