%
% Plot of the FRF of multi-resonant shunts (Current Blocking with series or
% parallel RL shunts) controlling a clamped-free piezoelectric beam.
%
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Mar 2022


addpath('IdealCircuitsCodes\')
addpath('CircuitsTuning\')

%% Beam model

load('BeamModel');                                  % Beam with open-circuited patch

% Dynamic elastance/capacitance identification
[wsc,woc,Cpe,r] = dynamicCapacitanceResidues(1/beamss(2,2));

% Laplace variable
s = zpk(0,[],1);

% Controlled modes
Nmodes = 2;
modes = 1:1:Nmodes;

% Residues
r1 = [0.9,0.5,0.1];      % Hollkamp's and SFCF shunts 

% Plots
xl = [10;250];                        % X axis limits (Hz)
yl = [-150,-70];                      % Y axis limits (dB)
w = linspace(10*2*pi,2*pi*250,1e4);   % Frequency vector (rad/s)

% Open-circuit FRF
m0 = bode(beamss(1,1),w);

%% Impedance-based approach

% CB shunt with series RL shunts
figure
hold on
plot(w/2/pi,db(squeeze(m0)),'-k','linewidth',1)
for ii = 1 : length(r1)
  % Residues
  ri = [r1(ii) , 1-r1(ii)];
  
  % Specification procedure
  [zi,zetai] = admittanceSCOCFrequencies(wsc,r/Cpe,modes,ri,'series');
  
  % Circuit tuning
  [Zcb,Lcb,Rcb,Cfcb,Lfcb] = CBSTuning(ri,zi,zetai,Cpe);

  % Controlled system FRF
  beamssC = feedback(beamss,-1/(Zcb*tf('s')),2,2);
  mC = bode(beamssC(1,1),w);
  plot(w/2/pi,db(squeeze(mC)),'-','linewidth',2)

end
box on
xlabel('Frequency (Hz)')
ylabel('Receptance (dB, ref. m/N)')
set(gca,'fontsize',14)
set(gca,'xscale','log')
xlim(xl)
ylim(yl)
set(gca,'XTick',[5e1,1e2,2e2])



%% Admittance-based approach

% CB shunt with parallel RL shunts
figure
hold on
plot(w/2/pi,db(squeeze(m0)),'-k','linewidth',1)
for ii = 1 : length(r1)
  % Residues
  ri = [r1(ii) , 1-r1(ii)];
  
  % Specification procedure
  [zi,zetai] = admittanceSCOCFrequencies(wsc,r/Cpe,modes,ri,'parallel');
  
  % Circuit tuning
  [Zcb,Lcb,Rcb,Cfcb,Lfcb] = CBPTuning(ri,zi,zetai,Cpe);

  % Controlled system FRF
  beamssC = feedback(beamss,-1/(Zcb*tf('s')),2,2);
  mC = bode(beamssC(1,1),w);
  plot(w/2/pi,db(squeeze(mC)),'-','linewidth',2)
  
end
box on
xlabel('Frequency (Hz)')
ylabel('Receptance (dB, ref. m/N)')
set(gca,'fontsize',14)
set(gca,'xscale','log')
xlim(xl)
ylim(yl)
set(gca,'XTick',[5e1,1e2,2e2])
