%
% Exact H infinity tuning of series and parallel RL shunts
%
% See
%
% Soltani, P., Kerschen, G. Tondreau, G., Deraemaeker, A. (2014) 
% Piezoelectric vibration damping using resonant shunt circuits: an exact solution,
% Smart Materials and Structures, 23(12), p 125014. doi: 10.1088/0964-1726/23/12/125014.
%
% Tang, X., Liu, Y., Cui, W., Zuo, L. (2016) 
% Analytical Solutions to H2 and H? Optimizations of Resonant Shunted Electromagnetic Tuned Mass Damper and Vibration Energy Harvester,
% Journal of Vibration and Acoustics, 138(1). doi: 10.1115/1.4031823.
%
% Ikegame, T., Takagi, K., Inoue, T. (2019) 
% Exact Solutions to H? and H2 Optimizations of Passive Resonant Shunt Circuit for Electromagnetic or Piezoelectric Shunt Damper,
% Journal of Vibration and Acoustics, 141(3). doi: 10.1115/1.4042819.
%
% Sep 2021
% 
% Input:
%   * Cp: piezoelectric capacitance at constant strain
%   * wsc: short-circuit resonance frequency
%   * woc: open-circuit resonance frequency
%   * shuntType: 'series' (default) or 'parallel'
%
% Output:
%   * R: shunt resistance
%   * L: shunt inductance
%   * zeta: electrical damping ratio
%   * delta: electrical frequency ratio
%
function [R,L,zeta,delta] = ikegameRLShunt(Cp,wsc,woc,shuntType)

  if nargin < 4
    shuntType = 'series';
  end

  if strcmp(shuntType,'series')
    Kc2 = (woc^2-wsc^2)/wsc^2;
    
    % Soltani/Ikegame
    r = (sqrt(64-16*Kc2-26*Kc2^2)-Kc2)/8;
    delta = sqrt((8+3*Kc2-4*r)/(Kc2+1))/2;
    zeta = 2*sqrt(64+80*Kc2+27*Kc2^2 - 16*r*(4+3*Kc2))/(sqrt(2)*(8+5*Kc2))/delta;
  
    R = zeta/(woc*Cp);
    L =  1/(delta^2*woc^2*Cp);
    
    zeta = delta*zeta/2;
  elseif strcmp(shuntType,'parallel')
    Kc2 = (woc^2-wsc^2)/wsc^2;
    
    % Tang/Ikegame
    a0 = 64;
    a1 = -16*Kc2;
    a2 = -64 + 16*Kc2 + 11*Kc2^2;
    a3 = 2*Kc2*(8 - Kc2)*(2 - Kc2);
    a4 = -2*Kc2^2*(2 - Kc2);
    b6 = 27*(a0*a3^2 + a1^2*a4) - 9*a2*(a1*a3 + 8*a0*a4) + 2*a2^3;
    b5 = 12*a0*a4 - 3*a1*a3 + a2^2;
    b4 = ((sqrt(b6^2 - 4*b5^3) + b6)/2)^(1/3);
    b3 = sqrt((a1/a0)^2 + 4*(b4^2 + b5 - 2*a2*b4)/(3*a0*b4))/2;
    b2 = -(8*a0^2*a3 - 4*a0*a1*a2 + a1^3)/(4*a0^3*b3);
    b1 = (3*a1^2*b4 - 2*a0*(4*a2*b4 + b4^2 + b5))/(6*a0^2*b4);
    
    r = -a1/(4*a0) + b3/2 + sqrt(b1 + b2)/2;
    delta2 = (2*r - Kc2 + sqrt(16*r^2 - 4*r*Kc2 + Kc2^2))/6;
    delta = sqrt(real(delta2));
    zeta = real(sqrt((r - 1)*(3*r - 2*Kc2)/(r*(2*r + Kc2 - sqrt(16*r^2 - 4*r*Kc2 + Kc2^2)))));
    
    L = real(1/(wsc^2*delta2*Cp));
    R = real(sqrt(L/Cp)/(2*zeta));
  else
    error(['Unknown shunt type: ',shuntType]);
  end

end