% Tuning of a Second Foster Canonical Form shunt
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Mar 2022
%
% Input:
%   * ri: residues vector of length Ns
%   * zi: zeros of Norton's admittance (vector of length Ns)
%   * zetai: electrical damping ratio (vector of length Ns)
%   * Cp: piezoelectric capacitance at constant strain
% 
% Output:
%   * Z: SFCF shunt impedance
%   * L: shunt inductance vector of length Ns
%   * R: shunt resistance vector of length Ns
%   * C: shunt capacitance vector of length Ns (first element is zero)
%
function [Z,L,R,C] = SFCFTuning(ri,zi,zetai,Cp)

  
  % Poles of the Thevenin equivalent impedance
  Nnorm = round(length(zi)/2);
  ri = ri(:);
  MP = zeros(length(zi));
  for ii = 1 : length(zi)
    pol = 1;
    for jj = 1 : length(zi)
      if jj ~= ii
        pol = conv(pol,[(zi(Nnorm)/zi(jj)).^2,1]);
      end
    end
    MP(ii,:) = pol;
  end
  pol = ri.'*MP;
  rr = roots(pol);
  pi = sort(sqrt(-rr))*zi(Nnorm);
  
  % Undamped circuit tuning
  L = -1./(Cp*(pi.^2));
  L0 = 1/Cp;
  for ii = 1 : length(L)
    indni = setdiff(1:1:length(pi),ii);
    L0 = L0*pi(ii)^2/zi(ii)^2;
    L(indni) = L(indni).*(1-(pi(indni)/zi(ii)).^2)./(1-(pi(indni)/pi(ii)).^2);
    L(ii) = L(ii).*(1 - (pi(ii)/zi(ii))^2);
  end
  L0 = L0/zi(end)^2;
  L = L.*(1 - (pi/zi(end)).^2);
  
  C = [0 ; 1./(L.*pi.^2)];
  L = [L0 ; L];
  
  % Resistances tuning
  CC = diag(C) + Cp*ones(size(L));
  LL = diag(1./L);
  [PhiE,Om2E] = eig(LL,CC);
  R = 1./((PhiE.'.^2)\(2*zetai.*zi));
  
  
  if any(R<0)
    warning('There is at least one negative resistance in the circuit')
  end
  
  % Impedance
  s = zpk(0,[],1);
  Z = 1/(1/(s*L(1)) + 1/R(1));
  for ii = 2 : length(L)
    Z = Z + 1/C(ii)*s/(s^2 + s/(R(ii)*C(ii)) + 1/(C(ii)*L(ii)));
  end
  
end