% Tuning of a Current Blocking shunt with series RL shunts
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Mar 2022
%
% Input:
%   * ri: residues vector of length Ns
%   * zi: zeros of Thevenin's impedance (vector of length Ns)
%   * zetai: electrical damping ratio (vector of length Ns)
%   * Cp: piezoelectric capacitance at constant strain
% 
% Output:
%   * Z: CB shunt impedance
%   * L: shunt inductance vector of length Ns
%   * R: shunt resistance vector of length Ns
%   * Cf: filter capacitance vector of length Ns-1
%   * Lf: filter inductance vector of length Ns-1
%
function [Z,L,R,Cf,Lf] = CBSTuning(ri,zi,zetai,Cp)

  % Filter characteristics
  Cf = zeros(length(ri)-1,1);
  for ii = 1 : length(Cf)
    Cti = sum(1./Cf(1:(ii-1)));
    rit = sum(ri(1:ii));
    Cf(ii) = (1 - rit)/(rit*(1/Cp+Cti) - Cti);
  end
  Lf = 1./(Cf.*zi(1:end-1).^2);
  
    
  % Shunt branches tuning
  L = zeros(length(zi),1);
  R = zeros(length(zi),1);
  for ii = 1 : length(L)
    % Damped circuit
    sz = 1i*zi(ii);
    H = [1 , 1/(sz*Cp) ; 0 , 1];
    
    for jj = 1 : ii-1
      % Damped circuit
      Ys = 1/(L(jj)*sz + R(jj));
      Zf = sz/Cf(jj)/(zi(jj)^2 + sz^2);
      H = H*[1, Zf ; Ys , 1+Zf*Ys];
    end
    
    % Ideal impedance
    Zt = 0;
    for jj = 1 : length(ri)
      Zt = Zt + ri(jj)/(1 + 2*zetai(jj)*sz/zi(jj) + sz^2/zi(jj)^2);
    end
    Zt = 1/(sz*Cp*Zt);
    Yk = ( H(2,1) * Zt - H(1,1) )/( H(1,2) - H(2,2) * Zt);
    
    % Inductance tuning
    L(ii) = 1/zi(ii)*imag(1/Yk);
    
    % Resistance tuning
    R(ii) = real(1/Yk);
    
  end
  
  % Impedance 
  s = zpk(0,[],1);
  Z = L(end)*s + R(end);
  for ii = (length(L)-1) : -1 : 1
    Ys = 1/(L(ii)*s + R(ii));
    Zf = s/Cf(ii)/(s^2 + 1/(Lf(ii)*Cf(ii)));
    Z = 1/( Ys + 1/( Zf + Z));
  end

end