%
% Plot the frequency at which a zero crosses the imaginary axis under the
% effect of delays for a digital shunt, using zero-order hold and pure
% delay models
%
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Apr 2022

colv = get(groot,'defaultAxesColorOrder');

% Starting, maximum and minimum electromechanical coupling factors
Kc0 = 0.5;
KcMax = 0.7;
KcMin = 1e-5;

% Critical delay and frequency
[tauc0,wc0] = criticalDelays(Kc0);
tauc0 = tauc0(3)/Kc0;
wc0 = wc0(3);

%% Homotopy

% Homotopy for the Zero-Order Hold model
[xv,Kcv] = homotopy(@(x,Kc) imaginaryAxisCrossingFunction_ZOH(Kc,x(1),x(2)),...
  [wc0;tauc0],Kc0,[KcMin,KcMax,-1,1e-3],3,1e4,false);
wc = xv(1,:);
tauc = xv(2,:);

% Homotopy for the pure delay model
[xv2,Kcv2] = homotopy(@(x,Kc) imaginaryAxisCrossingFunction_Delay(Kc,x(1),x(2)),...
  [wc0;tauc0],Kc0,[KcMin,KcMax,-1,1e-3],3,1e4,false);
wc2 = xv2(1,:);
tauc2 = xv2(2,:);

%% Plot


% Theoretical value
[taucth,wcth] = criticalDelays(Kcv);

% Plot
figure
hold on
plot(Kcv,tauc','-k','linewidth',2)
plot(Kcv,tauc2','linewidth',2,'color',0.5*[1,1,1])
plot(Kcv,taucth(:,3),'--','linewidth',2,'color',colv(1,:))
xlim([0,Kc0])
box on
set(gca,'fontsize',14)
xlabel('K_c (-)')
ylabel('\omega_{sc}\tau_{c} (-)')





%% Functions
% NB: functions in a script require Matlab R2016b or later.

% Characteristic polynomial of a single-degree-of-freedom oscillator
% controlled by a digital shunt with delays (zero-order hold model)
%
% Input:
%   * Kc: electromechanical coupling factor (-)
%   * w: circular frequency of the zero (rad/s)
%   * tau: sampling time (s)
%
% Output:
%   * f: characteristic polynomial value
%
function f = imaginaryAxisCrossingFunction_ZOH(Kc,w,tau)

  % Shunt characteristics
  [delta2,zeta] = ikegame(Kc);
  delta2 = 1/(delta2^2*(1+Kc^2));
  zeta = zeta/sqrt(1+Kc^2);
  
  % Poles
  p01 = conv([1,0,1+Kc^2],[delta2,zeta,0]);
  p02 = [0,0,1,0,1];

  p = 1i*w;
  f = abs([(polyval(p01,p) + polyval(p02,p).*(1-exp(-p.*tau))./(p.*tau)) ;
    (polyval(p01,-p) + polyval(p02,-p).*(1-exp(p.*tau))./(-p.*tau))]).^2/Kc^2;
end



% Characteristic polynomial of a single-degree-of-freedom oscillator
% controlled by a digital shunt with delays (pure delay model)
%
% Input:
%   * Kc: electromechanical coupling factor (-)
%   * w: circular frequency of the zero (rad/s)
%   * tau: sampling time (s)
%
% Output:
%   * f: characteristic polynomial value
%
function f = imaginaryAxisCrossingFunction_Delay(Kc,w,tau)

  % Shunt characteristics
  [delta2,zeta] = ikegame(Kc);
  delta2 = 1/(delta2^2*(1+Kc^2));
  zeta = zeta/sqrt(1+Kc^2);
  
  % Poles
  p01 = conv([1,0,1+Kc^2],[delta2,zeta,0]);
  p02 = [0,0,1,0,1];

  p = 1i*w;
  f = abs([(polyval(p01,p) + polyval(p02,p).*exp(-p.*tau/2)) ;
    (polyval(p01,-p) + polyval(p02,-p).*exp(p.*tau/2))]).^2/Kc^2;
end
