% Frequency tuning ratio, damping ratio and r-value for a series RL shunt
% circuit.
%
% See
%
% Soltani, P., Kerschen, G. Tondreau, G., Deraemaeker, A. (2014) 
% Piezoelectric vibration damping using resonant shunt circuits: an exact solution,
% Smart Materials and Structures, 23(12), p 125014. doi: 10.1088/0964-1726/23/12/125014.
%
% Ikegame, T., Takagi, K., Inoue, T. (2019) 
% Exact Solutions to H? and H2 Optimizations of Passive Resonant Shunt Circuit for Electromagnetic or Piezoelectric Shunt Damper,
% Journal of Vibration and Acoustics, 141(3). doi: 10.1115/1.4042819.
%
% Jan 2022
%
% Input:
%   * Kc, the enhanced/generalized electromechanical coupling factor
% Ouput:
%   * delta, the frequency ratio
%   * zeta, the damping ratio
%   * r, such that r^2 = 1 - 1/h^2, where h is the peak amplitude
%
function [delta,zeta,r] = ikegame(Kc)

  r = (sqrt(64-16*Kc^2-26*Kc^4)-Kc^2)/8;
  
  zeta = sqrt(64+80*Kc^2+27*Kc^4 - 16*r*(4+3*Kc^2))/(sqrt(2)*(8+5*Kc^2));
  
  delta = sqrt(8+3*Kc^2-4*r)/2;
  
  delta = delta/sqrt(Kc^2+1);
  zeta = zeta*2/delta;

end