%
% Perform a numerical homotopy on a parameter lambda to satisfy
% f(x,lambda)= 0, starting from an initial guess, until one of the bounds
% for lambda (upper or lower) is reached, until the maximum number of steps
% is reached or until convergence fails.
%
% x is assumed to have the same length as its initial guess, x0
%
% The function roots are found with fsolve, and the procedure can use an
% adaptive step length to attempt to set the number of iterations to a
% desired value.
%
% Input:
%   * f: function whose roots f(x,lambda) = 0 are to be found
%   * x0: initial guess for x
%   * lam0: initial value for lambda
%   * LambdaData = [lambdaMin , lambdaMax , lambdaDir , dl], where
%     * lambdaMin: minimum value for lambda
%     * lambdaMax: maximum value for lambda
%     * lambdaDir: set to 1 to increase lambda from lambda0 to lambdaMax;
%     set to -1 to decrease lambda from lambda0 to lambdaMin.
%     * dl: step length (initial step length if an adaptive approach is
%     used).
%   * iterMax: maximum number of homotopy steps
%   * iterId: ideal number of correction iterations for the adaptive scheme
%   (default: 3)
%   * adaptive: set to true to use an adaptive step length (default: true)
%
% Output:
%   * X: matrix of size length(x0) x iter, where number is the number of
%   iterations taken by the homotopy procedure, containing the unknown x
%   * Lambda: vector of size 1 x iter, whose values correspond to the
%   columns of X.
%
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Apr 2022
%
function [X,Lambda] = homotopy(f,x0,lam0,LambdaData,iterMax,iterId,adaptive)

  % Default adaptive scheme
  if nargin < 7
    adaptive = true;
    if nargin < 6
      iterId = 3;
    end
  end

  % Fsolve options
  opts = optimoptions(@fsolve,'display','none','FunctionTolerance',1e-12,'OptimalityTolerance',1e-12,...
    'StepTolerance',1e-12,'MaxFunctionEvaluations',1e4,'MaxIterations',iterMax,...
    'FiniteDifferenceType','central');

  % Initial guess correction
  x0 = x0(:);
  [x0,~,~,output] = fsolve(@(x) f(x,lam0),x0,opts);
  
  if output.iterations > iterMax
      disp('Homotopy could not converge: initial point not converged')
      X = zeros(length(x0),0);
      Lambda = zeros(1,0);
      return
  end
  
  % Results storage
  X = zeros(length(x0),iterMax);
  Lambda = zeros(1,iterMax);
  X(:,1) = x0;
  dx = zeros(size(x0));
  Lambda(1) = lam0;
  
  % Input data on the homotopy parameter
  lambdaMin = LambdaData(1);
  lambdaMax = LambdaData(2);
  lambdaDir = LambdaData(3);
  dl = lambdaDir*LambdaData(4);
  dl0 = dl;
  
  % Numerical homotopy procedure
  iter = 1;
  while Lambda(iter) > lambdaMin && Lambda(iter) < lambdaMax && iter < iterMax
    
    % Next trial step
    lam = Lambda(iter) + dl;
    x0 = X(:,iter) + dx*dl;
    
    iter = iter + 1;
    
    % Attempt to find the roots of the equation
    [x,~,~,output] = fsolve(@(x) f(x,lam),x0,opts);
    
    % Secant predictor
    dx1 = x - X(:,iter-1);
    
    % Check corrector output
    if output.iterations < iterMax
      X(:,iter) = x;
      Lambda(iter) = lam;
      if norm(dx1)
        dx = dx1/dl;
      else
        % fsolve did not change the solution under the given tolerances ->
        % set dx to zero
        dx = zeros(size(x));
      end
      if adaptive
        dl = iterId/max([output.iterations,1])*dl;
      end
    else
      % fsolve failed to converge: retry with a smaller step
      iter = iter - 1;
      dl = 0.5*dl;
      if dl < 1e-6*dl0
        disp('Continuation could not converge: minimum step reached')
        break
      end
    end
  end
  
  % Discard zero entries in the output
  X = X(:,1:iter);
  Lambda = Lambda(1:iter);
end