%
% Simulate the behavior of a single-degree-of-freedom oscillator controlled
% by a digital vibration absorber under different sample times. The
% oscillator is excited by an external, swept-sine force. The shunt
% admittance is modified to account for sampling delay effects.
%
% The output of the simulation can be monitored in real time in the
% Simulink file, using the scope named "Displacement monitoring".
%
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Apr 2022


% Warning: long computation!


addpath('../')

% Electromechanical coupling factors
Kcv = [0.01,0.1];

% Simulations
for nn = 1 : length(Kcv)
  Kc = Kcv(nn);
  
  % Critical delay
  tauc = criticalDelays(Kc);
  tauc = tauc(3);
  
  % Shunt characteristics
  [delta2,zeta,r] = ikegame(Kc);
  delta2 = 1/(delta2^2*(1+Kc^2));
  zeta = zeta/sqrt(1+Kc^2);
  
  % Closed-loop poles of the nominal system
  p011 = [1,0,1+Kc^2];
  p01 = conv(p011,[delta2,zeta,0]);
  p02 = [0,0,1,0,1];
  p0 = p01 + p02;
  pp = roots(p0);
  
  % Sample times
  tauv = [0.01,0.1,0.5,1,pi];
  
  figure
  hold on
  colv = get(groot,'defaultAxesColorOrder');
  for ii = 1: 1 : length(tauv)
    
    % Sample time
    tau = tauv(ii);
    
    % Modification procedure
    b = 1-(1-exp(-tau*pp))./(tau*pp);
    A = [-pp*delta2./(pp*delta2+zeta),-zeta./(pp*delta2+zeta)];
    d = real(pinv(A)*b);
    delta2m = (1+d(1))*delta2;
    zetam = (1+d(2))*zeta;
    
    % Admittance discretization
    Yd = c2d(tf(1,[delta2m,zetam]),tau,'Tustin');
    [nd,dd] = tfdata(Yd,'v');
    
    if nn == 2
      w0 = 0.5;
      w1 = 1.5;
      Tf = 1e4;
    else
      w0 = 0.95;
      w1 = 1.05;
      Tf = 1e5;
    end
    x0 = [0;0];
    q0 = 0;
    xd0 = 0;
    
    try   % Unstable systems may result in errors in Simulink => skip those cases
    
      % Run simulation
      simOut = sim('DigitalSystem','SimulationMode','normal');
      while ~strcmp(get_param('DigitalSystem','SimulationStatus'),'stopped')
        % Wait for simulation to complete
      end
      
      % Simulation output
      outputs = simOut.get('yout');
      xDigital = outputs{1}.Values.Data;
      xAnalog = outputs{2}.Values.Data;
      
      plot(linspace(w0,w1,length(xDigital)),db(envelope(xDigital)),'linewidth',2,'Color',colv(ii,:))
      
    catch e
    end
  end
  if nn == 2
    xlim([0.8,1.2])
    ylim([0,50])
  else
    xlim([0.97,1.03])
    ylim([0,75])
  end
  box on
  xlabel('Normalized instantaneous frequency (-)')
  ylabel('Normalized response (dB)')
  set(gca,'fontsize',14)
end
