%
% Plot the phase margin of a digital piezoelectric shunt
%
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Apr 2022


% Frequency vector
wv = linspace(0.8,1.2,10000);

% Electromechanical coupling factors
Kcv = [0.01,0.05,0.1,0.2];

% Laplace variable
s = zpk(0,[],1);


%% Open-loop FRF

figure
hold on
for ii = length(Kcv) : -1 : 1

  Kc = Kcv(ii);
  
  % Shunt characteristics
  [delta,zeta,r] = ikegame(Kc);
  delta = sqrt(1/(delta^2*(1+Kc^2)));
  zeta = zeta/sqrt(1+Kc^2);
  
  % Open-loop transfer function
  H = (s^2+1)/(s^2+1+Kc^2)/(delta^2*s^2+zeta*s);
  [m,p] = bode(H,wv);
  
  % Bode plot
  subplot(2,1,1)
  hold on
  plot(wv,squeeze(db(m)),'linewidth',2)
  
  subplot(2,1,2)
  hold on
  plot(wv,squeeze(p),'linewidth',2)
end

subplot(2,1,1)
hold on
plot([wv(1),wv(end)],[0,0],'--k')
ylim([-10,10])
box on
ylabel('Magnitude (dB)')
set(gca,'fontsize',14)
set(gca,'XTickLabel',{})

subplot(2,1,2)
hold on
plot([wv(1),wv(end)],[-180,-180],'--k')
ylim([-190,-150])
box on
xlabel('Normalized frequency (-)')
ylabel('Phase ()')
set(gca,'fontsize',14)


%% Phase margin as a function of the EEMCF

% Electromechanical coupling factors
Kcv = logspace(-3,log10(0.5),100);

% Phase margin vector
mp = zeros(size(Kcv));

for ii = 1 : length(Kcv)
  Kc = Kcv(ii);
  
  % Shunt characteristics
  [delta,zeta,r] = ikegame(Kc);
  delta = sqrt(1/(delta^2*(1+Kc^2)));
  zeta = zeta/sqrt(1+Kc^2);
  
  % Open-loop transfer function and phase margin
  H = (s^2+1)/(s^2+1+Kc^2)/(delta^2*s^2+zeta*s);
  [~,mp(ii)] = margin(H);
end


% Plot
figure
plot(Kcv,mp,'-','linewidth',2)
box on
set(gca,'xscale','log')
set(gca,'yscale','log')
xlabel('K_c (-)')
ylabel('Phase margin ()')
set(gca,'fontsize',14)


