%
% Plot root loci (parametrized by the sampling time) of a
% single-degree-of-freedom oscillator controlled by a digital vibration
% absorber.
%
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Apr 2022


% Electromechanical coupling factors
Kcv = [0.01,0.05,0.1,0.3];


colv = get(groot,'defaultAxesColorOrder');

%% Root locus
for ii = 1 : length(Kcv)
  Kc = Kcv(ii);

  % Critical delay
  tauc = criticalDelays(Kc);
  tauc = tauc(3);
  
  % Shunt characteristics
  [delta2,zeta,r] = ikegame(Kc);
  delta2 = 1/(delta2^2*(1+Kc^2));
  zeta = zeta/sqrt(1+Kc^2);

  % Characteristic polynomial and roots (nominal system)
  p01 = conv([1,0,1+Kc^2],[delta2,zeta,0]);
  p02 = [0,0,1,0,1];
  p0 = p01 + p02;
  pp = roots(p0);

  % Characteristic polynomial and roots (delayed system)
  characteristicPolynomial = @(p,tau) (p*tau.*polyval(p01,p) + polyval(p02,p).*(1-exp(-p*tau)));
  [pv,tauv] = homotopy(characteristicPolynomial,...
    pp,0,[-1e-6,pi,1,1e-3],3,1e5,false);
  
  % Roots at sampling times 10^-3, 10^-2 and 10^1
  indM = zeros(3,1);
  for jj = 1 : length(indM)
    [~,indM(jj)] = min(abs(tauv-10^(jj-3)));
  end
  
  % Root locus plot
  figure
  hold on
  plot(pv.','-','Linewidth',2,'Color',colv(1,:))
  plot(pp,'xk','Linewidth',2,'Markersize',10)
  plot(pv(:,end).','x','Color',colv(1,:),'Linewidth',2,'Markersize',10)
  plot(pv([1,3],indM(1)),'.','Linewidth',2,'Markersize',15,'Color',colv(2,:))
  plot(pv([1,3],indM(2)),'.','Linewidth',2,'Markersize',15,'Color',colv(3,:))
  plot(pv([1,3],indM(3)),'.','Linewidth',2,'Markersize',15,'Color',colv(4,:))
  ylim([0,1.5])
  xlim([-0.12,0.08])
  yl = ylim();
  xl = xlim();
  plot([0,0],yl,'-k')
  plot(xl,[0,0],'-k')
  box on
  xlabel('\Re (s/\omega_{sc})')
  ylabel('\Im (s/\omega_{sc})')
  set(gca,'fontsize',14)
end