% Modification of a digital shunt admittance to counter delay-induced
% instabilities
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Apr 2022
% 
% Input:
%   * Ys: nominal shunt admittance
%   * tau: sampling period
%   * Cp: piezoelectric capacitance at constant strain
%   * wsc: short-circuit (circular) resonance frequencies
%   * woc: open-circuit (circular) resonance frequencies
%   * modes: numbers of the modes targeted by the shunt
%   * g: parallel conductance (default: 0)
%
% Output:
%   * Yd: modified shunt admittance
%
function Yd = delaysModification(Ys,tau,Cp,wsc,woc,modes,g)
  
  % Parallel conductance
  if nargin < 7
    g = 0;
  end

  % Piezoelectric capacitance
  Yp = zpk(kron(woc,[1i;-1i]),kron(wsc,[1i;-1i]),Cp) + g/zpk(0,[],1) ;

  % Nominal admittance coefficients
  [nn,dd] = tfdata(Ys,'v');
  while nn(1) == 0
    nn = nn(2:end);
  end
  while dd(1) == 0
    dd = dd(2:end);
  end
  nn = nn(:);
  dd = dd(:);
  
  % Poles of the closed-loop system: find the four poles closest to the
  % avergage of wsc, woc.
  [~,zz0] = pzmap(Ys/zpk(0,[],1)+Yp);
  wm = 0.5*(woc(modes)+wsc(modes));
  zz = zeros(4,length(modes));
  for ii = 1 : length(modes)
    [~,ind] = sort(abs(abs(zz0)-wm(ii)));
    zz(:,ii) = zz0(ind(1:4));
  end
  zz = zz(:);
  

  % Delays corrections
  Hd = (1-exp(-tau*zz))./(tau*zz);
  b = 1 - Hd;
  
  zn = zz.^(((length(nn)-1):-1:0));
  zd = zz.^(((length(dd)-1):-1:0));
  
  A = [diag(Hd./(zn*nn))*zn*diag(nn),-diag(1./(zd*dd))*zd*diag(dd)];

  % Modification coefficients
  d = real(pinv(A(:,2:end))*b);
  d = [0;d];
  
  % Modified numerator and denominator
  nnd = (1+d(1:length(nn))).*nn;
  ddd = (1+d((length(nn)+1):end)).*dd;
  
  Yd = tf(nnd',ddd');
  
end