%
% Compare the approximate laws correcting for delay-induced instabilities
% to numerical results
%
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Apr 2022

colv = get(groot,'defaultAxesColorOrder');

% Electromechanical coupling factors
Kcv = [0.01,0.1,0.5];

% Sample times
tauv = linspace(1e-6,pi,200);


%% Laws comparison
f1 = figure();
f2 = figure();
for nn = 1 : length(Kcv)
  Kc = Kcv(nn);
  
  % Laplace variable vector
  sv = 1i*linspace(0.8,1.2,1e4);
  
  % Shunt characteristics
  [delta2,zeta,r] = ikegame(Kc);
  delta2 = 1/(delta2^2*(1+Kc^2));
  zeta = zeta/sqrt(1+Kc^2);
  
  % Nominal closed-loop poles
  p011 = [1,0,1+Kc^2];
  p01 = conv(p011,[delta2,zeta,0]);
  p02 = [0,0,1,0,1];
  p0 = p01 + p02;
  pp = roots(p0);
  pp0 = pp;
  
  % Critical delays
  tauc = criticalDelays(Kc);
  tauc = tauc(3);
  
  % Modified parameters
  delta2m = zeros(size(tauv));
  zetam = zeros(size(tauv));
  
  for ii = 1 : length(tauv)
    
    % Modification procedure (numerical)
    b = 1-(1-exp(-tauv(ii)*pp))./(tauv(ii)*pp);
    A = [-pp*delta2./(pp*delta2+zeta),-zeta./(pp*delta2+zeta)];
    d = real(pinv(A)*b);
    delta2m(ii) = d(1);
    zetam(ii) = d(2);
    
    
  end
  
  % Modification procedure (approximate)
  [deltaL,deltaR] = approximateDelaysModificationRLShunt(tauv,Kcv(nn));
  
  % Frequency modification
  figure(f1)
  hold on
  plot(tauv,delta2m,'-','linewidth',2,'Color',colv(nn,:))
  plot(tauv,deltaL,'--','linewidth',2,'Color',0.5*colv(nn,:))
  
  % Damping modification
  figure(f2)
  hold on
  plot(tauv,zetam,'-','linewidth',2,'Color',colv(nn,:))
  plot(tauv,deltaR,'--','linewidth',2,'Color',0.5*colv(nn,:))
  
end


figure(f1)
xlim([0,pi])
ylim([-1.6,0])
xlabel('\omega_{sc}\tau (-)')
ylabel('\delta_L (-)')
set(gca,'fontsize',14)
box on

figure(f2)
xlim([0,pi])
ylim([0,60])
xlabel('\omega_{sc}\tau (-)')
ylabel('\delta_R (-)')
set(gca,'fontsize',14)
box on
