%
% Damping of two modes of a clamped-free beam multimodal circuits with
% ideal characteristics
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Sep 2021

clear

% Beam models
load('BeamModel');
load('BeamSCModel');

% Frequency vector
w = linspace(2e2,5000,1e4);

% Modal dampings
xi1 = 1e-3;
xi2 = 1e-3;

% Uncontrolled maximum amplitudes
hmaxT1 = getPeakGain(beamSCss(1,1),1e-9,[0,1900]);
hmaxT2 = getPeakGain(beamSCss(1,1),1e-9,[1900,5800]);

% Controlled modes
modes = 1:1:2;

% Residues
r1 = logspace(-7,7,100);
r1 = r1./(1+r1);            % Residue on the first mode

r1Plot = r1./(1-r1);        % X-axis scaling for the attenuation plots
rt = [0.0001,0.5,0.9999];   % Ticks on the attenuation plots
r1r = [0.1,0.5,0.9];        % Residues for the FRF


% Laplace variable
s = zpk(0,[],1);

% Short- and open-circuit FRFs
m = bode(beamSCss(1,1),w);
m0 = bode(beamss(1,1),w);


%% Impedance-based tuning

% Dynamic capacitance identification
[wsc,woc,Cpe,r] = dynamicCapacitanceResidues(1/beamss(2,2));



% FRFs
figure
hold on
plot(w,squeeze(db(m0)),'-k')
for ii = 1 : length(r1r)
  % Impedance characteristics
  ri = [r1r(ii) , 1-r1r(ii)];
  [zi,zetai] = admittanceSCOCFrequencies(wsc,r/Cpe,modes,ri,'series');

  % Ideal Thevenin impedance
  Yid = idealImpedance(ri,zi,zetai,Cpe);
  Zid = 1/Yid;

  % Shunted system
  beamShunted = feedback(beamss,-minreal(1/(s*Zid)),2,2);
  mc = bode(beamShunted(1,1),w);
  plot(w,squeeze(db(mc)),'-','linewidth',2)
end
set(gca,'xscale','log')
set(gca,'fontsize',14)
xlim([w(1),w(end)])
xlabel('Frequency (rad s^{-1})')
ylabel('Receptance (dB, ref. m N^{-1})')
box on
set(gca,'XTick',[5e2,1e3,2e3])


% Residues effect

hmaxT = zeros(2,length(r1));    % FRF amplitude at the effective open-circuit frequency
wschv = zeros(2,length(r1));    % Effective short-circuit frequency
wochv = zeros(2,length(r1));    % Effective open-circuit frequency


for ii = 1 : length(r1)
  % Impedance characteristics
  ri = [r1(ii) , 1-r1(ii)];
  [zi,zetai,wschv(:,ii),wochv(:,ii)] = admittanceSCOCFrequencies(wsc,r/Cpe,modes,ri,'series');

  % Ideal Thevenin impedance
  Yid = idealImpedance(ri,zi,zetai,Cpe);
  Zid = 1/Yid;

  % Shunted system
  beamShunted = feedback(beamss,-minreal(1/(s*Zid)),2,2);
  
  % Amplitude at the effective open-circuit resonance frequency
  hmaxT(1,ii) = abs(freqresp(beamShunted(1,1),wochv(1,ii)));
  hmaxT(2,ii) = abs(freqresp(beamShunted(1,1),wochv(2,ii)));
end

% Effective EMCFs
Kcvv1 = sqrt((wochv(1,:)./wschv(1,:)).^2-1);
Kcvv2 = sqrt((wochv(2,:)./wschv(2,:)).^2-1);

% Predicted reduction
Adbv1 = ThomasReduction(Kcvv1,xi1);
Adbv2 = ThomasReduction(Kcvv2,xi2);

% Attenuation plot
figure
hold on
plot(r1Plot,db(hmaxT1./hmaxT(1,:)),'linewidth',2)
plot(r1Plot,Adbv1,'-.','linewidth',2)
plot(r1Plot,db(hmaxT2./hmaxT(2,:)),'linewidth',2)
plot(r1Plot,Adbv2,'-.','linewidth',2)
xlim([r1Plot(1),r1Plot(end)])
set(gca,'xscale','log')
set(gca,'XTick',rt./(1-rt))
set(gca,'XTickLabel',{'0.0001','0.5','0.9999'})
set(gca,'fontsize',14)
xlabel('r_1 (-)')
ylabel('Attenuation (dB)')
box on




%% Admittance-based tuning


% Dynamic capacitance identification
[wsc,woc,Cpe,r] = dynamicCapacitanceResidues(1/beamss(2,2));


% FRFs
figure
hold on
plot(w,squeeze(db(m0)),'-k')
for ii = 1 : length(r1r)
  % Admittance characteristics
  ri = [r1r(ii) , 1-r1r(ii)];
  [zi,zetai,wschv(:,ii),wochv(:,ii)] = admittanceSCOCFrequencies(wsc,r/Cpe,modes,ri,'parallel');

  % Ideal Norton admittance
  Yid = idealAdmittance(ri,zi,zetai,Cpe);

  % Shunted system
  beamShunted = feedback(beamss,-minreal(Yid/s),2,2);
  mc = bode(beamShunted(1,1),w);
  plot(w,squeeze(db(mc)),'-','linewidth',2)
end
set(gca,'xscale','log')
set(gca,'fontsize',14)
xlim([w(1),w(end)])
xlabel('Frequency (rad s^{-1})')
ylabel('Receptance (dB, ref. m N^{-1})')
box on
set(gca,'XTick',[5e2,1e3,2e3])



% Residues effect

hmaxT = zeros(2,length(r1));    % FRF amplitude at the effective open-circuit frequency
wschv = zeros(2,length(r1));    % Effective short-circuit frequency
wochv = zeros(2,length(r1));    % Effective open-circuit frequency

for ii = 1 : length(r1)
  % Admittance characteristics
  ri = [r1(ii) , 1-r1(ii)];
  [zi,zetai,wschv(:,ii),wochv(:,ii)] = admittanceSCOCFrequencies(wsc,r/Cpe,modes,ri,'parallel');

  % Ideal Norton admittance
  Yid = idealAdmittance(ri,zi,zetai,Cpe);

  % Shunted system
  beamShunted = feedback(beamss,-minreal(Yid/s),2,2);
  hmaxT(1,ii) = abs(freqresp(beamShunted(1,1),wochv(1,ii)));
  hmaxT(2,ii) = abs(freqresp(beamShunted(1,1),wochv(2,ii)));
  
end


% Effective EMCFs
Kcvv1 = sqrt((wochv(1,:)./wschv(1,:)).^2-1);
Kcvv2 = sqrt((wochv(2,:)./wschv(2,:)).^2-1);

% Predicted reduction
Adbv1 = ThomasReduction(Kcvv1,xi1);
Adbv2 = ThomasReduction(Kcvv2,xi2);


% Attenuation plot
figure
hold on
plot(r1Plot,db(hmaxT1./hmaxT(1,:)),'linewidth',2)
plot(r1Plot,Adbv1,'-.','linewidth',2)
plot(r1Plot,db(hmaxT2./hmaxT(2,:)),'linewidth',2)
plot(r1Plot,Adbv2,'-.','linewidth',2)
xlim([r1Plot(1),r1Plot(end)])
set(gca,'xscale','log')
set(gca,'fontsize',14)
xlabel('r_1 (-)')
ylabel('Attenuation (dB)')
set(gca,'XTick',rt./(1-rt))
set(gca,'XTickLabel',{'0.0001','0.5','0.9999'})
box on
