%
% Damping of five modes of a clamped-free beam multimodal circuits with
% ideal characteristics - comparison with the tuning from Fleming et al.
%
% See
%
% Fleming, A.J.,Behrens, S. and Moheimani, S.O.R. (2002)
% 'Optimization and implementation of multimode piezoelectric shunt damping
% systems',
% IEEE/ASME Transactions on Mechatronics, 7(1), pp 87-94. doi: 10.1109/3516.990891.
%
% Note: the gradients of the cost function are not evaluated analytically,
% leading to longer run times. These times should be considered only as
% rough estimates of the cost of optimization.
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Sep 2021


clear
addpath('./Optimization/')


% Beam model
load('BeamModel');

% Dynamic capacitance and elastance identification
[wsc,woc,Cpe,r] = dynamicCapacitanceResidues(1/beamss(2,2));
[woc2,wsc2,Epe,r2] = dynamicCapacitanceResidues(beamss(2,2));

% Laplace variable
s = zpk(0,[],1);

% Frequency vector
w = linspace(2e2,2.7e4,2e4);
w = w(:);

% Targeted modes
Nmodes = 5;
modes = 1:1:Nmodes;

% Residues
ri = ones(1,Nmodes)/Nmodes;
ri0 = ones(1,Nmodes+1)/(Nmodes+1);

%% Proposed approach

% Ideal Thevenin impedance for five modes
tic
[zi,zetai,wsch,woch] = impedanceSCOCFrequencies(woc2,-r2/Epe,modes,ri,'series');
Yzid = idealImpedance(ri,zi,zetai,Cpe);
tz = toc;

% Ideal Norton admittance for five modes
tic
[zi2,zetai2,wsch2,woch2] = admittanceSCOCFrequencies(wsc,r/Cpe,modes,ri,'parallel');
Yid = idealAdmittance(ri,zi2,zetai2,Cpe);
t = toc;

% Shunted beamsstems
beamssid = feedback(beamss,-minreal(Yid/s),2,2);
beamsszid = feedback(beamss,-minreal(Yzid/s),2,2);

% FRFs
m = bode(beamss(1,1),w);
mid = bode(beamssid(1,1),w);
mzid = bode(beamsszid(1,1),w);



%% Moheimani's tuning

% Initial tuning
Kc2 = (woc./wsc).^2-1;
zetaHag = 2*sqrt(Kc2./(1+Kc2).^3);

zetaInit1 = zeros(Nmodes,1);
zetaInit2 = zeros(Nmodes,1);

% Optimization - impedance
tic
for ii = 1 : Nmodes
  zetaInit1(ii) = singleModeImpedanceInit(beamss,modes(ii),ri(ii),woc,zetaHag(ii),Cpe);
end
zeta1Opt = dampingImpedanceOptimization(beamss,ri,woc(modes),zetaInit1,Cpe);
tmz = toc;

% Optimization - admittance
tic
for ii = 1 : Nmodes
  zetaInit2(ii) = singleModeAdmittanceInit(beamss,modes(ii),ri(ii),woc,zetaHag(ii),Cpe);
end
zeta2Opt = dampingAdmittanceOptimization(beamss,ri,woc(modes),zetaInit2,Cpe);
tm = toc;

% Shunts admittances
Yzmid = idealImpedance(ri,woc(modes),zeta1Opt,Cpe);
Ymid = idealAdmittance(ri,woc(modes),zeta2Opt,Cpe);

% Shunted systems
beamssmid = feedback(beamss,-minreal(Ymid/s),2,2);
beamsszmid = feedback(beamss,-minreal(Yzmid/s),2,2);

% FRFs
mmid = bode(beamssmid(1,1),w);
mzmid = bode(beamsszmid(1,1),w);


%% Plots - comparison


% Impedance-based approaches
figure('Units','normalized','Position',[0.1,0.25,0.8,0.5])
hold on
plot(w,db(w.*squeeze(m)),'-k')
plot(w,db(w.*squeeze(mzid)),'-','linewidth',2)
plot(w,db(w.*squeeze(mzmid)),'-.','linewidth',2)
box on
set(gca,'fontsize',14)
set(gca,'xscale','log')
xlabel('Frequency (rad s^{-1})')
ylabel('Mobility (dB, ref. m s^{-1} N^{-1})')
xlim([0 w(end)])
ylim([-40,60])


% Admittance-based approaches
figure('Units','normalized','Position',[0.1,0.25,0.8,0.5])
hold on
plot(w,db(w.*squeeze(m)),'-k')
plot(w,db(w.*squeeze(mid)),'-','linewidth',2)
plot(w,db(w.*squeeze(mmid)),'-.','linewidth',2)
box on
set(gca,'fontsize',14)
set(gca,'xscale','log')
xlabel('Frequency (rad s^{-1})')
ylabel('Mobility (dB, ref. m s^{-1} N^{-1})')
xlim([0 w(end)])
ylim([-40,60])