%
% Plot of the experimental results obtained with ideal Norton admittance
% and Thevenin impedance on a clamped-free piezoelectric beam
%
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Sep 2021

%% Results data

addpath('.\Experimental');

expFolder = '.\Experimental\Results\';
expFolderS = [expFolder,'Series\'];       % Impedance-based approach
expFolderP = [expFolder,'Parallel\'];     % Admittance-based approach

fineNames = {'r1-0-1_Fine','r1-0-5_Fine','r1-0-9_Fine'};
aNames = {'r1-0-01','r1-0-1','r1-0-2','r1-0-3','r1-0-4','r1-0-5',...
  'r1-0-6','r1-0-7','r1-0-8','r1-0-9','r1-0-99'};

% Short- and open-circuit FRFs
[frsc,hsc] = TestLabFRFData([expFolder,'SC']);
[froc,hoc] = TestLabFRFData([expFolder,'OC']);

% Locating peaks (same frequency vector for short- and open-circuit)
ind1sc = and(frsc>25,frsc<35);
ind2sc = and(frsc>125,frsc<160);
frscMode1 = frsc(ind1sc);
frscMode2 = frsc(ind2sc);

% Damping ratio (uncontrolled structure with short-circuited patches)
xi1 = halfPowerDamping(hsc(ind1sc),frscMode1);
xi2 = halfPowerDamping(hsc(ind2sc),frscMode2);

% Peak value and frequency (short circuit)
[h10,ind10] = max(abs(hsc(ind1sc)./(2*pi*frsc(ind1sc)).^2));
[h20,ind20] = max(abs(hsc(ind2sc)./(2*pi*frsc(ind2sc)).^2));

% Short-circuit frequencies estimation (from magnitude peak)
fsc1 = frscMode1(ind10);
fsc2 = frscMode2(ind20);

% Peak value and frequency (open circuit)
[~,ind102] = max(abs(hoc(ind1sc)./(2*pi*froc(ind1sc)).^2));
[~,ind202] = max(abs(hoc(ind2sc)./(2*pi*froc(ind2sc)).^2));

% Open-circuit frequencies estimation (from magnitude peak)
foc1 = frscMode1(ind102);
foc2 = frscMode2(ind202);

% EEMCFs
Kc1 = sqrt((foc1/fsc1)^2-1);
Kc2 = sqrt((foc2/fsc2)^2-1);

% Residues
r1v = [0.01,0.1:0.1:0.9,0.99];  % Experimental values
r1t = [0.1 0.5 0.9];            % Ticks



%% Impedance-based models

% Single-mode series RL shunts
[fr1,h1] = TestLabFRFData([expFolderS,'Mode1']);
[fr2,h2] = TestLabFRFData([expFolderS,'Mode2']);

% Locating peaks
ind1 = and(fr1>25,fr1<35);
ind2 = and(fr1>125,fr1<160);

% FRFs plot (Thevenin impedance)
figure
hold on
plot(froc,db(abs(hoc)./(2*pi*froc).^2),'-k')
plot(fr1(ind1),db(abs(h1(ind1))./(2*pi*fr1(ind1)).^2),'-','Color',0.7*[1,1,1],'linewidth',4)
plot(fr2(ind2),db(abs(h2(ind2))./(2*pi*fr2(ind2)).^2),'-','Color',0.7*[1,1,1],'linewidth',4)
for ii = 1 : 3
  [fr,h] = TestLabFRFData([expFolderS,fineNames{ii}]);
  plot(fr,db(abs(h)./(2*pi*fr).^2),'-')
end
xlim([20,250])
box on
set(gca,'xscale','log')
set(gca,'fontsize',14)
xlabel('Frequency (Hz)')
ylabel('Magnitude (dB, ref. m N^{-1})')
set(gca,'XTick',[5e1,1e2,2e2])


% Residues influence tests
hm1 = zeros(length(aNames),1);    % Maximum magnitude around mode 1
hm2 = zeros(length(aNames),1);    % Maximum magnitude around mode 2
for ii = 1 : length(aNames)
  [fr,h] = TestLabFRFData([expFolderS,aNames{ii}]);
  hm1(ii) = max(abs(h(ind1)./(2*pi*fr(ind1)).^2));
  hm2(ii) = max(abs(h(ind2)./(2*pi*fr(ind2)).^2));
end

% Theoretical attenuation
r1v2 = linspace(1e-2,0.99,1e3);
Adb1 = ThomasReduction(sqrt(r1v2)*Kc1,xi1);
Adb2 = ThomasReduction(sqrt(1-r1v2)*Kc2,xi2);

% Attenuation plot (Thevenin impedance)
figure
hold on
plot(r1v2./(1-r1v2),Adb1,'-','linewidth',2)
plot(r1v2./(1-r1v2),Adb2,'-','linewidth',2)
set(gca,'xscale','log')
plot(r1v./(1-r1v),db(h10./hm1),'o-')
plot(r1v./(1-r1v),db(h20./hm2),'o-')
set(gca,'XTick',r1t./(1-r1t))
set(gca,'XTickLabel',r1t)
box on
set(gca,'fontsize',14)
xlabel('r_1 (-)')
ylabel('Attenuation (dB)')




%% Admittance-based models

% Single-mode parallel RL shunts
[fr1,h1] = TestLabFRFData([expFolderP,'Mode1']);
[fr2,h2] = TestLabFRFData([expFolderP,'Mode2']);

% Locating peaks
ind1 = and(fr1>25,fr1<35);
ind2 = and(fr1>125,fr1<160);

% FRFs plot (Norton admittance)
figure
hold on
plot(froc,db(abs(hoc)./(2*pi*froc).^2),'-k')
plot(fr1(ind1),db(abs(h1(ind1))./(2*pi*fr1(ind1)).^2),'-','Color',0.7*[1,1,1],'linewidth',4)
plot(fr2(ind2),db(abs(h2(ind2))./(2*pi*fr2(ind2)).^2),'-','Color',0.7*[1,1,1],'linewidth',4)
for ii = 1 : 3
  [fr,h] = TestLabFRFData([expFolderP,fineNames{ii}]);
  plot(fr,db(abs(h)./(2*pi*fr).^2),'-')
end
xlim([20,250])
set(gca,'xscale','log')
box on
set(gca,'xscale','log')
set(gca,'fontsize',14)
xlabel('Frequency (Hz)')
ylabel('Magnitude (dB, ref. m N^{-1})')
set(gca,'XTick',[5e1,1e2,2e2])


% Residues influence tests
hm1 = zeros(length(aNames),1);    % Maximum magnitude around mode 1
hm2 = zeros(length(aNames),1);    % Maximum magnitude around mode 2
for ii = 1 : length(aNames)
  [fr,h] = TestLabFRFData([expFolderP,aNames{ii}]);
  hm1(ii) = max(abs(h(ind1)./(2*pi*fr(ind1)).^2));
  hm2(ii) = max(abs(h(ind2)./(2*pi*fr(ind2)).^2));
end

% Attenuation plot (Norton admittance)
figure
hold on
plot(r1v2./(1-r1v2),Adb1,'-','linewidth',2)
plot(r1v2./(1-r1v2),Adb2,'-','linewidth',2)
set(gca,'xscale','log')
plot(r1v./(1-r1v),db(h10./hm1),'o-')
plot(r1v./(1-r1v),db(h20./hm2),'o-')
set(gca,'XTick',r1t./(1-r1t))
set(gca,'XTickLabel',r1t)
box on
set(gca,'fontsize',14)
xlabel('r_1 (-)')
ylabel('Attenuation (dB)')
