%
% Identification of a dynamic capacitance / elastance characteristics
% C = k + sum( r/(s^2 + p^2) )
%
% Input:
%   * C: dynamic capacitance / elastance model object
%
% Output:
%   * p: poles of the dynamic capacitance / elastance
%   * z: zeros of the dynamic capacitance / elastance
%   * k: feedthrough term (capacitance / elastance at constant strain)
%   * r: residues associated with the poles
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Sep 2021
%
function [p,z,k,r] = dynamicCapacitanceResidues(C)

  % Dynamic capacitance data
  C = zpk(C);
  [zz,pp,kk] = zpkdata(C,'v');

  % Short- and open-circuit resonance frequencies
  z = sort(imag(zz(imag(zz)>0)));
  p = sort(imag(pp(imag(pp)>0)));
  
  % Capacitance at constant strain
  k = -kk;
  
  % Residual capacitance
  r = zeros(size(p));
  for ii = 1 : length(r)
    indi = setdiff(1:1:length(r),ii);
    if ~isempty(indi)
      r(ii) = -real(evalfr(zpk(kron([1i;-1i],z),kron([1i;-1i],p(indi)),kk),-1i*p(ii)));
    else
      r(ii) = -real(evalfr(zpk(zz,[],kk),-1i*p(ii)));
    end
  end
  
end