%
% Characteristics of an ideal Norton admittance for multimodal
% piezoelectric shunt damping
%
% Input:
%   * wsc: short-circuit resonance frequencies
%   * gpc: modal coupling vector (Phi_sc^T*gamma_p)
%   * modes: modes indices to be controlled
%   * r: residues associated with the targeted modes
%   * shuntType: baseline case 'series' or 'parallel'
%
% Output:
%   * z: zeros frequencies
%   * zetai: electrical damping ratios
%   * wsch: effective short-circuit resonance frequencies
%   * woch: effective open-circuit resonance frequencies
%
% (c) G. Raze, J. Dietrich and G. Kerschen
% Sep 2021
%
function [z,zetai,wsch,woch] = admittanceSCOCFrequencies(wsc,gpc,modes,r,...
  shuntType)

  wsc = wsc(:);
  gpc = gpc(:);
  modes = modes(:);
  r = r(:);
  
  if length(r) > length(modes)
    r0 = r(1);
    r = r(2:end);
  else
    r0 = 0;
  end

  % Linearized approximation
  wsch = zeros(size(modes));
  woch = zeros(size(modes));
  z = zeros(size(modes));
  zetai = zeros(size(modes));
  
  for ii = length(modes) : -1 : 1
    indMechSup = (modes(ii)+1) : length(gpc);
    indInfSC = 1:(ii-1);
    indInfOC = 1:ii;
    indSup = (ii+1):length(modes);
    
    % Residual contributions
    kr = gpc(modes(ii))/wsc(modes(ii))^2;
    knr = sum(gpc(indMechSup)./wsc(indMechSup).^2);
    flSC = r0 + sum(r(indInfSC));
    flOC = r0 + sum(r(indInfOC));
    fh = sum(wsc(modes(ii))^2*r(indSup)./z(indSup).^2);
    
    % Quadratic problem coefficients
    aSC = 1 + (kr+knr)*flSC;
    aOC = 1 + (kr+knr)*flOC;
    bSC = 1 + (kr+knr)*fh + knr*flSC;
    bOC = 1 + (kr+knr)*fh + knr*flOC;
    c = knr*fh;
    
    % Effective SC and OC frequencies
    wsch(ii) = wsc(modes(ii))/sqrt(bSC/(2*aSC) + sqrt((bSC/(2*aSC))^2-c/aSC));
    woch(ii) = wsc(modes(ii))/sqrt(bOC/(2*aOC) + sqrt((bOC/(2*aOC))^2-c/aOC));
    
    % Zero of the undamped admittance
    [~,~,zetai(ii),delta] = ikegameRLShunt(1,wsch(ii),woch(ii),shuntType);
    if strcmp(shuntType,'series')
      we = delta*woch(ii);
    else
      we = delta*wsch(ii);
    end
    z(ii) = we * sqrt(1 + r(ii) / (1/knr + flSC - fh*we^2/wsc(modes(ii))^2));
  end
  
end