% Pole placement optimization of the damping ratios of single-mode shunts
% Admittance-based approach
%
% See
%
% Fleming, A.J.,Behrens, S. and Moheimani, S.O.R. (2002)
% 'Optimization and implementation of multimode piezoelectric shunt damping
% systems',
% IEEE/ASME Transactions on Mechatronics, 7(1), pp 87-94. doi: 10.1109/3516.990891.
%
% Sep 2021
%
function zeta = singleModeAdmittanceInit(sys,mode,ri,woc,zetai,Cpe)
    
  zeta = fminunc(@(zetai) admittanceObjFun(sys,mode,ri,woc,zetai,Cpe),zetai,...
    optimoptions(@fminunc,'OptimalityTolerance',1e-9,'StepTolerance',1e-9,...
    'display','none'));


  function f = admittanceObjFun(sys,mode,ri,woc,zetai,Cpe)
    s = zpk(0,[],1);
    Yid = idealAdmittance(ri,woc(mode),zetai,Cpe);
    sysid = feedback(sys,-minreal(Yid/s),2,2);
    pp = pzmap(sysid);
    pp(imag(pp)<0) = [];
    [~,ind] = sort(abs(pp-1i*woc(mode)));
    pp = pp(ind(1:2));
    f = max(real(pp)./abs(pp));
  end

end