% Supplementary materials of the article
%
% Raze, G., Jadoul, A., Guichaux, S., Broun, V. and Kerschen, G., 2019. 
% A digital nonlinear piezoelectric tuned vibration absorber. 
% Smart Materials and Structures, 29(1), p.015007.
% doi: 10.1088/1361-665X/ab5176
% https://hdl.handle.net/2268/242115
%
% Test the principle of similarity with quadratic and quintic nonlinear
% capacitance in the NPTVA (Figures 18 and 19)

%% Data

% Sampling frequency
fs = 1600;

% Filter
bpFilt = designfilt('bandpassiir','DesignMethod','butter',...
         'PassBandFrequency1',25,'PassBandFrequency2',35, ...
         'StopBandFrequency1',15,'StopBandFrequency2',50,...
         'StopbandAttenuation1',40, 'StopbandAttenuation2',40,... 
         'SampleRate',fs);

filesA = './Data/Uncontrolled/SC_0-2N_a';
filesF = './Data/Uncontrolled/SC_0-2N_f';
load('SystemParameters')

[t,x,xdot,xddot,f,lims] = beamSignalProcessing(filesA,filesF,bpFilt);

% Least-squares fit of the stiffness curve
X = [xddot,xdot,x];
fnl = f - m*xddot - c*xdot - ksc*x;
tmax = 833;    % Time @ jump-down, the NL force looks bad after that
[tr,outr] = splineRoots(t(t<tmax)',X(t<tmax,2)',[X(t<tmax,3)';(f(t<tmax)-X(t<tmax,:)*[m;c;ksc])']);

k2 = pinv([sign(outr(1,:))'.*outr(1,:)'.^2])*(outr(2,:)');
k5 = pinv([outr(1,:)'.^5])*(outr(2,:)');

[~,indTmp] = sort(outr(1,:));


%% Plot fits

% Quadratic fit
figure
hold on
plot(outr(1,:),outr(2,:),'.k','markersize',18)
plot(outr(1,indTmp),k2*sign(outr(1,indTmp)).*outr(1,indTmp).^2,'-.r','linewidth',2)
box on
set(gca,'fontsize',16)
xlim([-1e-3,1e-3])
ylim([-3,3])
xlabel('x (m)')
ylabel('f_{nl} (N)')
axis normal

% Quintic fit
figure
hold on
plot(outr(1,:),outr(2,:),'.k','markersize',18)
plot(outr(1,indTmp),k5*outr(1,indTmp).^5,'-.r','linewidth',2)
box on
set(gca,'fontsize',16)
xlim([-1e-3,1e-3])
ylim([-3,3])
xlabel('x (m)')
ylabel('f_{nl} (N)')
axis normal

%% Nonlinear FRF

% Data - quadratic NPTVA
nonlinear_FRF_files = {'./Data/Pos/Quadratic/0-2N_FRF',...
  './Data/Pos/Quadratic/0-4N_FRF',...
  './Data/Pos/Quadratic/0-6N_FRF',...
  './Data/Pos/Quadratic/0-8N_FRF',...
  './Data/Pos/Quadratic/1-0N_FRF'};

% Plot - quadratic NPTVA
figure
hold on
for ii = 1 : length(nonlinear_FRF_files)
  dataFile = nonlinear_FRF_files{ii};
  [fr,y] = TestLabFRFData(dataFile);

  plot(fr,abs(y./(2*pi*fr).^2),'-','linewidth',2)
end
xlim([20 40])
xlabel('Frequency (Hz)')
ylim([0 2e-3])
ylabel('|x/f| (m/N)')
set(gca,'fontsize',16)
box on



% Data - quintic NPTVA
nonlinear_FRF_files = {'./Data/Pos/Quintic/0-2N_FRF',...
  './Data/Pos/Quintic/0-4N_FRF',...
  './Data/Pos/Quintic/0-6N_FRF',...
  './Data/Pos/Quintic/0-8N_FRF'};


% Plot - quintic NPTVA
figure
hold on
for ii = 1 : length(nonlinear_FRF_files)
  dataFile = nonlinear_FRF_files{ii};
  [fr,y] = TestLabFRFData(dataFile);

  plot(fr,abs(y./(2*pi*fr).^2),'-','linewidth',2)
end
xlim([20 40])
xlabel('Frequency (Hz)')
ylim([0 2e-3])
ylabel('|x/f| (m/N)')
set(gca,'fontsize',16)
box on
