% Supplementary materials of the article
%
% Raze, G., Jadoul, A., Guichaux, S., Broun, V. and Kerschen, G., 2019. 
% A digital nonlinear piezoelectric tuned vibration absorber. 
% Smart Materials and Structures, 29(1), p.015007.
% doi: 10.1088/1361-665X/ab5176
% https://hdl.handle.net/2268/242115
%
% Compute the parameters of the uncontrolled systems (Table 1) and the
% NPTVA parameters (Table 2)


Cpe = 250e-9;   % Piezoelectric capacitance (F)
fs = 1600;      % Sampling frequency (Hz)

%% Filter design
bpFilt = designfilt('bandpassiir','DesignMethod','butter',...
         'PassBandFrequency1',25,'PassBandFrequency2',35, ...
         'StopBandFrequency1',15,'StopBandFrequency2',50,...
         'StopbandAttenuation1',40, 'StopbandAttenuation2',40,... 
         'SampleRate',fs);
    


%% Linear short-circuit

% Load results
[t,x,xdot,xddot,f] = beamSignalProcessing('./Data/Uncontrolled/SC_0-01N_a','./Data/Uncontrolled/SC_0-01N_f',bpFilt);
X = [xddot,xdot,x];

% Least-squares fit
param_sc = (X.'*X)\(X.'*f);
msc = param_sc(1);
csc = param_sc(2);
ksc = param_sc(3);

% The parameters are polished with a nonlinear least-squares optimization
[fr,FRF] = TestLabFRFData('./Data/Uncontrolled/FRF_SC.mat');
objFun = @(xi) sum(abs(squeeze(freqresp(tf([1,0,0],[xi(1),xi(2),xi(3)]),2*pi*fr(and(fr>20,fr<40))))+FRF(and(fr>20,fr<40))));
xiOptsc = fminunc(objFun,[msc,csc,ksc],optimoptions(@fminunc,'display','none'));



omsc = sqrt(ksc/msc);

% Plot fit
fitsc = ((X*param_sc-f).'*(X*param_sc-f))/(f.'*f);
figure
wv = linspace(2*pi*20,2*pi*40,1000);
hold on
plot(wv/2/pi,abs(squeeze(freqresp(tf(1,[xiOptsc(1),xiOptsc(2),xiOptsc(3)]),wv))),'-k','linewidth',2)
plot(fr,abs(FRF)./((2*pi*fr).^2),'-.r','linewidth',2)


msc = xiOptsc(1);
csc = xiOptsc(2);
ksc = xiOptsc(3);

%% Linear open-circuit results

% Load results
[t,x,xdot,xddot,f] = beamSignalProcessing('./Data/Uncontrolled/OC_0-01N_a','./Data/Uncontrolled/OC_0-01N_f',bpFilt);
X = [xddot,xdot,x];

% Least-squares fit
param_oc = (X.'*X)\(X.'*f);
moc = param_oc(1);
coc = param_oc(2);
koc = param_oc(3);
fitoc = ((X*param_oc-f).'*(X*param_oc-f))/(f.'*f);

% The parameters are polished with a nonlinear least-squares optimization
[fr,FRF] = TestLabFRFData('./Data/Uncontrolled/FRF_OC.mat');
objFun = @(xi) sum(abs(squeeze(freqresp(tf([1,0,0],[xi(1),xi(2),xi(3)]),2*pi*fr(and(fr>20,fr<40))))+FRF(and(fr>20,fr<40))));
xiOptoc = fminunc(objFun,[moc,coc,koc],optimoptions(@fminunc,'display','none'));


% Plot fit
hold on
plot(wv/2/pi,abs(squeeze(freqresp(tf(1,[xiOptsc(1),xiOptsc(2),xiOptoc(3)]),wv))),'-','color',[0.4 0.4 0.4],'linewidth',2)
plot(fr,abs(FRF)./((2*pi*fr).^2),'-.b','linewidth',2)
xlim([25 35])
box on
set(gca,'fontsize',16)
xlabel('Frequency (Hz)')
ylabel('|x/f| (m/N)')


moc = xiOptoc(1);
coc = xiOptoc(2);
koc = xiOptoc(3);
omoc = sqrt(koc/moc);

%% Summary

% Piezoelectric structure characteristics
m = msc;
c = csc;
koc = omoc^2*m;
Kc = sqrt(omoc^2/omsc^2-1);
theta = sqrt((koc-ksc)/Cpe);


disp(' ')
disp('===========================================')
disp('Fit - linear parameters')
disp(' ')
disp(['Short-circuit resonance frequency: ',num2str(omsc/2/pi,'%5.3f'),' Hz'])
disp(['Open-circuit resonance frequency: ',num2str(omoc/2/pi,'%5.3f'),' Hz'])
disp(['Electromechanical coupling factor: ',num2str(Kc*100,'%5.2f'),'%'])
disp(' ')
disp(['Relative discrepancy in mass :',num2str(abs(moc-m)/m*100,'%5.2f'),'%'])
disp(['Relative discrepancy in damping :',num2str(abs(coc-c)/c*100,'%5.2f'),'%'])
disp(['LSE, short-circuit :',num2str(fitsc*100,'%5.2f'),'%'])
disp(['LSE, open-circuit :',num2str(fitoc*100,'%5.2f'),'%'])
disp(' ')

%% Nonlinear short-circuit

% Load results
filesA = './Data/Uncontrolled/SC_0-2N_a';
filesF = './Data/Uncontrolled/SC_0-2N_f';
[t,x,xdot,xddot,f,lims] = beamSignalProcessing(filesA,filesF,bpFilt);
X = [xddot,xdot,x];


% Stiffness curve
tmax = 833;    % Time @ jump-down, the NL force looks bad after that
[tr,outr] = splineRoots(t(t<tmax)',X(t<tmax,2)',[X(t<tmax,3)';(f(t<tmax)-X(t<tmax,:)*[m;c;ksc])']);

% Least-squares fit
ktmp = pinv([outr(1,:)',outr(1,:)'.^3])*(outr(2,:)');   % Linear characteristics are also corrected
k3 = 4/3*ktmp(2);       % Filtered force vs. (harmonic amplitude)^3 <-> coefficient

% Plot results
figure
hold on
[~,indTmp] = sort(outr(1,:));
plot(outr(1,:),4/3*(outr(2,:)-ktmp(1)*outr(1,:)),'.k')
plot(outr(1,indTmp),4/3*ktmp(2)*outr(1,indTmp).^3,'-.r','linewidth',2)
xlabel('x (m)')
ylabel('f_{nl} (N)')
box on
set(gca,'fontsize',16)
xlim([-1e-3,1e-3])
ylim([-3,3])
axis normal

%% Shunt parameters

% Piezoelectric structure characteristics
ksc = ksc + ktmp(1);
koc = koc + ktmp(1);
omsc = sqrt(ksc/m);
omoc = sqrt(koc/m);
Kc = sqrt(omoc^2/omsc^2-1);
theta = sqrt((koc-ksc)/Cpe);

% Soltani et al
Kc2 = Kc^2;
r = (sqrt(64-16*Kc2-26*Kc2^2)-Kc2)/8;
delta = sqrt((8+3*Kc2-4*r)/(Kc2+1))/2;
zeta = 2*sqrt(64+80*Kc2+27*Kc2^2 - 16*r*(4+3*Kc2))/(sqrt(2)*(8+5*Kc2))/delta;

% NPTVA characteristics
R = zeta/(omoc*Cpe);
L =  1/(delta^2*omoc^2*Cpe);
Cnl = 2*L^2/m^2*k3;               % NB: Cnl is not exactly the same as the value in the paper because it was tuned with a slightly more accurate method therein

% Save system parameters
save('SystemParameters','m','c','ksc','koc','theta','L','R','Cpe','k3','Cnl')
