% Compute and filter time data for the beam
% 
% Input:
%   * fileA, fileF: path to the .mat files from TestLab containing the time
%   series of the acceleration and force, respectively
%     They can also be a cell to process multiple files at once (fileA =
%     {'file1.mat','file2.mat',...})
%   * filt (optional): filter
% Output:
%   * t: time
%   * x,xdot,xddot: displacement, velocity and acceleration
%   * f: force
%   * lims: limits for the data in the different files
%
function [t,x,xdot,xddot,f,lims] = beamSignalProcessing(fileA,fileF,filt)
  lims = [];
  if ~iscell(fileA)
      [t,x,xdot,xddot,f] = processFiles(fileA,fileF,filt);
  else
    t = []; x = []; xdot = []; xddot = []; f =[];
    for ii = 1 : length(fileA)
      [tp,xp,xdotp,xddotp,fp] = processFiles(fileA{ii},fileF{ii},filt);
      t = [t;tp];
      x = [x;xp];
      xdot = [xdot;xdotp];
      xddot = [xddot;xddotp];
      f = [f;fp];
      if ii == 1
        lims = [1;length(tp)];
      else
        lims = [lims ; lims(ii-1) + length(tp)];
      end
    end
  end

  function [t,x,xdot,xddot,f] = processFiles(fileA,fileF,filt)
  % Acceleration measurements
  load(fileA,'Signal');
  t = Signal.x_values.start_value + Signal.x_values.increment*(0:1:(Signal.x_values.number_of_values-1)).';
  xddot = Signal.y_values.values;
  
  % Force measurements
  load(fileF,'Signal');
  f = Signal.y_values.values;
  
  % Time integration
  xdot = cumtrapz(t,xddot);
  x = cumtrapz(t,xdot);
  
  
  % Filtering
   if nargin > 2 && ~isempty(filt)
    x = filtfilt(filt,x);
    xdot = filtfilt(filt,xdot);
    xddot = filtfilt(filt,xddot);
    f = -filtfilt(filt,f);
   end
  end
end