clc
clear all

%% Initialisation of a simple suspension bridge

% Bridge Geometry
LLS = 500; % m - Length of the left span
LRS = 500; % m - Length of the right span
LMS = 1600; % m - Length of the main span
H = 200; % m - Height of the towers
HM = 5; % m - Height of main cable
SH = 20; % m - Spacing between hangers

NdivSH = 1; % Number of elements between two hangers 
NdivH = 10; % Number of elements for the tower

SideCables = 1; % 1 is for parabolic main cable for the spans; 2 is for straight main cable for the spans
SideHangers = 1; % 1 if there are hangers in side spans; 2 if there are not

% Material and geometric charactristics of elements
[Materials, Geometries] = MatGeom; 

% Aerodynamic characteristics (Normandie Bridge)
Cl0 = 0.2; % Lift coefficient
dCl0 = 0.08*60; % Derivative of lift coefficient with respect to the incidence angle
B = 30; % m - Width of the deck
ksi = 0.02; % Damping ratio (same for all modes)

% Wind parameters (Great Belt Bridge)
Umoy = 40; % m/s - Mean wind speed
lux = 200; % m - Length
Coh = 8; % Coherence coefficient
lwx = 50; % m - Length
Iu = 0.1; % Turbulence intensity along the main wind direction
Iw = 0.06; % Turbulence intensity along the vertical axis
rho = 1.1; % kg/m^3 - Air density
sigu = Iu*Umoy; % m/s - Standard deviation of main wind speed
sigw = Iw*Umoy; % m/s - Standard deviation of vertical wind speed

% Parameters for the following computations
trunc1 = 30; % Number of modes kept after the modal truncation
trunc2 = 30; % Number of modes kept for the figures

%% Nodes and Elements

[Nodes, Elements, NNode, NElem, SUPP, SBCR] = NodesElements(LLS, LRS, LMS, H, SH, NdivSH, NdivH, HM, SideCables, SideHangers); % Adjust cable tension NHOR (line 202) to obtain adequate cable stiffness

NDOF = NNode*3; % Number of DOFs

%% Nodal Stiffness and Mass Matrices

[KGapp,Mapp,Elements,CORRES] = Assemble(NNode,NElem,Nodes,Elements,Geometries,Materials, SUPP);

NFree = length(CORRES); % Number of free DOFs
KGapp = (KGapp+KGapp')/2;
Mapp = (Mapp+Mapp')/2;

%% Natural frequencies, mode shapes and truncated generalized matrices

[omgrs, order, Mst, Kst, VVst] = NatPulsMatGen(KGapp, Mapp, trunc1);

Cst = ksi*2*sqrt(Kst.*Mst); % Modal damping matrix

frs = omgrs/(2*pi); % Hz - Natural frequencies

%% Modal wind load matrices

Ltrib = SH/NdivSH;
Abu = zeros(NDOF, NDOF);
Abw = zeros(NDOF, NDOF);

for i = 1:SBCR % Vertical wind loads on the deck only
    Abu(i*3-1,i*3-1) = B*Ltrib*2*Cl0;
    Abw(i*3-1,i*3-1) = B*Ltrib*dCl0;
end

CAbu = conj(Abu);
CAbw = conj(Abw);
TAbu = Abu.';
TAbw = Abw.';

%% Computation of modal response covariances by integration
tic

N = 100*trunc1;
omg = linspace(0, omgrs(trunc1), N);
domg = omg(end)-omg(end-1);
    
Su = PSDiVK(Umoy, lux, sigu, omg);
Sw = PSDiVK(Umoy, lwx, sigw, omg);

XNOD = Nodes.XNOD;
YNOD = Nodes.YNOD;

Covq1 = zeros(trunc1, trunc1);
Covf = zeros(trunc1, trunc1);
PSDuu = zeros(NDOF, NDOF);
PSDuw = zeros(NDOF, NDOF);
PSDww = zeros(NDOF, NDOF);
PSDf = zeros(NDOF,NDOF);
PSDq = zeros(trunc1, trunc1);
Hst = zeros(trunc1,trunc1);

for k = 2:N-1
    for i = 1:SBCR 
        for j = 1:SBCR 
            dx = sqrt((XNOD(j)-XNOD(i))^2+(YNOD(j)-YNOD(i))^2);
            PSDuu(i*3-1,j*3-1) = Su(k)*exp(-Coh*omg(k)*dx/(4*pi*Umoy));
            PSDuw(i*3-1,j*3-1) = sqrt(Su(k)*Sw(k))*exp(-Coh*omg(k)*dx/(4*pi*Umoy));
            PSDww(i*3-1,j*3-1) = Sw(k)*exp(-Coh*omg(k)*dx/(4*pi*Umoy));
        end
    end
    
    PSDf = 1/4*rho^2*Umoy^2*(CAbu*PSDuu*TAbu+CAbu*PSDuw*TAbw+CAbw*PSDuw*TAbu+CAbw*PSDww*TAbw);
    PSDf = PSDf(CORRES,CORRES);
    PSDfst = VVst'*PSDf*VVst;
    
    Hst = diag(diag((-Mst*omg(k)^2+1i*omg(k)*Cst+Kst)^(-1)));
    PSDq = Hst*PSDfst*Hst';
          
    Covq1 = Covq1+2*domg*real(PSDq);
    Covf = Covf+2*domg*real(PSDfst);
end

Covq1 = Covq1(1:trunc2, 1:trunc2);
Covf = Covf(1:trunc2, 1:trunc2);

rhoq1 = inv(diag(sqrt(diag(Covq1))))*Covq1*inv(diag(sqrt((diag(Covq1)))));
rhof = inv(diag(sqrt(diag(Covf))))*Covf*inv(diag(sqrt((diag(Covf)))));

toc
%% Computation of modal response variances with MTSA
tic

Sfii = zeros(trunc2,1);
Bv = zeros(trunc2,1);
Rv = zeros(trunc2,1);
BRv = zeros(trunc2,1);
Varq = zeros(trunc2,1);

for k = 1:trunc2
    
    omgk = omgrs(k);
    
    for i = 1:SBCR
        for j = 1:SBCR
            Suk = PSDiVK(Umoy, lux, sigu, omgk);
            Swk = PSDiVK(Umoy, lwx, sigw, omgk);
            dx = sqrt((XNOD(j)-XNOD(i))^2+(YNOD(j)-YNOD(i))^2);
            PSDuu(i*3-1,j*3-1) = Suk*exp(-Coh*omgk*dx/(4*pi*Umoy));
            PSDuw(i*3-1,j*3-1) = sqrt(Suk*Swk)*exp(-Coh*omgk*dx/(4*pi*Umoy));
            PSDww(i*3-1,j*3-1) = Swk*exp(-Coh*omgk*dx/(4*pi*Umoy));
        end
    end
    
    PSDf = 1/4*rho^2*Umoy^2*(CAbu*PSDuu*TAbu+CAbu*PSDuw*TAbw+CAbw*PSDuw*TAbu+CAbw*PSDww*TAbw);
    PSDf = PSDf(CORRES,CORRES);

    Sfii(k) = VVst(:,k)'*PSDf(:,:)*VVst(:,k);

    Bv(k) = Covf(k,k)/(Kst(k,k)^2);
    Rv(k) = Sfii(k)*pi*omgrs(k)/(2*ksi*Kst(k,k)^2);
    BRv(k) = Bv(k)/Rv(k);
    
    Varq(k) = Bv(k)+Rv(k);

end

toc
%% Computation of modal response covariances with MTSA
tic

Covq2 = zeros(trunc2, trunc2);
Sfmn = zeros(trunc2,trunc2);
eps = zeros(trunc2,trunc2);
Bc2 = zeros(trunc2,1);
Rc2 = zeros(trunc2,1);

for m = 1:trunc2
    for n = 1:trunc2
        
        omgk = (omgrs(m)+omgrs(n))/2;
        
        for i = 1:SBCR
            for j = 1:SBCR
                Suk = PSDiVK(Umoy, lux, sigu, omgk);
                Swk = PSDiVK(Umoy, lwx, sigw, omgk);
                dx = sqrt((XNOD(j)-XNOD(i))^2+(YNOD(j)-YNOD(i))^2);
                PSDuu(i*3-1,j*3-1) = Suk*exp(-Coh*omgk*dx/(4*pi*Umoy));
                PSDuw(i*3-1,j*3-1) = sqrt(Suk*Swk)*exp(-Coh*omgk*dx/(4*pi*Umoy)); 
                PSDww(i*3-1,j*3-1) = Swk*exp(-Coh*omgk*dx/(4*pi*Umoy));
            end
        end
        
        PSDf = 1/4*rho^2*Umoy^2*(CAbu*PSDuu*TAbu+CAbu*PSDuw*TAbw+CAbw*PSDuw*TAbu+CAbw*PSDww*TAbw);
        PSDf = PSDf(CORRES,CORRES);
        
        Sfmn(m,n) = VVst(:,m)'*PSDf(:,:)*VVst(:,n);

        eps(m,n) = (omgrs(m)-omgrs(n))/(omgrs(m)+omgrs(n));
        
        Gamma2b(m,n) = Sfmn(m,n);
        
        Bc2(m,n) = Covf(m,n)/(Kst(m,m)*Kst(n,n));
        Rc2(m,n) = 1/(Kst(m,m)*Kst(n,n))*real(Sfmn(m,n)*(omgrs(m)+omgrs(n))/2*(ksi-1i*eps(m,n))/(ksi^2+eps(m,n)^2)*pi/2);

        Covq2(m,n) = Bc2(m,n)+Rc2(m,n);
       
    end
end

rhoq2 = inv(diag(sqrt(diag(Covq2))))*Covq2*inv(diag(sqrt((diag(Covq2)))));

toc
%% Computation of modal response covariances with MTSA and approx S((m+n)/2) = (S(m)+S(n))/2
tic

Covq3 = zeros(trunc2, trunc2);
Sfmnk = zeros(trunc2,trunc2,trunc2);
Bc3 = zeros(trunc2,1);
Rc3 = zeros(trunc2,1);

for k = 1:trunc2
    
    omgk = omgrs(k);
    
    for i = 1:SBCR
        for j = 1:SBCR
            Suk = PSDiVK(Umoy, lux, sigu, omgk);
            Swk = PSDiVK(Umoy, lwx, sigw, omgk);
            dx = sqrt((XNOD(j)-XNOD(i))^2+(YNOD(j)-YNOD(i))^2);
            PSDuu(i*3-1,j*3-1) = Suk*exp(-Coh*omgk*dx/(4*pi*Umoy));
            PSDuw(i*3-1,j*3-1) = sqrt(Suk*Swk)*exp(-Coh*omgk*dx/(4*pi*Umoy)); 
            PSDww(i*3-1,j*3-1) = Swk*exp(-Coh*omgk*dx/(4*pi*Umoy));
        end
    end
    
    PSDf = 1/4*rho^2*Umoy^2*(CAbu*PSDuu*TAbu+CAbu*PSDuw*TAbw+CAbw*PSDuw*TAbu+CAbw*PSDww*TAbw);
    PSDf = PSDf(CORRES,CORRES);
    PSDfst = VVst(:,1:trunc2)'*PSDf*VVst(:,1:trunc2);
    
    Sfmnk(:,:,k) = PSDfst;
    
end
toc

tic
for m = 1:trunc2
    for n = 1:trunc2
        
        Gamma3b(m,n) = (Sfmnk(m,n,m)+Sfmnk(m,n,n))/2;
        
        Bc3(m,n) = Covf(m,n)/(Kst(m,m)*Kst(n,n));
        Rc3(m,n) = 1/(Kst(m,m)*Kst(n,n))*real(Gamma3b(m,n)*(omgrs(m)+omgrs(n))/2*(ksi-1i*eps(m,n))/(ksi^2+eps(m,n)^2)*pi/2);
        
        Covq3(m,n) = Bc3(m,n)+Rc3(m,n);
        
    end
end

rhoq3 = inv(diag(sqrt(diag(Covq3))))*Covq3*inv(diag(sqrt((diag(Covq3)))));

toc
%% Computation of modal response covariances with MTSA and approx S((m+n)/2) = sqrt(S(m)S(n))
tic

Covq4 = zeros(trunc2, trunc2);
Bc4 = zeros(trunc2,1);
Rc4 = zeros(trunc2,1);

for m = 1:trunc2
    for n = 1:trunc2
        
        Gamma4b(m,n) = sqrt(Sfmnk(m,n,m)*Sfmnk(m,n,n))*sign(Sfmnk(m,n,m));
        
        Bc4(m,n) = Covf(m,n)/(Kst(m,m)*Kst(n,n));
        Rc4(m,n) = 1/(Kst(m,m)*Kst(n,n))*real(Gamma4b(m,n)*(omgrs(m)+omgrs(n))/2*(ksi-1i*eps(m,n))/(ksi^2+eps(m,n)^2)*pi/2);
        
        Covq4(m,n) = Bc4(m,n)+Rc4(m,n);
        
    end
end

rhoq4 = inv(diag(sqrt(diag(Covq4))))*Covq4*inv(diag(sqrt((diag(Covq4)))));
toc

%% Computation of modal response correlation coefficients

for m = 1:trunc2
    for n = 1:trunc2
        gb(m,n) = 1/(sqrt(1+1/BRv(m))*sqrt(1+1/BRv(n)));
        gr(m,n) = 1/(sqrt(1+BRv(m))*sqrt(1+BRv(n)));
        phi(m,n) = (omgrs(m)+omgrs(n))*ksi*(ksi-1i*eps(m,n))/(2*sqrt(omgrs(m)*omgrs(n))*(eps(m,n)^2+ksi^2));
        den(m,n) = sqrt(real(Sfii(m))*real(Sfii(n)));
        Gamma2c(m,n) = Gamma2b(m,n)*1/den(m,n);
        Gamma3c(m,n) = Gamma3b(m,n)*1/den(m,n);
        Gamma4c(m,n) = Gamma4b(m,n)*1/den(m,n);
        
        rhoq2b(m,n) = gb(m,n)*rhof(m,n)+gr(m,n)*real(Gamma2c(m,n)*phi(m,n));
        rhoq3b(m,n) = gb(m,n)*rhof(m,n)+gr(m,n)*real(Gamma3c(m,n)*phi(m,n));
        rhoq4b(m,n) = gb(m,n)*rhof(m,n)+gr(m,n)*real(Gamma4c(m,n)*phi(m,n));        
    end
end

%% Show what the bridge and the modes shapes look like

linespec = {'k', 'k', 'k', 'k'};
VS = zeros(NDOF,trunc1);
VS(CORRES,:) = VVst;

DVER = VS(2:3:end,:);
DHOR = VS(1:3:end,:);
ELEMNOA = Elements.ELEMNOA;
ELEMNOB = Elements.ELEMNOB;
XNOD = Nodes.XNOD;
YNOD = Nodes.YNOD;

modi = 19;
figure(6)
clf
hold on
for ielem = 1:NElem
    NOA = ELEMNOA(ielem);
    NOB = ELEMNOB(ielem);
    plot([XNOD(NOA) XNOD(NOB)], [YNOD(NOA) YNOD(NOB)], linespec{Elements.ELEMGEO(ielem)});
    plot([XNOD(NOA)-1*DHOR(NOA,modi) XNOD(NOB)-1*DHOR(NOB,modi)], [YNOD(NOA)-50*DVER(NOA,modi) YNOD(NOB)-50*DVER(NOB,modi)], 'r');
end

xlim([0 LLS+LRS+LMS])
ylim([-H 2*H])

%% Figure 1

M(1) = max(max(abs(gb.*rhof-diag(diag(gb.*rhof)))));
M(2) = max(max(abs(gr.*real(Gamma2c.*phi)-diag(diag(gr.*real(Gamma2c.*phi))))));
M(3) = max(max(abs(rhoq1-diag(diag(rhoq1)))));

top = max(M);

figure(1)
subplot(1,3,1)
pcolor(abs(gb.*rhof)+diag(NaN(trunc2,1)))
caxis([0 top])

subplot(1,3,2)
pcolor(abs(gr.*real(Gamma2c.*phi))+diag(NaN(trunc2,1)))
caxis([0 top])

subplot(1,3,3)
pcolor(abs(rhoq1)+diag(NaN(trunc2,1)))
colorbar
caxis([0 top])

%% Figure 2

figure(2)

subplot(1,3,1)
pcolor(gb)
colorbar

subplot(1,3,2)
pcolor(abs(rhof)+diag(NaN(trunc2,1)))
colorbar

subplot(1,3,3)
pcolor(abs(gb.*rhof)+diag(NaN(trunc2,1)))
colorbar

%% Figure 3

figure(3)
subplot(1,3,1)
pcolor(gr)
colorbar

subplot(1,3,2)
pcolor(abs(real(Gamma2c.*phi))+diag(NaN(trunc2,1)))
colorbar

subplot(1,3,3)
pcolor(gr.*abs(real(Gamma2c.*phi))+diag(NaN(trunc2,1)))
colorbar

