# coding=utf-8
from bisect import bisect_left

import scipy as sp
import numpy as np
import scipy.stats as st
import itertools as it
from matplotlib import pyplot as plt
from Orange.evaluation import graph_ranks
from natsort import natsorted
import os


def sort_like(ref, to_sort, *others):
    ref = np.array(ref, dtype=np.object)
    indexes = np.zeros(len(ref), dtype=np.int)
    for i, item in enumerate(to_sort):
        indexes[np.where(item == ref)[0]] = i
    return (np.array(arr, dtype=np.object)[indexes] for arr in [to_sort] + list(others))


def plot_average_rank(dest, name, data, labels, fontsize="medium", capsize=None, elinewidth=None, groups=None, dpi=600,
                      legend_kw=None, figure_kw=None, horizontal=False, ncol=1, grid=False):
    n_experiments, n_samples = data.shape
    ranks = np.empty_like(data, dtype=np.int)
    for i in range(n_samples):
        idx = np.argsort(data[:, i])
        ranks[:, i][idx] = np.arange(idx.shape[0])
    # plt.figure()
    # plt.boxplot(ranks, labels=networks)
    # plt.savefig(os.path.join(dest, "{}_boxplot.png".format(name)), dpi=600)
    # plt.tight_layout()
    # plt.close()

    avg_ranks = np.mean(ranks, axis=1)
    std_ranks = np.std(ranks, axis=1)
    fig = plt.figure(**(figure_kw if figure_kw is not None else dict()))
    if grid:
        plt.grid(linestyle='dotted', linewidth=1, color="k", alpha=0.3)
        fig.get_axes()[0].set_axisbelow(True)
    x = np.arange(n_experiments)
    bar_fn = plt.barh if horizontal else plt.bar
    if groups is None:
        bar_fn(
            x, avg_ranks, error_kw={"capsize": capsize, "elinewidth": elinewidth},
            **({"xerr": std_ranks} if horizontal else {"yerr": std_ranks})
        )
    else:
        groups = np.array(groups)
        un_groups, grp_idxs = np.unique(groups, return_index=True)
        grp_idxs = np.sort(grp_idxs)
        for grp in groups[grp_idxs]:
            idx = grp == groups
            bar_fn(
                x[idx], avg_ranks[idx], label=grp,
                error_kw={"capsize": capsize, "elinewidth": elinewidth},
                **({"xerr": std_ranks[idx]} if horizontal else {"yerr": std_ranks[idx]})
            )
        legend_kw = {
            "bbox_to_anchor": (0., 1.02, 1., .102),
            "loc": 3, "mode": "expand",
            "borderaxespad": 0.
        } if legend_kw is None else legend_kw
        legend_kw["ncol"] = ncol
        legend = plt.legend(**legend_kw)

    label = "Average rank (max: {})".format(n_experiments)
    if horizontal:
        plt.yticks(x, labels, fontsize=fontsize)
        plt.xlabel(label)
        if avg_ranks.shape[0] > 20:
            plt.ylim((-1, avg_ranks.shape[0]))
        plt.xlim(xmin=0)
    else:
        plt.xticks(x, labels, rotation=90, fontsize=fontsize)
        plt.ylabel(label)
        if avg_ranks.shape[0] > 20:
            plt.xlim((-1, avg_ranks.shape[0]))
        plt.ylim(ymin=0)
    plt.tight_layout()

    dest = os.path.join(dest, "{}_bars.png".format(name))
    if groups is None:
        plt.savefig(dest, dpi=dpi)
    else:
        plt.savefig(dest, dpi=dpi, bbox_extra_artists=(legend,), bbox_inches='tight')
    plt.close()


def friedman_test(*args):
    """
        Performs a Friedman ranking test.
        Tests the hypothesis that in a set of k dependent samples groups (where k >= 2) at least two of the groups represent populations with different median values.

        Parameters
        ----------
        sample1, sample2, ... : array_like
            The sample measurements for each group.

        Returns
        -------
        F-value : float
            The computed F-value of the test.
        p-value : float
            The associated p-value from the F-distribution.
        rankings : array_like
            The ranking for each group.
        pivots : array_like
            The pivotal quantities for each group.

        References
        ----------
        M. Friedman, The use of ranks to avoid the assumption of normality implicit in the analysis of variance, Journal of the American Statistical Association 32 (1937) 674–701.
        D.J. Sheskin, Handbook of parametric and nonparametric statistical procedures. crc Press, 2003, Test 25: The Friedman Two-Way Analysis of Variance by Ranks
    """
    k = len(args)
    if k < 2: raise ValueError('Less than 2 levels')
    n = len(args[0])
    if len(set([len(v) for v in args])) != 1: raise ValueError('Unequal number of samples')

    rankings = []
    for i in range(n):
        row = [col[i] for col in args]
        row_sort = sorted(row)
        rankings.append([row_sort.index(v) + 1 + (row_sort.count(v ) - 1) /2. for v in row])

    rankings_avg = [sp.mean([case[j] for case in rankings]) for j in range(k)]
    rankings_cmp = [r / sp.sqrt(k * (k + 1) / (6. * n)) for r in rankings_avg]

    chi2 = ((12 * n) / float((k * (k + 1)))) * ((sp.sum (r **2 for r in rankings_avg)) - ((k * (k + 1) ** 2) / float(4)))
    iman_davenport = ((n - 1) * chi2) / float((n * (k - 1) - chi2))

    p_value = 1 - st.f.cdf(iman_davenport, k- 1, (k - 1) * (n - 1))

    return iman_davenport, p_value, rankings_avg, rankings_cmp


def nemenyi_multitest(ranks):
    """
        Performs a Nemenyi post-hoc test using the pivot quantities obtained by a ranking test.
        Tests the hypothesis that the ranking of each pair of groups are different.

        Parameters
        ----------
        pivots : dictionary_like
            A dictionary with format 'groupname':'pivotal quantity'

        Returns
        ----------
        Comparions : array-like
            Strings identifier of each comparison with format 'group_i vs group_j'
        Z-values : array-like
            The computed Z-value statistic for each comparison.
        p-values : array-like
            The associated p-value from the Z-distribution wich depends on the index of the comparison
        Adjusted p-values : array-like
            The associated adjusted p-values wich can be compared with a significance level

        References
        ----------
        Bonferroni-Dunn: O.J. Dunn, Multiple comparisons among means, Journal of the American Statistical Association 56 (1961) 52–64.
    """
    k = len(ranks)
    values = list(ranks.values())
    keys = list(ranks.keys())
    versus = list(it.combinations(range(k), 2))

    comparisons = [keys[vs[0]] + " vs " + keys[vs[1]] for vs in versus]
    z_values = [abs(values[vs[0]] - values[vs[1]]) for vs in versus]
    p_values = [2 * (1 - st.norm.cdf(abs(z))) for z in z_values]
    # Sort values by p_value so that p_0 < p_1
    p_values, z_values, comparisons = map(list, zip(*sorted(zip(p_values, z_values, comparisons), key=lambda t: t[0])))
    m = int(k * (k - 1) / 2.)
    adj_p_values = [min(m * p_value, 1) for p_value in p_values]

    return comparisons, z_values, p_values, adj_p_values


def compute_CD(avranks, n, alpha="0.05", test="nemenyi"):
    """
    Returns critical difference for Nemenyi or Bonferroni-Dunn test
    according to given alpha (either alpha="0.05" or alpha="0.1") for average
    ranks and number of tested datasets N. Test can be either "nemenyi" for
    for Nemenyi two tailed test or "bonferroni-dunn" for Bonferroni-Dunn test.
    """
    k = len(avranks)
    d = {("nemenyi", "0.05"): [0, 0, 1.959964, 2.343701, 2.569032, 2.727774,
                               2.849705, 2.94832, 3.030879, 3.101730, 3.163684,
                               3.218654, 3.268004, 3.312739, 3.353618, 3.39123,
                               3.426041, 3.458425, 3.488685, 3.517073,
                               3.543799, 3.569, 3.593, 3.616, 3.637, 3.658, 3.678,
                               3.696, 3.714, 3.732, 3.749, 3.765, 3.780, 3.795, 3.810,
                               3.824, 3.837, 3.850, 3.863, 3.876, 3.888, 3.899, 3.911,
                               3.922, 3.933, 3.943, 3.954, 3.964, 3.973, 3.983, 3.992],
         ("nemenyi", "0.1"): [0, 0, 1.644854, 2.052293, 2.291341, 2.459516,
                              2.588521, 2.692732, 2.779884, 2.854606, 2.919889,
                              2.977768, 3.029694, 3.076733, 3.119693, 3.159199,
                              3.195743, 3.229723, 3.261461, 3.291224, 3.319233],
         ("bonferroni-dunn", "0.05"): [0, 0, 1.960, 2.241, 2.394, 2.498, 2.576,
                                       2.638, 2.690, 2.724, 2.773],
         ("bonferroni-dunn", "0.1"): [0, 0, 1.645, 1.960, 2.128, 2.241, 2.326,
                                      2.394, 2.450, 2.498, 2.539]}
    q = d[(test, alpha)]
    cd = q[k] * (k * (k + 1) / (6.0 * n)) ** 0.5
    return cd


def remove_common_parts(l, sep=","):
    splitted = [v.split(sep, 1) for v in l]
    prefs = [s[0] for s in splitted]
    prefs_gt0 = [p for p in prefs if len(p) != 0]
    lens = [len(s) for s in splitted]
    unique_prefs = np.unique(prefs_gt0)
    unique_parts = np.unique(lens)
    if np.all(unique_parts < 2):
        return l
    elif unique_prefs.shape[0] == 1:
        return remove_common_parts([s[1] if len(s) > 1 else "" for s in splitted], sep=sep)
    elif unique_prefs.shape[0] > 1:
        return ["{},{}".format(p, s) if len(s) > 0 else p for p, s in zip(prefs, remove_common_parts([s[1] if len(s) > 1 else "" for s in splitted], sep=sep))]


def plot(dest, scores, names, task):
    if scores.shape[0] < 2:
        print("> skip: too few experiments")
        return
    iman_davenport, p_value, rankings_avg, rankings_cmp = friedman_test(*scores)
    print("> p-value    : ", p_value)
    if p_value >= 0.05:
        print("> skip: not significant")
        return
    print(names[np.argsort(rankings_avg)[::-1]])
    cd = compute_CD(rankings_avg, n_datasets)
    print("> cd         : ", cd)
    graph_ranks(rankings_avg, names, cd, textspace=1.5, filename=os.path.join(dest, "cd/{}.png".format(task)))
    plt.close()


if __name__ == "__main__":
    n_datasets = 7
    I1 = [0.9250, 0.8268, 0.9551, 0.9805, 0.9345, 0.7568, 0.8547, 0.6960]

    a1 = [0.9749, 0.8844, 0.9935, 0.9953, 0.9427, 0.7611, 0.9043, 0.6882]
    a2 = [0.9794, 0.8852, 0.9938, 0.9864, 0.9257, 0.7010, 0.9133, 0.7820]
    a3 = [0.9795, 0.8698, 0.9928, 0.9982, 0.9485, 0.6566, 0.9077, 0.7351]
    a4 = [0.9748, 0.8893, 0.9924, 0.9882, 0.9372, 0.7633, 0.9122, 0.7791]
    a5 = [0.9722, 0.8670, 0.9910, 0.9964, 0.8951, 0.6371, 0.9088, 0.7175]
    a6 = [0.8853, 0.8654, 0.9860, 0.9905, 0.9241, 0.7237, 0.8885, 0.7302]
    a7 = [0.8824, 0.8808, 0.9859, 0.9893, 0.9413, 0.7438, 0.9020, 0.7028]

    b1 = [0.9608, 0.8848, 0.9854, 0.9840, 0.9487, 0.5872, 0.8648, 0.7126]
    b2 = [0.9726, 0.8889, 0.9891, 0.9870, 0.9556, 0.6381, 0.8874, 0.7410]
    b3 = [0.9618, 0.8699, 0.9824, 0.9953, 0.9408, 0.4789, 0.8570, 0.6676]
    b4 = [0.9634, 0.8832, 0.9758, 0.9929, 0.9507, 0.6186, 0.8851, 0.7752]
    b5 = [0.9481, 0.8795, 0.9793, 0.9929, 0.9428, 0.4583, 0.8300, 0.6305]
    b6 = [0.8551, 0.8430, 0.9795, 0.9861, 0.9231, 0.5379, 0.8536, 0.6989]
    b7 = [0.8412, 0.8791, 0.9690, 0.9888, 0.9254, 0.5791, 0.8659, 0.6833]

    ba1 = [0.9796, 0.8661, 0.9794, 0.9935, 0.9603, 0.7941, 0.8986, 0.6823]
    ba2 = [0.9822, 0.8668, 0.8316, 0.9852, 0.9482, 0.7291, 0.9054, 0.7664]
    ba3 = [0.9756, 0.8676, 0.9729, 0.9976, 0.9597, 0.6598, 0.9043, 0.7038]
    ba4 = [0.9726, 0.8670, 0.9771, 0.9899, 0.9583, 0.7996, 0.9133, 0.7674]
    ba5 = [0.9714, 0.8417, 0.9796, 0.9893, 0.9377, 0.6538, 0.8998, 0.7038]
    ba6 = [0.8447, 0.8553, 0.9661, 0.9899, 0.9237, 0.6636, 0.8863, 0.7410]
    ba7 = [0.8298, 0.8718, 0.9573, 0.9852, 0.9421, 0.6956, 0.9088, 0.7185]

    c1 = [0.9610, 0.7876, 0.9794, 0.9870, 0.9597, 0.7421, 0.8682, 0.6618]
    c2 = [0.9347, 0.8212, 0.8316, 0.9888, 0.9436, 0.6614, 0.7984, 0.6931]
    c3 = [0.9665, 0.8476, 0.9729, 0.9976, 0.9443, 0.6403, 0.8682, 0.7214]
    c4 = [0.9578, 0.8337, 0.9771, 0.9722, 0.9492, 0.7438, 0.8806, 0.7644]
    c5 = [0.9562, 0.8308, 0.9796, 0.9964, 0.9436, 0.6430, 0.8750, 0.6843]
    c6 = [0.8284, 0.8488, 0.9661, 0.9888, 0.8860, 0.6750, 0.8784, 0.7038]
    c7 = [0.8071, 0.8810, 0.9573, 0.9899, 0.9131, 0.7362, 0.8941, 0.7038]

    d1 = [0.9617, 0.8798, 0.9799, 0.9888, 0.9581, 0.6582, 0.8694, 0.7400]
    d2 = [0.9676, 0.8861, 0.9843, 0.9994, 0.9489, 0.6939, 0.8919, 0.6667]
    d3 = [0.9609, 0.8646, 0.9743, 0.9944, 0.9421, 0.5330, 0.8491, 0.6500]
    d4 = [0.9503, 0.8786, 0.9799, 0.9852, 0.9488, 0.6961, 0.8863, 0.7038]
    d5 = [0.9473, 0.8410, 0.9786, 0.9959, 0.9466, 0.5531, 0.8378, 0.7703]
    d6 = [0.8506, 0.8492, 0.9732, 0.9947, 0.9186, 0.5850, 0.8468, 0.7019]
    d7 = [0.8232, 0.8774, 0.9659, 0.9953, 0.9282, 0.6349, 0.8615, 0.6745]

    e1 = [0.9897, 0.8573, 0.9948, 0.9858, 0.8851, 0.8169, 0.9155, 0.7928]
    e2 = [0.9784, 0.8984, 0.9912, 0.9864, 0.9549, 0.6896, 0.8615, 0.6063]

    f1 = [0.9757, 0.8090, 0.9835, 0.9870, 0.9470, 0.7042, 0.8840, 0.7761]
    f2 = [0.9808, 0.8418, 0.9920, 0.9964, 0.9559, 0.7031, 0.9155, 0.7761]
    f3 = [0.9789, 0.8576, 0.9927, 0.9953, 0.9234, 0.7941, 0.9268, 0.7977]
    f4 = [0.9811, 0.8185, 0.9943, 0.9893, 0.9478, 0.7503, 0.9212, 0.7312]

    g1 = [0.9605, 0.8892, 0.9875, 0.9911, 0.9588, 0.6993, 0.8818, 0.7370]
    g2 = [0.9799, 0.8906, 0.9944, 0.9920, 0.9639, 0.6495, 0.8897, 0.7370]
    g3 = [0.9424, 0.8787, 0.9847, 0.9929, 0.9619, 0.7080, 0.8885, 0.7683]
    g4 = [0.9699, 0.8861, 0.9879, 0.9923, 0.9618, 0.6809, 0.8705, 0.6393]

    h1 = [0.9883, 0.8556, 0.9944, 0.9870, 0.9777, 0.8342, 0.9119, 0.8553]
    h2 = [0.9841, 0.8377, 0.9909, 0.9941, 0.9403, 0.6847, 0.9039, 0.7390]
    h3 = [0.9921, 0.8705, 0.9897, 0.9941, 0.9637, 0.8147, 0.9119, 0.8456]

    ha1 = [0.9828, 0.8965, 0.9950, 0.9876, 0.9827, 0.7887, 0.8982, 0.8094]
    ha2 = [0.9769, 0.8776, 0.9850, 0.9929, 0.9477, 0.5406, 0.8446, 0.7048]
    ha3 = [0.9909, 0.8806, 0.9879, 0.9870, 0.9772, 0.7763, 0.8845, 0.8289]

    i1 = [0.9892, 0.8797, 0.9977, 0.9893, 0.9835, 0.8483, 0.9405, 0.8641]
    i2 = [0.9851, 0.8795, 0.9971, 0.9929, 0.9873, 0.8727, 0.9165, 0.8182]
    i3 = [0.9926, 0.8778, 0.9953, 0.9970, 0.9827, 0.8288, 0.8971, 0.8416]

    ia1 = [0.9846, 0.8644, 0.9924, 0.9882, 0.9387, 0.7828, 0.9110, 0.8299]
    ia2 = [0.9862, 0.7944, 0.9902, 0.9959, 0.9465, 0.6831, 0.8896, 0.7331]
    ia3 = [0.9578, 0.8915, 0.9900, 0.9882, 0.9223, 0.7958, 0.9032, 0.8045]

    ib1 = [0.9875, 0.8900, 0.9905, 0.9864, 0.9679, 0.7595, 0.8829, 0.8025]
    ib2 = [0.9775, 0.8777, 0.9864, 0.9917, 0.9445, 0.5125, 0.8333, 0.6745]
    ib3 = [0.9574, 0.8989, 0.9843, 0.9864, 0.9647, 0.7990, 0.8401, 0.8123]

    ic1 = [0.9857, 0.8995, 0.9957, 0.9953, 0.9537, 0.8353, 0.9133, 0.8338]
    ic2 = [0.9725, 0.8822, 0.9943, 0.9870, 0.9664, 0.7958, 0.8930, 0.8280]
    ic3 = [0.9386, 0.9034, 0.9900, 0.9864, 0.9652, 0.8527, 0.9212, 0.8192]

    j1 = [0.7720, 0.8415, 0.9275, 0.9811, 0.9100, 0.4946, 0.8153, 0.5758]
    j2 = [0.8283, 0.8275, 0.9723, 0.9964, 0.9390, 0.7308, 0.8806, 0.6285]
    j3 = [0.8456, 0.8276, 0.9772, 0.9888, 0.9351, 0.7275, 0.8930, 0.6755]
    j4 = [0.8574, 0.8292, 0.9759, 0.9882, 0.9439, 0.6717, 0.8930, 0.6491]
    j5 = [0.8859, 0.8608, 0.9824, 0.9888, 0.9483, 0.7313, 0.9077, 0.6188]
    j6 = [0.8975, 0.8418, 0.9860, 0.9876, 0.9478, 0.7356, 0.9054, 0.6598]
    j7 = [0.8877, 0.8503, 0.9892, 0.9888, 0.9499, 0.7270, 0.9077, 0.6510]
    j8 = [0.9244, 0.8763, 0.9892, 0.9882, 0.9555, 0.7010, 0.9223, 0.6940]
    j9 = [0.9506, 0.8785, 0.9933, 0.9858, 0.9639, 0.7736, 0.9223, 0.7253]
    j10 = [0.9489, 0.8884, 0.9935, 0.9876, 0.9638, 0.8131, 0.9245, 0.7634]
    j11 = [0.9519, 0.8724, 0.9938, 0.9876, 0.9659, 0.7996, 0.9201, 0.7351]
    j12 = [0.9584, 0.8947, 0.9940, 0.9876, 0.9606, 0.7514, 0.9223, 0.7977]
    j13 = [0.9671, 0.8959, 0.9942, 0.9876, 0.9663, 0.7600, 0.9280, 0.7996]
    j14 = [0.9621, 0.8894, 0.9949, 0.9864, 0.9664, 0.7914, 0.9155, 0.8113]
    j15 = [0.9710, 0.8950, 0.9942, 0.9852, 0.9648, 0.8017, 0.9223, 0.8074]
    j16 = [0.9712, 0.8848, 0.9937, 0.9870, 0.9652, 0.7860, 0.9291, 0.8094]
    j17 = [0.9748, 0.8893, 0.9924, 0.9882, 0.9640, 0.7860, 0.9122, 0.7791]

    k1 = [0.7187, 0.8276, 0.8994, 0.9533, 0.9227, 0.4821, 0.7826, 0.4653]
    k2 = [0.7982, 0.8374, 0.9609, 0.9905, 0.9374, 0.6300, 0.8536, 0.5259]
    k3 = [0.8185, 0.8296, 0.9570, 0.9893, 0.9510, 0.6235, 0.8570, 0.5337]
    k4 = [0.9024, 0.8361, 0.9861, 0.9882, 0.9522, 0.6696, 0.9020, 0.6823]
    k5 = [0.9309, 0.8900, 0.9832, 0.9893, 0.9382, 0.6300, 0.9088, 0.6686]
    k6 = [0.9803, 0.8876, 0.9962, 0.9870, 0.9699, 0.8012, 0.9223, 0.7674]
    k7 = [0.9843, 0.8984, 0.9954, 0.9870, 0.9613, 0.7703, 0.9268, 0.7859]
    k8 = [0.9862, 0.8981, 0.9955, 0.9917, 0.9623, 0.7806, 0.9201, 0.7879]
    k9 = [0.9794, 0.8852, 0.9938, 0.9864, 0.9257, 0.7010, 0.9133, 0.7820]

    l1 = [0.8403, 0.8091, 0.9716, 0.9941, 0.9340, 0.6143, 0.8851, 0.6158]
    l2 = [0.8265, 0.8146, 0.9771, 0.9905, 0.9424, 0.6945, 0.8897, 0.6461]
    l3 = [0.8325, 0.8412, 0.9776, 0.9941, 0.9412, 0.6576, 0.8897, 0.6373]
    l4 = [0.8673, 0.8770, 0.9834, 0.9923, 0.9556, 0.6495, 0.8998, 0.6716]
    l5 = [0.8981, 0.8709, 0.9844, 0.9935, 0.9590, 0.6354, 0.9043, 0.7048]
    l6 = [0.9219, 0.8692, 0.9900, 0.9935, 0.9616, 0.6549, 0.9110, 0.7253]
    l7 = [0.9445, 0.8747, 0.9920, 0.9953, 0.9706, 0.7172, 0.9088, 0.7439]
    l8 = [0.9681, 0.8665, 0.9945, 0.9917, 0.9695, 0.8066, 0.9190, 0.7713]
    l9 = [0.9711, 0.8687, 0.9958, 0.9935, 0.9720, 0.8137, 0.9234, 0.7674]
    l10 = [0.9762, 0.8939, 0.9960, 0.9923, 0.9622, 0.7985, 0.9144, 0.7419]
    l11 = [0.9860, 0.8948, 0.9957, 0.9923, 0.9649, 0.7741, 0.9155, 0.7693]
    l12 = [0.9873, 0.8905, 0.9959, 0.9953, 0.9676, 0.7790, 0.9190, 0.7273]
    l13 = [0.9868, 0.8871, 0.9953, 0.9923, 0.9686, 0.7562, 0.9212, 0.7468]
    l14 = [0.9824, 0.8773, 0.9946, 0.9964, 0.9632, 0.7427, 0.9144, 0.7468]
    l15 = [0.9868, 0.8934, 0.9962, 0.9959, 0.9602, 0.7893, 0.9178, 0.7214]
    l16 = [0.9795, 0.8698, 0.9928, 0.9982, 0.9485, 0.6566, 0.9077, 0.7351]

    m1 = [0.8004, 0.8096, 0.9708, 0.9959, 0.9363, 0.6993, 0.8885, 0.6364]
    m2 = [0.8581, 0.8559, 0.9800, 0.9947, 0.9624, 0.7443, 0.8908, 0.6667]
    m3 = [0.8681, 0.8466, 0.9867, 0.9964, 0.9583, 0.7156, 0.8987, 0.6813]
    m4 = [0.8681, 0.8584, 0.9853, 0.9929, 0.9547, 0.7411, 0.8964, 0.6950]
    m5 = [0.9367, 0.8677, 0.9905, 0.9923, 0.9671, 0.7281, 0.9122, 0.6696]
    m6 = [0.9587, 0.8829, 0.9935, 0.9923, 0.9697, 0.7730, 0.9166, 0.7136]
    m7 = [0.9729, 0.8852, 0.9949, 0.9917, 0.9731, 0.7958, 0.9155, 0.7390]
    m8 = [0.9793, 0.8899, 0.9951, 0.9947, 0.9665, 0.8104, 0.9133, 0.7507]
    m9 = [0.9675, 0.8797, 0.9930, 0.9959, 0.9503, 0.7828, 0.9110, 0.6764]
    m10 = [0.9812, 0.8766, 0.9941, 0.9947, 0.9553, 0.7952, 0.9201, 0.7439]
    m11 = [0.9836, 0.8576, 0.9925, 0.9959, 0.9508, 0.7416, 0.9088, 0.7370]
    m12 = [0.9722, 0.8670, 0.9910, 0.9964, 0.8951, 0.6371, 0.9088, 0.7175]

    all = [
        I1, a1, a2, a3, a4, a5, a6, a7, b1, b2, b3, b4, b5, b6, b7, ba1, ba2, ba3, ba4, ba5, ba6, ba7, c1, c2, c3, c4,
        c5, c6, c7, d1, d2, d3, d4, d5, d6, d7, e1, e2, f1, f2, f3, f4, g1, g2, g3, g4, h1, h2, h3, ha1, ha2, ha3, i1,
        i2, i3, ia1, ia2, ia3, ib1, ib2, ib3, ic1, ic2, ic3, j1, j2, j3, j4, j5, j6, j7, j8, j9, j10, j11, j12, j13,
        j14, j15, j16, j17, k1, k2, k3, k4, k5, k6, k7, k8, k9, l1, l2, l3, l4, l5, l6, l7, l8, l9, l10, l11, l12, l13,
        l14, l15, l16, m1, m2, m3, m4, m5, m6, m7, m8, m9, m10, m11, m12
    ]

    exp_names = {"etfl": "ET-FL", "last": "Last layer", "mmod": "Merging networks", "mlay": "Merging layers", "ft": "Fine-tuning",
                 "ft_noda": "Fine-tuning (no aug)", "lay:resnet": "Inner: ResNet", "lay:dense": "Inner: Dense", "lay:incresv2": "Inner: IncResV2",
                 "lay:incv3": "Inner: IncV3", "rfe": "RFE"}
    exp_groups = {
        "I1": exp_names["etfl"],
        "a1": exp_names["last"],
        "a2": exp_names["last"],
        "a3": exp_names["last"],
        "a4": exp_names["last"],
        "a5": exp_names["last"],
        "a6": exp_names["last"],
        "a7": exp_names["last"],
        "b1": exp_names["last"],
        "b2": exp_names["last"],
        "b3": exp_names["last"],
        "b4": exp_names["last"],
        "b5": exp_names["last"],
        "b6": exp_names["last"],
        "b7": exp_names["last"],
        "ba1": exp_names["last"],
        "ba2": exp_names["last"],
        "ba3": exp_names["last"],
        "ba4": exp_names["last"],
        "ba5": exp_names["last"],
        "ba6": exp_names["last"],
        "ba7": exp_names["last"],
        "c1": exp_names["rfe"],
        "c2": exp_names["rfe"],
        "c3": exp_names["rfe"],
        "c4": exp_names["rfe"],
        "c5": exp_names["rfe"],
        "c6": exp_names["rfe"],
        "c7": exp_names["rfe"],
        "d1": exp_names["rfe"],
        "d2": exp_names["rfe"],
        "d3": exp_names["rfe"],
        "d4": exp_names["rfe"],
        "d5": exp_names["rfe"],
        "d6": exp_names["rfe"],
        "d7": exp_names["rfe"],
        "e1": exp_names["mmod"],
        "e2": exp_names["mmod"],
        "f1": exp_names["mlay"],
        "f2": exp_names["mlay"],
        "f3": exp_names["mlay"],
        "f4": exp_names["mlay"],
        "g1": exp_names["mlay"],
        "g2": exp_names["mlay"],
        "g3": exp_names["mlay"],
        "g4": exp_names["mlay"],
        "h1": exp_names["ft"],
        "h2": exp_names["ft"],
        "h3": exp_names["ft"],
        "ha1": exp_names["ft"],
        "ha2": exp_names["ft"],
        "ha3": exp_names["ft"],
        "i1": exp_names["ft"],
        "i2": exp_names["ft"],
        "i3": exp_names["ft"],
        "ia1": exp_names["ft_noda"],
        "ia2": exp_names["ft_noda"],
        "ia3": exp_names["ft_noda"],
        "ib1": exp_names["ft_noda"],
        "ib2": exp_names["ft_noda"],
        "ib3": exp_names["ft_noda"],
        "ic1": exp_names["ft_noda"],
        "ic2": exp_names["ft_noda"],
        "ic3": exp_names["ft_noda"],
        "j1": exp_names["lay:resnet"],
        "j2": exp_names["lay:resnet"],
        "j3": exp_names["lay:resnet"],
        "j4": exp_names["lay:resnet"],
        "j5": exp_names["lay:resnet"],
        "j6": exp_names["lay:resnet"],
        "j7": exp_names["lay:resnet"],
        "j8": exp_names["lay:resnet"],
        "j9": exp_names["lay:resnet"],
        "j10": exp_names["lay:resnet"],
        "j11": exp_names["lay:resnet"],
        "j12": exp_names["lay:resnet"],
        "j13": exp_names["lay:resnet"],
        "j14": exp_names["lay:resnet"],
        "j15": exp_names["lay:resnet"],
        "j16": exp_names["lay:resnet"],
        "j17": exp_names["lay:resnet"],
        "k1": exp_names["lay:dense"],
        "k2": exp_names["lay:dense"],
        "k3": exp_names["lay:dense"],
        "k4": exp_names["lay:dense"],
        "k5": exp_names["lay:dense"],
        "k6": exp_names["lay:dense"],
        "k7": exp_names["lay:dense"],
        "k8": exp_names["lay:dense"],
        "k9": exp_names["lay:dense"],
        "l1": exp_names["lay:incresv2"],
        "l2": exp_names["lay:incresv2"],
        "l3": exp_names["lay:incresv2"],
        "l4": exp_names["lay:incresv2"],
        "l5": exp_names["lay:incresv2"],
        "l6": exp_names["lay:incresv2"],
        "l7": exp_names["lay:incresv2"],
        "l8": exp_names["lay:incresv2"],
        "l9": exp_names["lay:incresv2"],
        "l10": exp_names["lay:incresv2"],
        "l11": exp_names["lay:incresv2"],
        "l12": exp_names["lay:incresv2"],
        "l13": exp_names["lay:incresv2"],
        "l14": exp_names["lay:incresv2"],
        "l15": exp_names["lay:incresv2"],
        "l16": exp_names["lay:incresv2"],
        "m1": exp_names["lay:incv3"],
        "m2": exp_names["lay:incv3"],
        "m3": exp_names["lay:incv3"],
        "m4": exp_names["lay:incv3"],
        "m5": exp_names["lay:incv3"],
        "m6": exp_names["lay:incv3"],
        "m7": exp_names["lay:incv3"],
        "m8": exp_names["lay:incv3"],
        "m9": exp_names["lay:incv3"],
        "m10": exp_names["lay:incv3"],
        "m11": exp_names["lay:incv3"],
        "m12": exp_names["lay:incv3"]
    }
    results = {
        "I1": I1,
        "a1": a1,
        "a2": a2,
        "a3": a3,
        "a4": a4,
        "a5": a5,
        "a6": a6,
        "a7": a7,
        "b1": b1,
        "b2": b2,
        "b3": b3,
        "b4": b4,
        "b5": b5,
        "b6": b6,
        "b7": b7,
        "ba1": ba1,
        "ba2": ba2,
        "ba3": ba3,
        "ba4": ba4,
        "ba5": ba5,
        "ba6": ba6,
        "ba7": ba7,
        "c1": c1,
        "c2": c2,
        "c3": c3,
        "c4": c4,
        "c5": c5,
        "c6": c6,
        "c7": c7,
        "d1": d1,
        "d2": d2,
        "d3": d3,
        "d4": d4,
        "d5": d5,
        "d6": d6,
        "d7": d7,
        "e1": e1,
        "e2": e2,
        "f1": f1,
        "f2": f2,
        "f3": f3,
        # "f4": f4,
        "g1": g1,
        "g2": g2,
        "g3": g3,
        # "g4": g4,
        "h1": h1,
        "h2": h2,
        "h3": h3,
        "ha1": ha1,
        "ha2": ha2,
        "ha3": ha3,
        "i1": i1,
        "i2": i2,
        "i3": i3,
        # "ia1": ia1,
        # "ia2": ia2,
        # "ia3": ia3,
        # "ib1": ib1,
        # "ib2": ib2,
        # "ib3": ib3,
        # "ic1": ic1,
        # "ic2": ic2,
        # "ic3": ic3,
        "j1": j1,
        "j2": j2,
        "j3": j3,
        "j4": j4,
        "j5": j5,
        "j6": j6,
        "j7": j7,
        "j8": j8,
        "j9": j9,
        "j10": j10,
        "j11": j11,
        "j12": j12,
        "j13": j13,
        "j14": j14,
        "j15": j15,
        "j16": j16,
        "j17": j17,
        "k1": k1,
        "k2": k2,
        "k3": k3,
        "k4": k4,
        "k5": k5,
        "k6": k6,
        "k7": k7,
        "k8": k8,
        "k9": k9,
        "l1": l1,
        "l2": l2,
        "l3": l3,
        "l4": l4,
        "l5": l5,
        "l6": l6,
        "l7": l7,
        "l8": l8,
        "l9": l9,
        "l10": l10,
        "l11": l11,
        "l12": l12,
        "l13": l13,
        "l14": l14,
        "l15": l15,
        "l16": l16,
        # "m1": m1,
        # "m2": m2,
        # "m3": m3,
        # "m4": m4,
        # "m5": m5,
        # "m6": m6,
        # "m7": m7,
        # "m8": m8,
        # "m9": m9,
        # "m10": m10,
        # "m11": m11,
        # "m12": m12
    }
    """ (exp, clf, net, layer)"""
    exp_details = {
        "I1": ("etfl", "svm", "ET-FL", ""),
        "a1": ("last", "svm", "Mobile", "last"),
        "a2": ("last", "svm", "Dense", "last"),
        "a3": ("last", "svm", "IncResV2", "last"),
        "a4": ("last", "svm", "ResNet", "last"),
        "a5": ("last", "svm", "IncV3", "last"),
        "a6": ("last", "svm", "VGG19", "last"),
        "a7": ("last", "svm", "VGG16", "last"),
        "b1": ("last", "et", "Mobile", "last"),
        "b2": ("last", "et", "Dense", "last"),
        "b3": ("last", "et", "IncResV2", "last"),
        "b4": ("last", "et", "ResNet", "last"),
        "b5": ("last", "et", "IncV3", "last"),
        "b6": ("last", "et", "VGG19", "last"),
        "b7": ("last", "et", "VGG16", "last"),
        "ba1": ("last", "fc", "Mobile", "last"),
        "ba2": ("last", "fc", "Dense", "last"),
        "ba3": ("last", "fc", "IncResV2", "last"),
        "ba4": ("last", "fc", "ResNet", "last"),
        "ba5": ("last", "fc", "IncV3", "last"),
        "ba6": ("last", "fc", "VGG19", "last"),
        "ba7": ("last", "fc", "VGG16", "last"),
        "c1": ("rfe", "svm", "Mobile", "last"),
        "c2": ("rfe", "svm", "Dense", "last"),
        "c3": ("rfe", "svm", "IncResV2", "last"),
        "c4": ("rfe", "svm", "ResNet", "last"),
        "c5": ("rfe", "svm", "IncV3", "last"),
        "c6": ("rfe", "svm", "VGG19", "last"),
        "c7": ("rfe", "svm", "VGG16", "last"),
        "d1": ("rfe", "et", "Mobile", "last"),
        "d2": ("rfe", "et", "Dense", "last"),
        "d3": ("rfe", "et", "IncResV2", "last"),
        "d4": ("rfe", "et", "ResNet", "last"),
        "d5": ("rfe", "et", "IncV3", "last"),
        "d6": ("rfe", "et", "VGG19", "last"),
        "d7": ("rfe", "et", "VGG16", "last"),
        "e1": ("mmod", "svm", "merged", "last"),
        "e2": ("mmod", "et", "merged", "last"),
        "f1": ("mlay", "svm", "Dense", "merged"),
        "f2": ("mlay", "svm", "IncResV2", "merged"),
        "f3": ("mlay", "svm", "ResNet", "merged"),
        "f4": ("mlay", "svm", "IncV3", "merged"),
        "g1": ("mlay", "et", "Dense", "merged"),
        "g2": ("mlay", "et", "IncResV2", "merged"),
        "g3": ("mlay", "et", "ResNet", "merged"),
        "g4": ("mlay", "et", "IncV3", "merged"),
        "h1": ("ft", "svm", "Dense", "last"),
        "h2": ("ft", "svm", "IncResV2", "last"),
        "h3": ("ft", "svm", "ResNet", "last"),
        "ha1": ("ft", "et", "Dense", "last"),
        "ha2": ("ft", "et", "IncResV2", "last"),
        "ha3": ("ft", "et", "ResNet", "last"),
        "i1": ("ft", "net", "Dense", "last"),
        "i2": ("ft", "net", "IncResV2", "last"),
        "i3": ("ft", "net", "ResNet", "last"),
        "ia1": ("ft_noda", "svm", "Dense", "last"),
        "ia2": ("ft_noda", "svm", "IncResV2", "last"),
        "ia3": ("ft_noda", "svm", "ResNet", "last"),
        "ib1": ("ft_noda", "et", "Dense", "last"),
        "ib2": ("ft_noda", "et", "IncResV2", "last"),
        "ib3": ("ft_noda", "et", "ResNet", "last"),
        "ic1": ("ft_noda", "net", "Dense", "last"),
        "ic2": ("ft_noda", "net", "IncResV2", "last"),
        "ic3": ("ft_noda", "net", "ResNet", "last"),
        "j1": ("lay:resnet", "svm", "ResNet", "activation_1"),
        "j2": ("lay:resnet", "svm", "ResNet", "activation_4"),
        "j3": ("lay:resnet", "svm", "ResNet", "activation_7"),
        "j4": ("lay:resnet", "svm", "ResNet", "activation_10"),
        "j5": ("lay:resnet", "svm", "ResNet", "activation_13"),
        "j6": ("lay:resnet", "svm", "ResNet", "activation_16"),
        "j7": ("lay:resnet", "svm", "ResNet", "activation_19"),
        "j8": ("lay:resnet", "svm", "ResNet", "activation_22"),
        "j9": ("lay:resnet", "svm", "ResNet", "activation_25"),
        "j10": ("lay:resnet", "svm", "ResNet", "activation_28"),
        "j11": ("lay:resnet", "svm", "ResNet", "activation_31"),
        "j12": ("lay:resnet", "svm", "ResNet", "activation_34"),
        "j13": ("lay:resnet", "svm", "ResNet", "activation_37"),
        "j14": ("lay:resnet", "svm", "ResNet", "activation_40"),
        "j15": ("lay:resnet", "svm", "ResNet", "activation_43"),
        "j16": ("lay:resnet", "svm", "ResNet", "activation_46"),
        "j17": ("lay:resnet", "svm", "ResNet", "last"),
        "k1": ("lay:dense", "svm", "Dense", "pool1"),
        "k2": ("lay:dense", "svm", "Dense", "conv2_block6"),
        "k3": ("lay:dense", "svm", "Dense", "pool2_pool"),
        "k4": ("lay:dense", "svm", "Dense", "conv3_block12"),
        "k5": ("lay:dense", "svm", "Dense", "pool3_pool"),
        "k6": ("lay:dense", "svm", "Dense", "conv4_block48"),
        "k7": ("lay:dense", "svm", "Dense", "pool4_pool"),
        "k8": ("lay:dense", "svm", "Dense", "conv5_block32"),
        "k9": ("lay:dense", "svm", "Dense", "last"),
        "l1": ("lay:incresv2", "svm", "IncResV2", "max_pool2d_2"),
        "l2": ("lay:incresv2", "svm", "IncResV2", "mixed_5b"),
        "l3": ("lay:incresv2", "svm", "IncResV2", "block35_1_ac"),
        "l4": ("lay:incresv2", "svm", "IncResV2", "block35_4_ac"),
        "l5": ("lay:incresv2", "svm", "IncResV2", "block35_7_ac"),
        "l6": ("lay:incresv2", "svm", "IncResV2", "block35_10_ac"),
        "l7": ("lay:incresv2", "svm", "IncResV2", "mixed_6a"),
        "l8": ("lay:incresv2", "svm", "IncResV2", "block17_5_ac"),
        "l9": ("lay:incresv2", "svm", "IncResV2", "block17_10_ac"),
        "l10": ("lay:incresv2", "svm", "IncResV2", "block17_15_ac"),
        "l11": ("lay:incresv2", "svm", "IncResV2", "block17_20_ac"),
        "l12": ("lay:incresv2", "svm", "IncResV2", "block8_3_ac"),
        "l13": ("lay:incresv2", "svm", "IncResV2", "block8_6_ac"),
        "l14": ("lay:incresv2", "svm", "IncResV2", "block8_9_ac"),
        "l15": ("lay:incresv2", "svm", "IncResV2", "mixed_7a"),
        "l16": ("lay:incresv2", "svm", "IncResV2", "last"),
        "m1": ("lay:incv3", "svm", "IncV3", "max_pool2d_2"),
        "m2": ("lay:incv3", "svm", "IncV3", "mixed0"),
        "m3": ("lay:incv3", "svm", "IncV3", "mixed1"),
        "m4": ("lay:incv3", "svm", "IncV3", "mixed2"),
        "m5": ("lay:incv3", "svm", "IncV3", "mixed3"),
        "m6": ("lay:incv3", "svm", "IncV3", "mixed4"),
        "m7": ("lay:incv3", "svm", "IncV3", "mixed5"),
        "m8": ("lay:incv3", "svm", "IncV3", "mixed6"),
        "m9": ("lay:incv3", "svm", "IncV3", "mixed7"),
        "m10": ("lay:incv3", "svm", "IncV3", "mixed8"),
        "m11": ("lay:incv3", "svm", "IncV3", "mixed9"),
        "m12": ("lay:incv3", "svm", "IncV3", "last")
    }

    #dest = "C:/data/cytomine/features"
    dest = "D:/data/features"
    plt.style.use('seaborn-paper')

    rfe_last_svm = ["I1", "a1", "a2", "a3", "a4", "a5", "a6", "a7", "c1", "c2", "c3", "c4", "c5", "c6", "c7"]
    rfe_last_et = ["I1", "b1", "b2", "b3", "b4", "b5", "b6", "b7", "d1", "d2", "d3", "d4", "d5", "d6", "d7"]
    plot_average_rank(
        os.path.join(dest, "plots"), "rfe_last_svm",
        np.array([results[k] for k in rfe_last_svm]),
        ["{}".format(exp_details[k][2]) for k in rfe_last_svm],
        groups=[exp_groups[k] + (" (svm)" if k.startswith("a") or k.startswith("c") else "") for i, k in enumerate(rfe_last_svm)],
        capsize=3, dpi=300,
        figure_kw={"figsize": (6.4, 2.4)},
        grid=True, ncol=4
    )

    plot_average_rank(
        os.path.join(dest, "plots"), "rfe_last_et",
        np.array([results[k] for k in rfe_last_et]),
        ["{}".format(exp_details[k][2]) for k in rfe_last_et],
        groups=[exp_groups[k] + (" (et)" if k.startswith("b") or k.startswith("d") else "") for i, k in enumerate(rfe_last_et)],
        capsize=3, dpi=300,
        figure_kw={"figsize": (6.4, 2.4)},
        grid=True, ncol=4
    )

    merged_layers_all = ["I1", "f1", "f2", "f3",
                         #"f4",
                         "g1", "g2", "g3",
                         #"g4"
                         ]
    plot_average_rank(
        os.path.join(dest, "plots"), "merged_layers_all",
        np.array([results[k] for k in merged_layers_all]),
        ["{}".format(exp_details[k][2]) for k in merged_layers_all],
        groups=[exp_groups[k] + (" (svm)" if k.startswith("f") else (" (et)" if k.startswith("g") else "")) for i, k in enumerate(merged_layers_all)],
        capsize=3, dpi=300,
        figure_kw={"figsize": (6.4, 2.4)},
        grid=True, ncol=4
    )

    last_rfe = ["I1", "c1", "c2", "c3", "c4", "c5", "c6", "c7", "d1", "d2", "d3", "d4", "d5", "d6", "d7"]
    plot_average_rank(
        os.path.join(dest, "plots"), "last_rfe",
        np.array([results[k] for k in last_rfe]),
        ["{}".format(exp_details[k][2]) for k in last_rfe],
        groups=[exp_groups[k] + (" (svm)" if k.startswith("c") else (" (et)" if k.startswith("d") else "")) for i, k in
                enumerate(last_rfe)],
        capsize=3, dpi=300
    )

    last_baseline = ["I1", "a1", "a2", "a3", "a4", "a5", "a6", "a7", "b1", "b2", "b3", "b4", "b5", "b6", "b7", "ba1",
                     "ba2", "ba3", "ba4", "ba5", "ba6", "ba7"]
    plot_average_rank(
        os.path.join(dest, "plots"), "last_baseline",
        np.array([results[k] for k in last_baseline]),
        ["{}".format(exp_details[k][2]) for k in last_baseline],
        groups=[exp_groups[k] + " ({})".format(exp_details[k][1]) for k in last_baseline],
        capsize=3, dpi=300,
        figure_kw={"figsize": (6.4, 2.4)},
        grid=True, ncol=4
    )

    ft_baseline = ["I1", "h1", "h2", "h3", "ha1", "ha2", "ha3", "i1", "i2", "i3",
                   #"ia1", "ia2", "ia3", "ib1", "ib2","ib3", "ic1", "ic2", "ic3"
                   ]
    plot_average_rank(
        os.path.join(dest, "plots"), "ft_baseline",
        np.array([results[k] for k in ft_baseline]),
        ["{}".format(exp_details[k][2]) for k in ft_baseline],
        groups=[exp_groups[k] + " ({})".format(exp_details[k][1]) for k in ft_baseline],
        capsize=3, dpi=300,
        figure_kw={"figsize": (6.4, 2.4)},
        grid=True, ncol=3
    )

    all_inner_merge_layers_keys = [
        "I1",
        "f1", "f2", "f3",
        # "f4",
        "g1", "g2", "g3",
        # "g4"
        "j1", "j2", "j3", "j4", "j5", "j6", "j7", "j8", "j9", "j10", "j11", "j12", "j13", "j14", "j15", "j16",
        "j17", "k1", "k2", "k3", "k4", "k5", "k6", "k7", "k8", "k9", "l1", "l2", "l3", "l4", "l5", "l6", "l7", "l8",
        "l9", "l10", "l11", "l12", "l13", "l14", "l15", "l16",
        # "m1", "m2", "m3", "m4", "m5", "m6", "m7", "m8", "m9",
        # "m10", "m11", "m12"
    ]
    plot_average_rank(
        os.path.join(dest, "plots"), "all_inner_merged_layer",
        np.array([results[k] for k in all_inner_merge_layers_keys]),
        [exp_details[all_inner_merge_layers_keys[0]][2]] +
            ["{} ({})".format(exp_details[k][2], exp_details[k][1]) for k in all_inner_merge_layers_keys[1:7]] +
            ["{}".format(exp_details[k][3]) for k in all_inner_merge_layers_keys[7:]],
        fontsize=7,
        capsize=1.25,
        elinewidth=1.0,
        groups=[exp_groups[k] for k in all_inner_merge_layers_keys], dpi=300,
        ncol=2,
        legend_kw={
            "bbox_to_anchor": (0., 1.005, 1., .102),
            "loc": 3, "mode": "expand",
            "borderaxespad": 0., "fontsize": 7
        },
        figure_kw={"figsize": (4, 6)},
        horizontal=True,
        grid=True
    )

    all_per_layers_keys = [
        "I1", "j1", "j2", "j3", "j4", "j5", "j6", "j7", "j8", "j9", "j10", "j11", "j12", "j13", "j14", "j15", "j16",
        "j17", "k1", "k2", "k3", "k4", "k5", "k6", "k7", "k8", "k9", "l1", "l2", "l3", "l4", "l5", "l6", "l7", "l8",
        "l9", "l10", "l11", "l12", "l13", "l14", "l15", "l16",
        # "m1", "m2", "m3", "m4", "m5", "m6", "m7", "m8", "m9",
        # "m10", "m11", "m12"
    ]
    plot_average_rank(
        os.path.join(dest, "plots"), "all_per_layer",
        np.array([results[k] for k in all_per_layers_keys]),
        [exp_details[all_per_layers_keys[0]][2]] + ["{}".format(exp_details[k][3]) for k in all_per_layers_keys[1:]],
        fontsize=7,
        capsize=1.25,
        elinewidth=1.0,
        groups=[exp_groups[k] for k in all_per_layers_keys], dpi=300,
        ncol=2,
        legend_kw={
            "bbox_to_anchor": (0., 1.005, 1., .102),
            "loc": 3, "mode": "expand",
            "borderaxespad": 0., "fontsize": 7
        },
        figure_kw={"figsize": (4, 6)},
        horizontal=True,
        grid=True
    )


    plot_average_rank(os.path.join(dest, "plots"), "last_net", np.array([
        a1 + b1,
        a2 + b2,
        a3 + b3,
        a4 + b4,
        a5 + b5,
        a6 + b6,
        a7 + b7
    ]), ["Mobile", "Dense", "IncResV2", "ResNet", "IncV3", "VGG19", "VGG16"], dpi=300)

    plot_average_rank(os.path.join(dest, "plots"), "last_mlay_net", np.array([
        a2 + b2 + c2 + d2 + f1 + g1,
        a3 + b3 + c3 + d3 + f2 + g2,
        a4 + b4 + c4 + d4 + f3 + g3,
        a5 + b5 + c5 + d5 + f4 + g4
    ]), ["Dense", "IncResV2", "ResNet", "IncV3"], dpi=300)

    all_exp_keys = natsorted(results.keys())
    plot_average_rank(
        os.path.join(dest, "plots"), "all_exp",
        np.array([results[k] for k in all_exp_keys]),[
            {
                "etfl": lambda t: t[2],
                "last": lambda t: "{} ({})".format(t[2], t[1]),
                "rfe": lambda t: "{} ({})".format(t[2], t[1]),
                "mmod": lambda t: "Models ({})".format(t[1]),
                "mlay": lambda t: "{} ({})".format(t[2], t[1]),
                "ft": lambda t: "{} ({})".format(t[2], t[1]),
                "ft_noda": lambda t: "{} ({})".format(t[2], t[1]),
                "lay:resnet": lambda t: "{}".format(t[3]),
                "lay:dense": lambda t: "{}".format(t[3]),
                "lay:incresv2": lambda t: "{}".format(t[3]),
                "lay:incv3": lambda t: "{}".format(t[3]),
            }[exp_details[k][0]](exp_details[k]) for k in all_exp_keys
        ],
        fontsize=7,
        capsize=0.5,
        elinewidth=0.5,
        groups=[exp_groups[k] for k in all_exp_keys], dpi=300,
        legend_kw={
            "bbox_to_anchor": (0., 1.005, 1., .102),
            "loc": 3, "mode": "expand",
            "borderaxespad": 0., "fontsize": 7
        },
        figure_kw={"figsize": (4, 10)},
        horizontal=True,
        ncol=2,
        grid=True
    )

    # for subgroup_name, subgroup_exps in cross_experiments:
    #     print(subgroup_name)
    #     scores = np.array([results[exp] for exps in subgroup_exps for exp in subgroups_idx[exps] if exp in results])
    #     names = np.array(remove_common_parts([exp_names[exp] for exps in subgroup_exps for exp in subgroups_idx[exps] if exp in results]))
    #     plot(scores, names, subgroup_name)
    #
    # # per network (all)
    # scores_per_net = np.vstack([np.hstack([results[exp] for exp in exps]) for _, exps in per_network_all.items()])
    # names_per_net = np.array([k for k, _ in per_network_all.items()])
    # plot(scores_per_net, names_per_net, "per_network_all")
    #
    # # pre networks recent
    # scores_per_net = np.vstack([np.hstack([results[exp] for exp in exps]) for _, exps in per_network_recent.items()])
    # names_per_net = np.array([k for k, _ in per_network_recent.items()])
    # plot(scores_per_net, names_per_net, "per_network")
    #
    # computed = set()
    # all_exps = np.array(list(results.keys()))
    # sorted_idx = np.array(natsorted(range(len(all_exps)), key=lambda i: all_exps[i]))
    # for exp in all_exps[sorted_idx]:
    #     print(exp_names[exp], end="\t")
    # print()
    # for exp1 in all_exps[sorted_idx]:
    #     for exp2 in all_exps[sorted_idx]:
    #         if exp1 == exp2 or (exp1, exp2) in computed or (exp2, exp1) in computed:
    #             print("-1\t", end="")
    #             continue
    #         computed.add((exp1, exp2))
    #         ranks, wilcox_pvalue = st.wilcoxon(results[exp1], results[exp2])
    #         print("{}".format(ranks), end="\t")
    #     print()