import os

import numpy as np
from argparse import ArgumentParser
import matplotlib
matplotlib.use('Agg')


def print_table(array, row_headers, col_headers):
    row_headers_width = np.max([len(str(h)) for h in row_headers]) + 1
    row_h_fmt = "{{0: <{}}}".format(row_headers_width)
    n_rows, n_cols = array.shape
    col_fmt = np.zeros(n_cols, dtype=np.object)
    for j in range(n_cols):
        max_width = max(
            len(str(col_headers[j])),
            np.max([len(str(v)) for v in array[:, j]])
        )
        col_fmt[j] = "{{0: <{}}}".format(int(max_width))

    # print
    print("\t".join([" " * row_headers_width] + [col_fmt[j].format(col_headers[j]) for j in range(n_cols)]))
    for i in range(n_rows):
        print("\t".join([row_h_fmt.format(row_headers[i])] + [col_fmt[j].format(array[i, j]) for j in range(n_cols)]))


def main(argv):
    parser = ArgumentParser()

    # Model loading and saving
    parser.add_argument("--path", dest="path")
    parser.add_argument("--model", dest="model")
    parser.add_argument("--dataset", dest="dataset")
    parser.add_argument("--n_features", dest="n_features", default="1,0.5", help="in % if is_count is false, number of features otherwise")
    params, unknown = parser.parse_known_args(argv)
    params.n_features = [int(n_feat) if float(n_feat) >= 1 else float(n_feat) for n_feat in params.n_features.split(",")]
    print("Parameters: {}".format(params))

    print("> load data")
    data = np.load(os.path.join(params.path, "{}_{}_importances.npz".format(params.model, params.dataset)))
    importances = np.sort(data["importances"])
    inv_cumsum = np.cumsum(importances[::-1])
    actual_features = importances.shape[0]

    computed = list()
    for v in params.n_features:
        to_take = int(v * actual_features) if isinstance(v, float) else v
        perc = inv_cumsum[to_take - 1]
        computed.append((to_take, perc))

    for t, _ in computed:
        print "{}\t".format(t),
    for _, p in computed:
        print "{}\t".format(p),


if __name__ == "__main__":
    import sys
    main(sys.argv[1:])
