import cv2
import random
import numpy as np

class Ellipse:

	def __init__(self):
		self.center_x = 0
		self.center_y = 0
		self.major_axis = 0
		self.minor_axis = 0
		self.angle = 0

		self.focal_1_x = 0
		self.focal_1_y = 0
		self.focal_2_x = 0
		self.focal_2_y = 0

		self.exists = False


		self.criteria ={
				"min_angle":176,
				"max_angle":184,
				"min_axis_ratio":3.5,
				"max_axis_ratio":5.5,
		}

	def test_ellipse(self):
		"""
		Tests if the found ellipse can be associated to a real main circle ellipse
		in the soccer field. The parameters are tested through the defined criteria
		"""

		if self.angle > self.criteria["max_angle"] or self.angle < self.criteria["min_angle"]:
			self.exists = False
			return False

		if self.major_axis > 0 and self.minor_axis > 0:
			axis_ratio = self.major_axis/self.minor_axis

			if axis_ratio > self.criteria["max_axis_ratio"] or axis_ratio < self.criteria["min_axis_ratio"]:
				self.exists = False
				return False
		else:
			self.exists = False
			return False	

		self.exists = True
		return True

	def are_points_inside_ellipse(self, points):


		distance_1 = np.sqrt( (points[:,0] - self.focal_1_x)**2 + (points[:,1] - self.focal_1_y)**2 )
		distance_2 = np.sqrt( (points[:,0] - self.focal_2_x)**2 + (points[:,1] - self.focal_2_y)**2 )

		return distance_1 + distance_2 <= 2*self.major_axis


	def draw_ellipse(self, frame, thickness = 3, color = (255,0,0)):

		ellipse_frame = np.copy(frame)

		cv2.ellipse(ellipse_frame,(int(round(self.center_x)),int(round(self.center_y))),(int(round(self.major_axis)),int(round(self.minor_axis))),int(round(self.angle)),0,360,color,thickness)
		return ellipse_frame

	def ellipse_from_ellipse(self, factor):

		new_ellipse = Ellipse()

		new_ellipse.center_x = self.center_x
		new_ellipse.center_y = self.center_y

		new_ellipse.major_axis = self.major_axis + factor
		new_ellipse.minor_axis = self.minor_axis + factor

		new_ellipse.angle = self.angle

		axis_factor = np.sqrt(abs(new_ellipse.major_axis**2 - new_ellipse.minor_axis**2))

		new_ellipse.focal_1_x = new_ellipse.center_x - axis_factor * np.cos(new_ellipse.angle*np.pi/180)
		new_ellipse.focal_1_y = new_ellipse.center_y - axis_factor * np.sin(new_ellipse.angle*np.pi/180)
		new_ellipse.focal_2_x = new_ellipse.center_x + axis_factor * np.cos(new_ellipse.angle*np.pi/180)
		new_ellipse.focal_2_y = new_ellipse.center_y + axis_factor * np.sin(new_ellipse.angle*np.pi/180)

		return new_ellipse


	def evaluate_model(self, points, threshold):

		self.list_of_inliers = []
		score = 0

		if not self.major_axis >0 or not self.minor_axis > 0:
			return score, []

		smaller_ellipse = self.ellipse_from_ellipse(-threshold)
		larger_ellipse = self.ellipse_from_ellipse(threshold)

		

		in_larger_points = larger_ellipse.are_points_inside_ellipse(points)
		in_smaller_points = smaller_ellipse.are_points_inside_ellipse(points)

		near_ellipse_points = np.logical_and(in_larger_points, np.logical_not(in_smaller_points))

		inliers = points[near_ellipse_points]

		return np.sum(near_ellipse_points), inliers


	def build_ellipse(self, contour):
		ellipse_box = cv2.fitEllipse(contour)

		self.center_x = ellipse_box[0][0]
		self.center_y = ellipse_box[0][1]

		self.major_axis = ellipse_box[1][0]/2
		self.minor_axis = ellipse_box[1][1]/2

		self.angle = ellipse_box[2]


		if self.major_axis < self.minor_axis:
			inverse_axis = self.major_axis
			self.major_axis = self.minor_axis
			self.minor_axis = inverse_axis
			self.angle = self.angle +90

		self.angle = self.angle


		axis_factor = np.sqrt(abs(self.major_axis**2 - self.minor_axis**2))

		self.focal_1_x = self.center_x - axis_factor * np.cos(self.angle*np.pi/180)
		self.focal_1_y = self.center_y - axis_factor * np.sin(self.angle*np.pi/180)
		self.focal_2_x = self.center_x + axis_factor * np.cos(self.angle*np.pi/180)
		self.focal_2_y = self.center_y + axis_factor * np.sin(self.angle*np.pi/180)

	def __str__(self):
		str1 = "Center : (" + str(self.center_x) + "," + str(self.center_y) + ")" + " - "
		str2 = "Axis : (" + str(self.major_axis) + "," + str(self.minor_axis) + ")" +" - "
		str3 = "Angle : " + str(self.angle)
		return str1+str2+str3


class Point:

	def __init__(self, x, y):
		self.x = x
		self.y = y

class EllipseRANSAC:

	def __init__(self):

		self.number_of_tests = 10000
		self.number_of_points = 10

		self.list_of_points = []
		self.list_of_inliers = []
		self.list_of_outliers = []

	def compute_all_points(self, binary_map):

		self.reset()
		self.list_of_points = cv2.findNonZero(binary_map)


	def select_random_points(self):

		indexes = random.sample(range(1, self.list_of_points.shape[0]), self.number_of_points)
		return self.list_of_points[indexes,0]

	def reset(self):
		self.list_of_points = []
		self.list_of_inliers = []

	def find_ellipse(self, binary_map):

		binary_map = self.clean_lines(binary_map)

		self.compute_all_points(binary_map)
		max_score = 0
		max_ellipse = Ellipse()

		if self.list_of_points is None:
			return max_ellipse

		img = binary_map
		for number in np.arange(self.number_of_tests):

			points = self.select_random_points()
			ellipse = Ellipse()
			ellipse.build_ellipse(points)
			score, inliers = ellipse.evaluate_model(self.list_of_points[:,0,:],3)
			

			

			if score > max_score:
				max_score = score
				max_ellipse = ellipse.ellipse_from_ellipse(0)
				self.list_of_inliers = np.copy(inliers)


		max_ellipse.build_ellipse(self.list_of_inliers)

		max_ellipse.test_ellipse()

		return max_ellipse

	def clean_lines(self, binary_map):

		 lines_to_remove = cv2.HoughLinesP(binary_map, rho=1, theta = np.pi/180, threshold = 50, minLineLength =300, maxLineGap = 100)
		 binary_map_cleaned = np.copy(binary_map)
		 if not lines_to_remove is None:
		 	for line in lines_to_remove[:,0,:]:
		 		cv2.line(binary_map_cleaned, (line[0],line[1]), (line[2],line[3]), 0, 8)

		 return binary_map_cleaned