function results = quakeSimulation
%function results = quakeSimulation
%   This function computes the time response of an elastic bouncing bar, as presented in Section 4.3 of
%      Depouhon A., Detournay E. and Denoël V., "Event-driven integration of linear structural 
%      dynamics models under unilateral elastic constraints", CMAME, 2014
%
%   The function calls timeIntegration.m
%
%   Problem parameters include:
%     - Building definition stored in _buildingModel.mat_ (K,M,nodes coordinates, etc.)
%     - Initial gap opening between building substructures (g0)
%     - Accelerograms representing ground horizontal acceleration stored in _ACCEL.ACG_
%     - Contact stiffness (k_con)
%     - Minimum number of computed points per lowest eigenperiod of the system (nPtsPer)
%     - Time integration parameters (schemeID,rho,hOut,nRedH)
%     - Root-solving parameters (gTol,maxIter,tTol)
%
%   Results are stored in binary files by timeIntegration.m. These are loaded and returned by the current function
%   under the form of data structure _results_.
%
%   Written by A. Depouhon - 12/4/2013
%   Contact: alexandre.depouhon@ulg.ac.be

    % Clean environment
    close all
    clc
    clear functions     % Required to clear all persistent variables !
          
    % Load building data - NB: BC's are already enforced in governing matrices
    load buildingModel
    
    % Define contact constraints from connected_dofs array
    n       = size(connected_dofs,1);       %#ok<NODEF>
    m       = size(K,1);                    %#ok<NODEF>
    W       = zeros(m,n);
    for ii = 1:n
        W(connected_dofs(ii,:),ii) ...
                = [-1 1];
    end
    W       = sparse(W);
    
    % Define initial gap between buildings
    g0      = 0.04; % [m]
    
    % Add structural damping
    w       = sqrt(eigs(K,M,2,'sm'));
    xi      = 0.05;
    [alpha,beta] ...
            = RayleighDamping(w(1),w(2),xi,xi);
    C       = alpha*M+beta*K;
    
    % Define initial conditions (rest)
    u0      = zeros(m,1);
    v0      = zeros(m,1);
     
    % Define function handle to seismic loading
    realizationID ...
            = 1;        % Belongs to 1:20
    
    hDofs   = 3*(1:length(XNOD))-2;
    [~,~,shkDofs] ...
            = intersect(hDofs,corres);
    r       = zeros(m,1);
    r(shkDofs) ...      % Vector r contains unit entries at dofs corresponding 
            = 1;        % to horizontal displacements, 0 at other ones
    quakeAmpl ...
            = 1;
    gDistri = quakeAmpl*M*r;
    defineGroundAcceleration([],'ACCEL.ACG',realizationID,gDistri);
    fHdle   = @(t) defineGroundAcceleration(t);
    
    % Define contact stiffness
    k_con   = max(K(:))*100;
    
    % Compute timesteps
    nPtsPer     = 2.5;	% Define minimum number of computed points per lowest eigenperiod of the system
    wMax1       = sqrt(eigs(K,M,1));
    hCp         = 2*pi/(wMax1*nPtsPer);
    KCON        = zeros(m);
    for ii = 1:size(W,2)
        KCON        = KCON+W(:,ii)*W(:,ii)'*k_con;
    end
    wMax2       = sqrt(eigs(K+KCON,M,1));
    hRed        = 2*pi/(wMax2*nPtsPer);
       
    % Define integration parameters
    schemeID ...
            = 1;        % 0: trapezoidal scheme - 1: 2-stage BoTr - 2: Generalized-A
    rho     = 0.5;      % Does not apply to schemeID 0
    hOut    = 1e-2;     % Storage timestep
    tf      = 35;       % Final integration time
    nRedH   = 10;       % If any status changed over the last nRedH increments, use reduced timestep as it is likely that we are in a persistent contact phase
    integParam ...
            = [schemeID rho hCp hRed nRedH hOut tf];       
        
    % Define root-solving parameters
    gTol    = 1e-6;
    maxIter = 100;
    tTol    = 1e-6;
    rsParam = [gTol maxIter tTol];
    
    header  = 'quakeSimulation';
    
    % Create structure with model parameters
    modelParam.K ...
            = K;
    modelParam.C ...
            = C;
    modelParam.M ...
            = M;
    modelParam.fHdle ...
            = fHdle;
    modelParam.W ...
            = W;
    modelParam.u0 ...
            = u0;
    modelParam.v0 ...
            = v0;
    modelParam.g0 ...
            = g0;
    modelParam.k_con ...
            = k_con;
    
    % Perform time integration
    switch schemeID
        case 0
            warning('Trapezoidal scheme is conservative, parameter rho is discarded')
            fHeader = timeIntegration(modelParam,integParam,rsParam,header);
        case {1,2}
            fHeader = timeIntegration(modelParam,integParam,rsParam,header);
        case 3
            warning('Noh-Bathe scheme is not implemented in its parameterized form, parameter rho is discarded; Rayleigh damping is suppressed.')
            % Compute critical timestep
            p       = 0.54;     % Integration scheme parameter, do not modify
            WS      = 0.1*2/sqrt((1-p)*(3*p-1));
            assert(hCp < WS/wMax1 && hRed < WS/wMax2,'criticalTimestep:Increase number of points per period')
            % Discard Rayleigh damping
            modelParam.C ...
                    = sparse(m,m);
            fHeader = timeIntegration_NoBa13(modelParam,integParam,rsParam,header);
        otherwise
            error('undefinedSchemeID:integration schemeID must range in {0,1,2,3}')
    end
        
    % Open results files
    fidTime = fopen([fHeader '_time.his'],'r');
    fidDisp = fopen([fHeader '_displacements.his'],'r');
    fidVelo = fopen([fHeader '_velocities.his'],'r');
    fidFlag = fopen([fHeader '_flags.his'],'r');
    fidGap  = fopen([fHeader '_gaps.his'],'r');
    fidNrg  = fopen([fHeader '_nrg.his'],'r');
    fidCtc  = fopen([fHeader '_ctcForces.his'],'r');
    
    % Read binary files
    results.time ...
            = fread(fidTime,[1 inf],'double');
    results.gap ...
            = fread(fidGap,[n inf],'double');
    results.flag ...
            = fread(fidFlag,[n inf],'double');
    results.disp ...
            = fread(fidDisp,[m inf],'double');
    results.velo ...
            = fread(fidVelo,[m inf],'double');
    results.nrg ...
            = fread(fidNrg,[1 inf],'double');
    results.fCtc ...
            = fread(fidCtc,[n inf],'double');
    
    % Close all open files
    fclose('all');
    
    plot(results.time,results.gap)
    xlabel('Time [s]')
    ylabel('Gap [m]')
    title('Gap evolution at slip surface joints')
end

%%%
%%% AUXILIARY FUNCTIONS
%%%

function grAcc = defineGroundAcceleration(t,file,spectrumID,gDistri)

    persistent p_data p_spectrumID p_gDistri

    if nargin == 1
        if t <= p_data(end,1)
            ind     = find(t >= p_data(:,1));
            grAcc   = p_data(ind(end),p_spectrumID)*p_gDistri;
        else
            grAcc   = zeros(size(p_gDistri));
        end
    else % This is initialization
        p_data  = load(file);
        p_spectrumID ...
                = spectrumID+1;
        p_gDistri ...
                = gDistri;
        
        grAcc   = [];
    end

end

function [alpha,beta] = RayleighDamping(omega1,omega2,ksi1,ksi2)
    % Return the coefficients of the 2-point Rayleigh damping
    % C = alpha.M + beta.K

    beta    = 2*(omega2*ksi2-omega1*ksi1)/(omega2^2-omega1^2);
    alpha   = 2*omega1*ksi1-beta*omega1^2;
end
