function results = newtonCradle
%function results = newtonCradle
%   This function computes the time response of an elastic bouncing bar, as presented in Section 4.2 of
%      Depouhon A., Detournay E. and Denoël V., "Event-driven integration of linear structural 
%      dynamics models under unilateral elastic constraints", CMAME, 2014
%
%   The function calls timeIntegration.m
%
%   Problem parameters include:
%     - Model parameters (N,v0,L,c0,g0,G0)
%     - Spatial discretization parameter (nEl)
%     - Contact stiffness (k_con)
%     - Minimum number of computed points per lowest eigenperiod of the system (nPtsPer)
%     - Time integration parameters (schemeID,rho,hOut,nRedH)
%     - Root-solving parameters (gTol,maxIter,tTol)
%
%   Results are stored in binary files by timeIntegration.m. These are loaded and returned by the current function
%   under the form of data structure _results_.
%
%   Written by A. Depouhon - 12/4/2013
%   Contact: alexandre.depouhon@ulg.ac.be

    % Clean environment
    close all
    clc
    clear functions     % Required to clear all persistent variables !
    
    % Define model parameters
    N       = 5;        % Number of bars
    v0      = 0.05;     % Initial velocity of bar 1
    L       = 1;        % Length of bars
    c0      = 1;        % Wave propagation speed
    g0      = v0*1.5;	% Initial gap between bars
    G0      = v0*2.5;	% Initial gap between extreme bars and side walls
    
    % Define # mesh elements
    nEl     = 100;
    
    % Form governing matrices and loading handle
    [K,M]   = formGoverningMatrices(N,nEl);
    m       = size(K,1);
    C       = sparse(m,m);
    spZeros = sparse(m,1);
    fHdle   = @(t) spZeros;
    
    % Define initial conditions and gaps
    U0      = zeros(m,1);
    [xMesh,V0,g0] ...
            = initialConditions(N,nEl,v0,g0,G0);
    W       = formActiveDofMatrix(N,nEl);
    n       = size(W,2);
    
    
    % Define contact stiffness
    k_con       = 100*max(K(:));
       
    % Compute timesteps
    nPtsPer     = 5;	% Define minimum number of computed points per lowest eigenperiod of the system
    wMax1       = sqrt(eigs(K,M,1));
    hCp         = 2*pi/(wMax1*nPtsPer);
    KCON        = zeros(m);
    for ii = 1:size(W,2)
        KCON        = KCON+W(:,ii)*W(:,ii)'*k_con;
    end
    wMax2       = sqrt(eigs(K+KCON,M,1));
    hRed        = 2*pi/(wMax2*nPtsPer);
    
   % Define integration parameters
    schemeID ...
            = 1;        % 0: trapezoidal scheme - 1: 2-stage BoTr - 2: Generalized-A - 3: Noh-Bathe explicit
    rho     = 0.5;      % Does not apply to schemeID 0
    hOut    = 1e-2;     % Storage timestep
    T       = (4*g0(1)/v0+2*(N-1)*g0(2)/v0+2*(N+1)*L/c0);
    tf      = T;      % Final integration time
    nRedH   = 10;       % If any status changed over the last nRedH increments, use reduced timestep as it is likely that we are in a persistent contact phase
    integParam ...
            = [schemeID rho hCp hRed nRedH hOut tf];        
        
    % Define root-solving parameters
    gTol    = 1e-8;
    maxIter = 100;
    tTol    = 1e-8;
    rsParam = [gTol maxIter tTol];
    
    
    % Define header
    header  = 'newtonCradle';
    
    % Create structure with model parameters
    modelParam.K ...
            = K;
    modelParam.C ...
            = C;
    modelParam.M ...
            = M;
    modelParam.fHdle ...
            = fHdle;
    modelParam.W ...
            = W;
    modelParam.u0 ...
            = U0;
    modelParam.v0 ...
            = V0;
    modelParam.g0 ...
            = g0;
    modelParam.k_con ...
            = k_con;
    
    % Perform integration
    switch schemeID
        case 0
            warning('Trapezoidal scheme is conservative, parameter rho is discarded')
            fHeader = timeIntegration(modelParam,integParam,rsParam,header);
        case {1,2}
            fHeader = timeIntegration(modelParam,integParam,rsParam,header);
        case 3
            warning('Noh-Bathe scheme is not implemented in its parameterized form, parameter rho is discarded; mass matrix is diagonalized.')
            % Compute critical timestep
            p       = 0.54;     % Integration scheme parameter, do not modify
            WS      = 0.1*2/sqrt((1-p)*(3*p-1));
            assert(hCp < WS/wMax1 && hRed < WS/wMax2,'criticalTimestep:Increase number of points per period')
            % Lump mass matrix
            modelParam.M ...
                    = diag(sum(M,2));
            fHeader = timeIntegration_NoBa13(modelParam,integParam,rsParam,header);
        otherwise
            error('undefinedSchemeID:integration schemeID must range in {0,1,2,3}')
    end
    
    % Open results files
    fidTime = fopen([fHeader '_time.his'],'r');
    fidDisp = fopen([fHeader '_displacements.his'],'r');
    fidVelo = fopen([fHeader '_velocities.his'],'r');
    fidFlag = fopen([fHeader '_flags.his'],'r');
    fidGap  = fopen([fHeader '_gaps.his'],'r');
    fidNrg  = fopen([fHeader '_nrg.his'],'r');
    fidCtc  = fopen([fHeader '_ctcForces.his'],'r');
    
    % Read binary files
    results.time ...
            = fread(fidTime,[1 inf],'double');
    results.gap ...
            = fread(fidGap,[n inf],'double');
    results.flag ...
            = fread(fidFlag,[n inf],'double');
    results.disp ...
            = fread(fidDisp,[m inf],'double');
    results.velo ...
            = fread(fidVelo,[m inf],'double');
    results.nrg ...
            = fread(fidNrg,[1 inf],'double');
    results.fCtc ...
            = fread(fidCtc,[n inf],'double');
    [results.dispCG,results.veloCG] ...
            = computeCGmotion(N,M,results.disp,results.velo);
    results.xMesh ...
            = xMesh;
    
    % Close all open files
    fclose('all');
    
    % Plot average response
%     [~,ax]  = averageResponse(N,round(tf/T),v0,g0(1),g0(2),L,c0);
    ax(1)   = subplot(211);
    ax(2)   = subplot(212);
    hold(ax(1),'on')
    plot(ax(1),results.time,results.dispCG,'--')
    plot(ax(2),results.time,results.veloCG,'--')
    title(ax(1),'Bar average responses, exact (solid) and numerical (dash)')
    xlabel(ax(2),'Time')
    ylabel(ax(1),'Displacement')
    ylabel(ax(2),'Velocity')
    
end

%%%
%%% AUXILIARY FUNCTIONS
%%%

function [dispCG,veloCG] = computeCGmotion(nBody,M,disp,velo)
    nDofSglBody = round(size(M,1)/nBody);
    M_sglBody   = M(1:nDofSglBody,1:nDofSglBody);
    m_sglBody   = sum(M_sglBody(:));
    nSv         = size(disp,2);
    dispCG      = nan(nBody,nSv);
    veloCG      = nan(nBody,nSv);
    for ii = 1:nBody
        iBody           = (1:nDofSglBody)+(ii-1)*nDofSglBody;
        tmp1            = sum(bsxfun(@times,full(diag(M_sglBody)),disp(iBody,:)),1);
        tmp2            = sum(bsxfun(@times,full(diag(M_sglBody,1)),disp(iBody(1:end-1),:)),1);
        tmp3            = sum(bsxfun(@times,full(diag(M_sglBody,-1)),disp(iBody(2:end),:)),1);
        dispCG(ii,:)    = (tmp1+tmp2+tmp3)/m_sglBody;
        tmp1            = sum(bsxfun(@times,full(diag(M_sglBody)),velo(iBody,:)),1);
        tmp2            = sum(bsxfun(@times,full(diag(M_sglBody,1)),velo(iBody(1:end-1),:)),1);
        tmp3            = sum(bsxfun(@times,full(diag(M_sglBody,-1)),velo(iBody(2:end),:)),1);
        veloCG(ii,:)    = (tmp1+tmp2+tmp3)/m_sglBody;
    end
end

function [X0,V0,initialGap] = initialConditions(nBody,nEl,v0,g0,G0)
    
    % Define physical quantities
    borderGap   = G0;
    intrBodyGap = g0;
    xLocMesh    = (0:nEl)/nEl;
    
    % Assign initial displacement
    nDof        = nEl+1;
    X0          = nan(nDof*nBody,1);
    V0          = [v0(ones(nDof,1)) ; zeros(nDof*(nBody-1),1)];
    for ii = 1:nBody
        ind         = (1:nDof)+nDof*(ii-1);
        X0(ind)     = borderGap+(ii-1)*(1+intrBodyGap)+xLocMesh;
    end
    initialGap  = [borderGap ; intrBodyGap(ones(nBody-1,1)) ; borderGap];
end

function S = formActiveDofMatrix(nBody,nEl)
    nCstr       = nBody+1;
    nDof        = nEl+1;
    totalNDof   = nBody*nDof;
    row         = nan(1,2+(nBody-1)*2);
    col         = nan(1,2+(nBody-1)*2);
    data        = nan(1,2+(nBody-1)*2);
    row(1)      = 1;
    col(1)      = 1;
    data(1)     = 1;
    cnt         = 2;
    for ii = 2:nCstr-1
        row(cnt:cnt+1) ...
                    = (ii-1)*nDof+(0:1);
        col(cnt:cnt+1) ...
                    = [ii ii];
        data(cnt:cnt+1) ...
                    = [-1 1];
        cnt         = cnt+2;
    end
    row(cnt)    = totalNDof;
    col(cnt)    = nCstr;
    data(cnt)   = -1;
    S           = sparse(row,col,data);
end

function [K,M] = formGoverningMatrices(nBody,nEl)
    % This function assembles the global mass & stiffness matrices for a
    % bar element of constant cross section, using linear elements
    %
    % Inputs:
    %  - barParam : cell array containing the physical parameters of the
    %  bar and number of elements per segment of constant area
    %
    % Outputs:
    %  - K        : stiffness matrix [sparse]
    %  - M        : mass matrix [sparse]
    %
    % ADN - APR '13
    

    % Define elemental matrices
    elmtLength  = 1/nEl;
    k           = 1/elmtLength;
    m           = elmtLength;
    K_el        = k*[1 -1;-1 1];
    M_el        = m/12*[5 1;1 5];

    % Construct triplet representation of global matrices
    nDof        = nEl+1;
        % Preallocate vectors
    nnz         = (nDof-2)*3+4;
    ii          = zeros(1,nnz);
    jj          = zeros(1,nnz);
    s_M         = zeros(1,nnz);
    s_K         = zeros(1,nnz);
        % Feed vectors from the matrix definition, column by column
    for mm = 1:nDof
        switch mm
            case 1      % First column
                ind             = 1:2;
                ii(ind)         = 1:2;
                jj(ind)         = mm(ones(1,2));
                s_M(ind)        = M_el(1:2);
                s_K(ind)        = K_el(1:2);
            case nDof   % Last column
                ind             = nnz-1:nnz;
                ii(ind)         = nDof-1:nDof;
                jj(ind)         = mm(ones(1,2));
                s_M(ind)        = M_el(3:4);
                s_K(ind)        = K_el(3:4);
            otherwise   % Inner columns
                ind             = (1:3)+(mm-2)*3+2;
                ii(ind)         = (1:3)+mm-2;
                jj(ind)         = mm(ones(1,3));
                s_M(ind)        = [M_el(2) 2*M_el(1) M_el(3)];
                s_K(ind)        = [K_el(2) 2*K_el(1) K_el(3)];
        end
    end

    % Assemble triplet in sparse matrix representation
    M_body      = sparse(ii,jj,s_M,nDof,nDof);
    K_body      = sparse(ii,jj,s_K,nDof,nDof); 

    K           = K_body;
    M           = M_body;
    for ii = 1:nBody-1
        K           = assembleDisjointBodies(K,K_body);
        M           = assembleDisjointBodies(M,M_body);
    end
end

function MAT = assembleDisjointBodies(MAT1,MAT2)
    % Global matrix assemblage for two independent bodies
    % 
    % INPUTS:
    %  - MAT1 & MAT2 : matrices to assemble [sparse]
    % 
    % OUTPUT:
    %  - MAT         : assembled matrix = [MAT1 0 ; 0 MAT2]
    
    % Extract data
    [row1,col1,v1]  = find(MAT1);
    nDof1           = length(MAT1);
    [row2,col2,v2]  = find(MAT2);
    nDof2           = length(MAT2);
    
    % Assemble block matrices
    row             = [row1 ; row2+nDof1];
    col             = [col1 ; col2+nDof1];
    val             = [v1 ; v2];
    nDof            = nDof1+nDof2;
    MAT             = sparse(row,col,val,nDof,nDof);
end