function results = bouncingBar
%function results = bouncingBar
%   This function computes the time response of an elastic bouncing bar, as presented in Section 4.1 of
%      Depouhon A., Detournay E. and Denoël V., "Event-driven integration of linear structural 
%      dynamics models under unilateral elastic constraints", CMAME, 2014
%
%   The function calls timeIntegration.m
%
%   Problem parameters include:
%     - Model parameters (C0,L,G,HINI,VINI)
%     - Spatial discretization parameter (nEl)
%     - Contact stiffness (k_con)
%     - Minimum number of computed points per lowest eigenperiod of the system (nPtsPer)
%     - Time integration parameters (schemeID,rho,hOut,nRedH)
%     - Root-solving parameters (gTol,maxIter,tTol)
%
%   Results are stored in binary files by timeIntegration.m. These are loaded and returned by the current function
%   under the form of data structure _results_.
%
%   Written by A. Depouhon - 12/4/2013
%   Contact: alexandre.depouhon@ulg.ac.be

    % Clean environment
    close all
    clc
    clear functions     % Required to clear all persistent variables !

    % Define model parameters
    C0      = 30;	% Wave propagation speed
    L       = 10;	% Bar length
    G       = 10;	% Gravity acceleration
    HINI    = 5;	% Bar initial height
    VINI    = 0;	% Bar initial velocity
        
    % Define # mesh elements
    nEl     = 100;
    m       = nEl+1;
    
    % Form governing matrices
    [K,M]   = formGoverningMatrices(L,C0,nEl);
    C       = sparse(m,m);
    
    % Precompute constant acceleration force
    f_G     = -M*G(ones(m,1));
    fHdle   = @(t) f_G;
    
    % System is initially at rest
    u0      = HINI(ones(m,1));
    v0      = VINI(ones(m,1));
    
    % Gap definition (gap coincides with bottom node displacement > g0 = 0)
    g0      = 0;
    W       = sparse(1,1,1,m,1);    
    
    % Define contact stiffness
    k_con   = 100*full(max(K(:)));
           
    % Define integration parameters
        % Compute timesteps   
    nPtsPer = 25;	% Define minimum number of computed points per lowest eigenperiod of the system
    wMax1   = sqrt(eigs(K,M,1));
    wMax2   = sqrt(eigs(K+W*W'*k_con,M,1));
    hCp     = 2*pi/(wMax1*nPtsPer);
    hRed    = 2*pi/(wMax2*nPtsPer);
    schemeID ...
            = 3;        % 0: trapezoidal scheme - 1: 2-stage BoTr - 2: Generalized-A - 3: Noh-Bathe explicit
    rho     = 0.5;        % Does not apply to schemeID 0 & 3
    hOut    = 1e-2;     % Storage timestep
    tf      = 32/3;     % Final integration time
    nRedH   = 10;       % If any status changed over the last nRedH increments, use reduced timestep as it is likely that we are in a persistent contact phase
    integParam ...
            = [schemeID rho hCp hRed nRedH hOut tf];
        
    % Define root-solving parameters
    gTol    = 1e-8;
    maxIter = 100;
    tTol    = 1e-8;
    rsParam = [gTol maxIter tTol];
    
    % Define problem name
    header  = 'bouncingBar';
    
    % Create structure with model parameters
    modelParam.K ...
            = K;
    modelParam.C ...
            = C;
    modelParam.M ...
            = M;
    modelParam.fHdle ...
            = fHdle;
    modelParam.W ...
            = W;
    modelParam.u0 ...
            = u0;
    modelParam.v0 ...
            = v0;
    modelParam.g0 ...
            = g0;
    modelParam.k_con ...
            = k_con;
           
    % Perform integration
    switch schemeID
        case 0
            warning('Trapezoidal scheme is conservative, parameter rho is discarded')
            fHeader = timeIntegration(modelParam,integParam,rsParam,header);
        case {1,2}
            fHeader = timeIntegration(modelParam,integParam,rsParam,header);
        case 3
            warning('Noh-Bathe scheme is not implemented in its parameterized form, parameter rho is discarded; mass matrix is diagonalized')
            % Compute critical timestep
            p       = 0.54;     % Integration scheme parameter, do not modify
            WS      = 0.1*2/sqrt((1-p)*(3*p-1));
            assert(hCp < WS/wMax1 && hRed < WS/wMax2,'criticalTimestep:Increase number of points per period')
            % Lump mass matrix and redefine loading
            modelParam.M ...
                    = diag(sum(M,2));
            modelParam.fHdle ...
                    = @(t) -modelParam.M*G(ones(m,1));
            fHeader = timeIntegration_NoBa13(modelParam,integParam,rsParam,header);
        otherwise
            error('undefinedSchemeID:integration schemeID must range in {0,1,2,3}')
    end
    
    % Open results files
    fidTime = fopen([fHeader '_time.his'],'r');
    fidDisp = fopen([fHeader '_displacements.his'],'r');
    fidVelo = fopen([fHeader '_velocities.his'],'r');
    fidFlag = fopen([fHeader '_flags.his'],'r');
    fidGap  = fopen([fHeader '_gaps.his'],'r');
    fidNrg  = fopen([fHeader '_nrg.his'],'r');
    fidCtc  = fopen([fHeader '_ctcForces.his'],'r');
    
    % Read binary files
    n       = size(W,2);
    results.time ...
            = fread(fidTime,[1 inf],'double');
    results.gap ...
            = fread(fidGap,[n inf],'double');
    results.flag ...
            = fread(fidFlag,[n inf],'double');
    results.disp ...
            = fread(fidDisp,[m inf],'double');
    results.velo ...
            = fread(fidVelo,[m inf],'double');
    results.nrg ...
            = fread(fidNrg,[1 inf],'double');
    results.fCtc ...
            = fread(fidCtc,[n inf],'double');
    results.totalNrg ...
            = (results.nrg-sum(bsxfun(@times,results.disp,f_G),1))/(HINI*G*L);
    % Close all open files
    fclose('all');
    
    % Plot displacement field at bar extremities and CG
    if mod(nEl,2) == 0
        yPlot   = bsxfun(@plus,[0 ; 0.5 ; 1]*L,results.disp([1 0.5*nEl+1 m],:));
    else
        yPlot   = bsxfun(@plus,[0 ; 0.5 ; 1]*L,[results.disp(1,:) ; mean(results.disp(0.5*[nEl-1 nEl+1],:),1) ; results.disp(m,:)]);
    end
    plot(results.time,yPlot)
    xlabel('Time')
    ylabel('Bar position')
    title('Bouncing bar motion')
    xlim([0 tf])
    ylim([-0.5 15.5])
    
    
end

function [K,M] = formGoverningMatrices(L,C0,nEl)
    % This function assembles the global mass & stiffness matrices for a
    % bar element of constant cross section, using high order linear elements
    %
    % ADN - APR '13
    

    % Define elemental matrices
    elmtLength  = L/nEl;
    k           = C0^2/elmtLength;
    m           = elmtLength;
    K_el        = k*[1 -1;-1 1];
    M_el        = m/12*[5 1;1 5];

    % Construct triplet representation of global matrices
    nDof        = nEl+1;
        % Preallocate vectors
    nnz         = (nDof-2)*3+4;
    ii          = zeros(1,nnz);
    jj          = zeros(1,nnz);
    s_M         = zeros(1,nnz);
    s_K         = zeros(1,nnz);
        % Feed vectors from the matrix definition, column by column
    for mm = 1:nDof
        switch mm
            case 1      % First column
                ind             = 1:2;
                ii(ind)         = 1:2;
                jj(ind)         = mm(ones(1,2));
                s_M(ind)        = M_el(1:2);
                s_K(ind)        = K_el(1:2);
            case nDof   % Last column
                ind             = nnz-1:nnz;
                ii(ind)         = nDof-1:nDof;
                jj(ind)         = mm(ones(1,2));
                s_M(ind)        = M_el(3:4);
                s_K(ind)        = K_el(3:4);
            otherwise   % Inner columns
                ind             = (1:3)+(mm-2)*3+2;
                ii(ind)         = (1:3)+mm-2;
                jj(ind)         = mm(ones(1,3));
                s_M(ind)        = [M_el(2) 2*M_el(1) M_el(3)];
                s_K(ind)        = [K_el(2) 2*K_el(1) K_el(3)];
        end
    end

    % Assemble triplet in sparse matrix representation
    M           = sparse(ii,jj,s_M,nDof,nDof);
    K           = sparse(ii,jj,s_K,nDof,nDof); 

end