function fHeader = timeIntegration_NoBa13(modelParam,integParam,rsParam,header)
%function fHeader = timeIntegration(modelParam,integParam,rsParam,header)
%   This function proposes an implementation of the event-driven procedure detailed in 
%      Depouhon A., Detournay E. and Denoël V., "Event-driven integration of linear structural 
%      dynamics models under unilateral elastic constraints", CMAME, 2014
%
%   It is to be called by bouncingBar.m, newtonCradle.m or quakeSimulation.m. See the paper appendix 
%   and  these files for an appropriate definition of the input variables.
%
%   Written by A. Depouhon - 12/4/2013
%   Contact: alexandre.depouhon@ulg.ac.be
    
    tic
    
    % Assign model data to local variables
    K       = modelParam.K;
    C       = modelParam.C;
    M       = modelParam.M;
    fHdle   = modelParam.fHdle;
    W       = modelParam.W;
    uOld    = modelParam.u0;
    vOld    = modelParam.v0;
    g0      = modelParam.g0;
    k_con   = modelParam.k_con;
      
    % Factorize mass matrix if not diagonal
    if nnz(M)~=nnz(diag(M)) % Mass matrix is not diagonal, factorize it
        [LM,UM,PM,QM] ...
                = lu(M); 
        M       = {M,LM,UM,PM,QM};
        mDiag   = 0;
    else
        M       = {M};
        mDiag   = 1;
    end
    
    % Assign contact data - initial gap & contact stiffness can be provided
	% as a vector or as a scalar (it is then replicated for each interface)
    n       = size(W,2);
    if ~(length(g0) == n)
        g0      = g0*ones(n,1);
    else
        g0      = g0(:);
    end
    if ~(length(k_con) == n)
        k_con   = full(k_con)*ones(n,1);
    else
        k_con   = k_con(:);
    end
    
    % Define integration scheme parameters
    hCp     = integParam(3);
    hRed    = integParam(4);
    nRedH   = integParam(5);
    hOut    = integParam(6);
    tf      = integParam(7);
    maxIncr = round(tf/hCp*100);
       
    % Define root-solving parameters
    gTol    = rsParam(1);
    maxIter = rsParam(2);
    tTol    = rsParam(3);
    
    % Initialize system
        % a) Compute constraint status
    [g,dg]  = gaps(W,g0,uOld,vOld);
    iC      = getActiveConstraints(g);
        % b) Form contact stiffness matrix
    [KCON,fCON] ...
            = formContactData(iC,k_con,W,g0);
        % c) Compute contact force
    fCtc    = zeros(n,1);
    ind     = iC == 1;
    fCtc(ind) ...
            = -k_con(ind).*g(ind);
        % d) Initialize acceleration vector
    if ~mDiag
        % M       = {M,LM,UM,PM,QM};
        tmp     = M{2}\(M{4}*(fHdle(0)+fCON-(K+KCON)*uOld-C*vOld));
        aOld    = M{5}*(M{3}\tmp);
    else
        aOld    = M{1}\(fHdle(0)+fCON-(K+KCON)*uOld-C*vOld);
    end
        
    tOld    = 0;
    nIcrRed = 2*nRedH;
    gOld    = g;
    dgOld   = dg;
    
    % Open storage file
    fHeader = strrep(strtrim(header),' ','_');
    fidLog  = fopen([fHeader '.log'],'w');
    fidTime = fopen([fHeader '_time.his'],'W');
    fidDisp = fopen([fHeader '_displacements.his'],'W');
    fidVelo = fopen([fHeader '_velocities.his'],'W');
    fidFlag = fopen([fHeader '_flags.his'],'W');
    fidGap  = fopen([fHeader '_gaps.his'],'W');
    fidNrg  = fopen([fHeader '_nrg.his'],'W');
    fidCtc  = fopen([fHeader '_ctcForces.his'],'W');
    
    fprintf(fidLog,['      *** ' header ' ***\n']);
    fwrite(fidTime,0,'double');
    fwrite(fidDisp,uOld,'double');
    fwrite(fidVelo,vOld,'double');
    fwrite(fidGap,gOld,'double');
    fwrite(fidFlag,iC,'double');
    fwrite(fidNrg,0.5*(uOld'*K*uOld+vOld'*M{1}*vOld),'double');
    fwrite(fidCtc,fCtc,'double');
   
    fprintf(['      *** ' header ' ***\n']);
    prcCpl  = 0;
    
    % Initialize output points
    tOut    = 0;
    
    % Form amplification matrix
    h       = hCp;
    p       = 0.54;
    a       = NoBaParameters(p,h);
      
    % Loop over time
    incr    = 2;
    
    while true
        % Increment time
        tNew    = tOld+h;
        
        % Update state
        [uNew,vNew,aNew] ...
                = stepState(p,a,uOld,vOld,aOld,tOld,tNew,K,C,M,mDiag,fHdle,KCON,fCON);
        
        % Compute constraints
        [gNew,dgNew] ...
                = gaps(W,g0,uNew,vNew);
              
        % Detect constraint status changes
        [iEvt,hEvt,dfEvt] ...
                = detectEvents(h,gOld,gNew,dgOld,dgNew);

        % Localize constraints
        if ~isempty(iEvt)
            % Localize earliest event
            [iEvt,hNext,tNew,uNew,vNew,aNew,gNew,dgNew] ...
                = localizeEvents(W,g0,tOld,uOld,vOld,aOld,gOld,dgOld,tNew,gNew, ...
                dgNew,iEvt,hEvt,dfEvt,gTol,maxIter,fidLog,tTol,hRed,K,C,M,mDiag,fHdle,KCON,fCON,p);
            
            % Indicate point has been accepted prior to event occurrence
            hFlag   = 1;
        else
            hFlag   = 0;
        end
        
        % Compute contact force
        fCtc    = zeros(n,1);
        ind     = iC == 1;
        fCtc(ind) ...
                = -k_con(ind).*gNew(ind);
                
        % Update constraint flags, if required
        if ~isempty(iEvt)
            for ii = iEvt'
                if dgNew(ii) < 0        % ACTIVATE CONSTRAINT
                    iC(ii) = 1;
                elseif dgNew(ii) > 0    % DEACTIVATE CONSTRAINT
                    iC(ii) = 0;
                else % DO NOTHING WHEN GRAZING
                end
            end             
            
            % Update contact data
            [KCON,fCON] ...
                    = formContactData(iC,k_con,W,g0);
            
            % Form iteration matrices
            nIcrRed = 1;
            h       = hRed;
            a       = NoBaParameters(p,h);
        else
            if hFlag
                h       = hNext;
                a       = NoBaParameters(p,h);
                nIcrRed = 1;
            else
                nIcrRed = nIcrRed+1;
            end
        end
                        
        % Use large timestep if already in free flight for nRedH increments
        if nIcrRed > nRedH && h < hCp 
            h       = hCp;
            a       = NoBaParameters(p,h);
        end
        
        % Store data when required (event localized or time output requested)
        if tNew-tOut >= hOut || ~isempty(iEvt)
            fwrite(fidTime,tNew,'double');
            fwrite(fidDisp,uNew,'double');
            fwrite(fidVelo,vNew,'double');
            fwrite(fidGap,gNew,'double');
            fwrite(fidFlag,iC,'double');
            fwrite(fidNrg,0.5*(uNew'*K*uNew+vNew'*M{1}*vNew),'double');
            fwrite(fidCtc,fCtc,'double');
            tOut    = tNew;
        end
        
        % Print computation progression
        if floor(tNew/tf*100) > prcCpl    
            prcCpl  = floor(tNew/tf*100);
            fprintf(' > Computation completed at %2i %%\n',prcCpl);
        end
               
        % Update old data
        tOld    = tNew;
        uOld    = uNew;
        vOld    = vNew;
        aOld    = aNew;
        gOld    = gNew;
        dgOld   = dgNew;
        incr    = incr+1;
            
        % Verify inLoop conditions
        if tNew > tf || incr > maxIncr
            % Write output at final step
            fwrite(fidTime,tNew,'double');
            fwrite(fidDisp,uNew,'double');
            fwrite(fidVelo,vNew,'double');
            fwrite(fidGap,gNew,'double');
            fwrite(fidFlag,iC,'double');
            fwrite(fidNrg,0.5*(uNew'*K*uNew+vNew'*M{1}*vNew),'double');
            fwrite(fidCtc,fCtc,'double');
            if incr > maxIncr
                warning('Maximum number of increments has been reached.')
            end
            break
        end
    end
  
    cpTime  = toc;
    
    fprintf('\n > TOTAL NUMBER OF INCREMENTS: %i\n',incr-1);
    fprintf(fidLog,' > TOTAL NUMBER OF INCREMENTS: %i\n',incr-1);
    fprintf(' > TOTAL COMPUTATION TIME: %10.2f s\n',cpTime);
    fprintf(fidLog,' > TOTAL COMPUTATION TIME: %10.2f s\n',cpTime);
    
    fclose('all');
    
end

%%%
%%% AUXILIARY FUNCTIONS
%%%
function a = NoBaParameters(p,h)
 
    q1      = (1-2*p)/(2*p*(1-p));
    q2      = 0.5-p*q1;
    q0      = -q1-q2+0.5;
    
    a0      = p*h;
    a1      = 0.5*(p*h)^2;
    a2      = 0.5*a0;
    a3      = (1-p)*h;
    a4      = 0.5*((1-p)*h)^2;
    a5      = q0*a3;
    a6      = (0.5+q1)*a3;
    a7      = q2*a3;
    
    a       = [a0 a1 a2 a3 a4 a5 a6 a7];
    
end

function [uNew,vNew,aNew] = stepState(p,a,uOld,vOld,aOld,tOld,tNew,K,C,M,mDiag,fHdle,KCON,fCON)

    % Compute internal stage displacement
    uMid    = uOld+a(1)*vOld+a(2)*aOld;

    % Form effective load vector
    fMid    = (1-p)*(fHdle(tOld)+fCON)+p*(fHdle(tNew)+fCON);
    fMid    = fMid-(K+KCON)*uMid-C*(vOld+a(1)*aOld);
    
    % Compute internal stage acceleration
    if ~mDiag
        % M       = {M,LM,UM,PM,QM};
        tmp     = M{2}\(M{4}*fMid);
        aMid    = M{5}*(M{3}\tmp);
    else
        aMid    = M{1}\fMid;
    end

    % Compute internal stage velocity
    vMid    = vOld+a(3)*(aOld+aMid);
    
    % Compute final displacement
    uNew    = uMid+a(4)*vMid+a(5)*aMid;
    
    % Form effective load
    fNew    = (fHdle(tNew)+fCON)-(K+KCON)*uNew-C*(vMid+a(4)*aMid);
    
    % Compute final acceleration
    if ~mDiag
        % M       = {M,LM,UM,PM,QM};
        tmp     = M{2}\(M{4}*fNew);
        aNew    = M{5}*(M{3}\tmp);
    else
        aNew    = M{1}\fNew;
    end
    
    % Compute final velocity
    vNew    = vMid+a(6)*aOld+a(7)*aMid+a(8)*aNew;

end

function iC = getActiveConstraints(g)
    
    iC      = zeros(size(g));
    iC(g < 0) ...
            = 1;

end

function [g,dg] = gaps(W,g0,u,v)

    g       = W'*u+g0;
    dg      = W'*v;
    
end

function [iEvt,hNext,tTmp,uTmp,vTmp,aTmp,gTmp,dgTmp] = localizeEvents ...
    (W,g0,tOld,uOld,vOld,aOld,gOld,dgOld,tNew,gNew,dgNew,iEvt,hEvt, ...
    dfEvt,gTol,maxIter,fidLog,tTol,hRed,K,C,M,mDiag,fHdle,KCON,fCON,p)
            iter    = 1;
            iEvtIni = iEvt;
            hNext   = [];
            if sum(iEvt == iEvt(1)) > 1
                hMax    = 0.5*sum(hEvt(iEvt == iEvt(1)));
            else
                hMax    = (tNew-tOld);
            end
            hTmp    = min([hMax hEvt(1)+gTol/abs(dfEvt(1))]);
            
            % Define storage array [cpTime ; f ; df]
            cstrAry = nan(1+2*length(gOld),maxIter+1);
            cstrAry(:,1) ...
                    = [tOld ; gOld ; dgOld];
            cstrAry(:,2) ...
                    = [tNew ; gNew ; dgNew];
            colEvt  = 2;

            while true
                % Update temporary time
                tTmp    = tOld+hTmp;

                % Update NoBa parameters (depend on timestep h)
                a       = NoBaParameters(p,hTmp);

                % Update state
                [uTmp,vTmp,aTmp] ...
                        = stepState(p,a,uOld,vOld,aOld,tOld,tTmp,K,C,M,mDiag,fHdle,KCON,fCON);
                                               
                % Compute constraints
                [gTmp,dgTmp] ...
                        = gaps(W,g0,uTmp,vTmp);
                
                % Detect constraint status changes in left subinterval
                [iTmp,hTmp,dgEvtTmp] ...
                        = detectEvents(tTmp-tOld,gOld,gTmp,dgOld,dgTmp);

                if isempty(iTmp) 
                    % No significant event takes place in the left subinterval
                    % i.e, no constraint crossing has occurred. An additional 
                    % increment is required. Define hNext unless constraint is exactly zero.
                    iEvt    = find(gTmp == 0);
                    if isempty(iEvt)
                        [iTmp,hTmp,dgEvtTmp] ...
                            = detectEvents(tNew-tTmp,gTmp,gNew,dgTmp,dgNew);
                        if isempty(iTmp) 
                            % No event in right subinterval > numerical grazing
                            % Return to hRed
                            hNext   = hRed;
                        else
                            hNext   = max([sqrt(eps*tTol) min([tNew-tTmp hTmp(1)+0.5*gTol/abs(dgEvtTmp(1))])]);
                        end
                    end
                    break

                else
                    % At least one event takes place in the left subinterval.
                    % Zero crossing is achieved for at least one event.
                    
                    % Get event indices that are within accumulation
                    % distance aTol of the computed point.
                    indTmp  = abs((tTmp-tOld)-hTmp)  < min([gTol/abs(dgTmp(iTmp(1))) tTol]);
                    
                    % If any
                    if sum(indTmp) > 0
                        % Identify the ones that are within the event tolerance gTol
                        iEvt    = iTmp(abs(gTmp(iTmp(indTmp))+gTol*sign(gOld(iTmp(indTmp)))) <= gTol);
                        if ~isempty(iEvt) 
                            % Events have been localized
                            break
                        end
                    end
                    if sum(iTmp == iTmp(1)) > 1
                        hMax    = 0.5*sum(hTmp(iTmp == iTmp(1)));
                    else
                        hMax    = (tTmp-tOld);
                    end
                    scFctr  = ((maxIter-iter)/maxIter)^(100*maxIter/(maxIter-iter));
                    hTmp    = max([sqrt(eps*tTol) min([hMax hTmp(1)+scFctr*gTol/abs(dgEvtTmp(1))])]);
                    colInd  = colEvt:iter+1;
                    cstrAry(:,colInd+1) ...
                            = cstrAry(:,colInd);
                    cstrAry(:,colEvt) ...
                            = [tTmp ; gTmp ; dgTmp];

                    iEvt    = iTmp;
                end

                iter    = iter+1;
                if iter > maxIter
                    fclose('all');
                    error('EVENT-LOCALIZATION DID NOT CONVERGE')
                end
            end
            
            
            fprintf(fidLog,'>> tEvt = %16.15e -- iter = %3i',tTmp,iter);
            if ~isempty(iEvt)
                for ii = 1:length(iEvt)
                   fprintf(fidLog,' -- iEvt = %3i -- gRes = %4.3e',iEvt(ii),abs(gTmp(iEvt(ii))));
                end
                fprintf(fidLog,'\n');
            else
                for ii = 1:length(iEvtIni)
                   fprintf(fidLog,' -- iEvt = %3i -- belongs to right sub-interval',iEvtIni);
                end
                fprintf(fidLog,'\n');
            end
            if hNext < 0
                keyboard
            end

end

function [iEvt,hEvt,dfEvt] = detectEvents(h,gOld,gNew,dgOld,dgNew)
    
    persistent degTol nCon iList
    if isempty(degTol)
        % Set degree degeneracy tolerance
        degTol  = 1e-12;
        nCon    = length(gOld);
        iList   = reshape(repmat(1:nCon,3,1),3*nCon,1);
    end

    % Form Hermite polynomial coefficients
    a3      = 2*gOld+h*dgOld-2*gNew+h*dgNew;
    a2      = -3*gOld-2*h*dgOld+3*gNew-h*dgNew;
    a1      = h*dgOld;
    a0      = gOld;
    
    % Initialize vector
    xiEvt   = Inf(3*nCon,1);  
      
    % Loop over event functions to compute roots (event times)
    for mm = 1:nCon
        if abs(a3(mm)) >= degTol
            A       = diag(ones(2,1),-1);
            A(1,:)  = -[a2(mm)/a3(mm) a1(mm)/a3(mm) a0(mm)/a3(mm)];
            SOL     = eig(A);
        else
            % Use barycentric representation
            C0      = [0 0 0 0 gOld(mm) ; 1 0 0 0 h*dgOld(mm) ; 0 0 1 0 gNew(mm) ; 0 0 1 1 h*dgNew(mm) ;-2 -1 2 -1 0];
            C1      = [1 0 0 0 0; 0 1 0 0 0 ; 0 0 1 0 0 ; 0 0 0 1 0 ; 0 0 0 0 0];
            SOL     = eig(C0,C1);
        end
        ind     = SOL > 0 & SOL <= 1 & imag(SOL) == 0;
        lInd    = sum(ind);
        xiEvt((1:lInd)+(mm-1)*3) ...
                = real(SOL(ind));
    end
    
    % Return results
    if all(isinf(xiEvt))
        iEvt    = [];
        hEvt    = [];
        dfEvt   = [];
    else
        ind     = find(~isinf(xiEvt));
        hEvt    = h*xiEvt(ind);
        iEvt    = iList(ind);
        [hEvt,ind] ...
                = sort(hEvt);
        iEvt    = iEvt(ind);
        dfEvt   = zeros(size(iEvt));
        % Compute derivative at zero crossing
        for ii = 1:length(iEvt)
            dfEvt(ii)   = polyval([3*a3(iEvt(ii)) 2*a2(iEvt(ii)) a1(iEvt(ii))],hEvt(ii)/h)/h;
        end
    end
end

function [KCON,fCON] = formContactData(iC,k_con,W,g0)

    persistent p_ctcDof
    
    % Compute number of degrees of freedom involved in contact interactions
    if isempty(p_ctcDof)
        nCon    = size(W,2);
        p_ctcDof= nan(nCon,2);
        for ii = 1:nCon
            tmp     = find(W(:,ii)); 
            p_ctcDof(ii,1:length(tmp)) ...
                    = tmp; 
        end
    end

    % Compute number of non-zero entries of contact stiffness matrix
    nnz     = 4*sum(iC)-3*sum(any(isnan(p_ctcDof(logical(iC),:)),2));

    % Compute stiffness matrix and contact load vector
    m   = size(W,1);
    KCON    = sparse([],[],[],m,m,nnz);
    fCON    = sparse(m,1);
    for ii = 1:length(iC)
        if iC(ii)
            KCON    = KCON+k_con(ii)*(W(:,ii)*W(:,ii)');
            fCON    = fCON-k_con(ii)*W(:,ii)*g0(ii);
        end
    end
end
